package com.biz.crm.cps.business.reward.cost.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.reward.cost.local.entity.CostTerminalDetailEntity;
import com.biz.crm.cps.business.reward.cost.local.entity.CostTerminalEntity;
import com.biz.crm.cps.business.reward.cost.local.repository.CostTerminalDetailRepository;
import com.biz.crm.cps.business.reward.cost.local.service.CostTerminalDetailService;
import com.biz.crm.cps.business.reward.cost.local.service.CostTerminalService;
import com.biz.crm.cps.business.reward.cost.sdk.common.CostCodeConstant;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @Author: zengxingwang
 * @Date: 2021/8/21 10:33
 */
@Slf4j
@Service
public class CostTerminalDetailServiceImpl implements CostTerminalDetailService {

  @Autowired
  private CostTerminalDetailRepository costTerminalDetailRepository;
  @Autowired
  private CostTerminalService costTerminalService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired
  private GenerateCodeService generateCodeService;
  /**
   * 多条件分页查询
   * @param costTerminalId
   * @param pageable
   * @return
   */
  @Override
  public Page<CostTerminalDetailEntity> findByCostTerminalId(String costTerminalId, Pageable pageable) {
    if(StringUtils.isBlank(costTerminalId)){
      return new Page<>();
    }
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    return costTerminalDetailRepository.findByCostTerminalId(costTerminalId, pageable);
  }

  /**
   * 创建
   * @param terminalDetailEntity
   */
  @Override
  @Transactional
  public CostTerminalDetailEntity create(CostTerminalDetailEntity terminalDetailEntity) {
    log.debug("新增费用流水传入参数={}", terminalDetailEntity);
    this.createForm(terminalDetailEntity);
    CostTerminalEntity terminal = terminalDetailEntity.getTerminal();
    /**
     * 主表里面是否存在此参与者信息,如果没有需要插入一条数据到主表里面,如果有 获取主表的id
     */
    CostTerminalEntity old = costTerminalService.findByTerminalCode(terminal.getTerminalCode());
    if (old != null) {

      BigDecimal totalAmount = old.getTotalAmount() == null ? BigDecimal.ZERO : old.getTotalAmount();
      BigDecimal unAuditFee = old.getUnAuditFee() == null ? BigDecimal.ZERO : old.getUnAuditFee();
      totalAmount = totalAmount.add(terminalDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
      unAuditFee = unAuditFee.add(terminalDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
      old.setTotalAmount(totalAmount);
      old.setUnAuditFee(unAuditFee);
      old.setModifyAccount(loginUserService.getLoginAccountName());
      old.setModifyTime(new Date());
      this.costTerminalService.update(old);
      terminalDetailEntity.setCostTerminalId(old.getId());
    } else {
      terminal.setAuditedFee(terminalDetailEntity.getAuditedFee());
      terminal.setTotalAmount(terminalDetailEntity.getAmount());
      terminal.setUnAuditFee(terminalDetailEntity.getAmount());
      CostTerminalEntity returnEntity = this.costTerminalService.create(terminal);
      terminalDetailEntity.setCostTerminalId(returnEntity.getId());
    }
    this.costTerminalDetailRepository.saveOrUpdate(terminalDetailEntity);
    return terminalDetailEntity;
  }

  /**
   * 组装保存entity数据
   *
   * @param terminalDetailEntity
   */
  private CostTerminalDetailEntity createForm(CostTerminalDetailEntity terminalDetailEntity) {
    //校验
    this.createValidation(terminalDetailEntity);
    if (Objects.nonNull(this.generateCodeService)) {
      terminalDetailEntity.setCode(this.generateCodeService.generateCode(CostCodeConstant.COST_FLOW, 1).get(0));
    }
    CostTerminalEntity terminal = terminalDetailEntity.getTerminal();
    terminal.setTenantCode(TenantUtils.getTenantCode());
    Date date = new Date();
    String loginAccountName = loginUserService.getLoginAccountName();
    terminal.setCreateTime(date);
    terminal.setCreateAccount(loginAccountName);
    terminal.setModifyTime(date);
    terminal.setModifyAccount(loginAccountName);
    terminal.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    terminal.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    terminalDetailEntity.setCreateTime(date);
    terminalDetailEntity.setCreateAccount(loginAccountName);
    terminalDetailEntity.setModifyTime(date);
    terminalDetailEntity.setModifyAccount(loginAccountName);

    // 红包余额 查询此客户最新的流水 获取最新的红包余额以此为基准 再做本次流水的运算
    CostTerminalEntity lastTerminal = costTerminalService.findTop1ByTerminalCode(terminal.getTerminalCode());
    if (lastTerminal != null) {
      //已核销费用
      BigDecimal auditedFee = lastTerminal.getAuditedFee() == null ? BigDecimal.ZERO : lastTerminal.getAuditedFee();
      terminalDetailEntity.setAuditedFee(auditedFee);
      //未核销费用
      BigDecimal unAuditFee = lastTerminal.getUnAuditFee() == null ? BigDecimal.ZERO : lastTerminal.getUnAuditFee();
      unAuditFee = unAuditFee.add(terminalDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
      terminalDetailEntity.setUnAuditFee(unAuditFee);
    } else {
      terminalDetailEntity.setAuditedFee(BigDecimal.ZERO);
      terminalDetailEntity.setUnAuditFee(terminalDetailEntity.getAmount());
    }
    return terminalDetailEntity;
  }

  /**
   * 校验费用流水信息
   *
   * @param terminalDetailEntity
   */
  private void createValidation(CostTerminalDetailEntity terminalDetailEntity) {

    // 验证必填项，以及字段长度
    Validate.notNull(terminalDetailEntity, "新增操作时，新增对象不能为空！");
    Validate.notBlank(terminalDetailEntity.getSourceCode(), "业务来源编号不能为空;");
    Validate.notBlank(terminalDetailEntity.getSourceType(), "业务类型不能为空;");
    CostTerminalEntity terminal = terminalDetailEntity.getTerminal();
    Validate.notNull(terminal, "新增操作时，终端费用信息不能为空！");
    Validate.notBlank(terminal.getTerminalCode(), "终端编号不能为空;");
    Validate.notBlank(terminal.getTerminalName(), "终端名称不能为空;");


    Validate.isTrue(terminalDetailEntity.getSourceCode().length() < 255, "业务来源编号，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(terminalDetailEntity.getSourceType().length() < 255, "业务类型，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 按 参与者code 和扫码code集合汇总获得的分利
   * @param participatorCode
   * @param recordCodes
   * @return
   */
  @Override
  public BigDecimal sumByParticipatorCodeAndRecordCodes(String participatorCode, List<String> recordCodes) {
    if(StringUtils.isBlank(participatorCode) && CollectionUtils.isEmpty(recordCodes)){
      return null;
    }
    return this.costTerminalDetailRepository.sumByParticipatorCodeAndRecordCodes(participatorCode,recordCodes);
  }

  /**
   * 按 租户和类型 汇总获得分利
   * @param tenantCode
   * @param triggerAction
   * @return
   */
  @Override
  public BigDecimal sumByTenantCodeAndTriggerAction(String tenantCode, String triggerAction) {
    if(StringUtils.isBlank(tenantCode) || StringUtils.isBlank(triggerAction)){
      return null;
    }
    return this.costTerminalDetailRepository.sumByTenantCodeAndTriggerAction(tenantCode, triggerAction);
  }

  /**
   * 修改
   * @param detail
   */
  @Override
  @Transactional
  public void update(CostTerminalDetailEntity detail) {
    this.updateValidation(detail);
    costTerminalDetailRepository.updateById(detail);
  }

  /**
   * 校验费用流水信息
   *
   * @param terminalDetailEntity
   */
  private void updateValidation(CostTerminalDetailEntity terminalDetailEntity) {

    // 验证必填项，以及字段长度
    Validate.notNull(terminalDetailEntity, "修改操作时，新增对象不能为空！");
    Validate.notBlank(terminalDetailEntity.getId(), "修改操作时，主键不能为空！");
    Validate.notBlank(terminalDetailEntity.getSourceCode(), "业务来源编号不能为空;");
    Validate.notBlank(terminalDetailEntity.getSourceType(), "业务类型不能为空;");

    Validate.isTrue(terminalDetailEntity.getSourceCode().length() < 255, "业务来源编号，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(terminalDetailEntity.getSourceType().length() < 255, "业务类型，在进行添加时填入值超过了限定长度(255)，请检查!");
  }
}
