package com.biz.crm.cps.business.reward.cost.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.reward.cost.local.entity.CostTerminalEntity;
import com.biz.crm.cps.business.reward.cost.local.repository.CostTerminalRepository;
import com.biz.crm.cps.business.reward.cost.local.service.CostTerminalService;
import com.biz.crm.cps.business.reward.cost.sdk.dto.CostTerminalDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @Author: zengxingwang
 * @Date: 2021/8/21 10:33
 */
@Slf4j
@Service
public class CostTerminalServiceImpl implements CostTerminalService {

  @Autowired
  private CostTerminalRepository costTerminalRepository;
  /**
   * 多条件分页查询
   * @param costTerminalDto
   * @param pageable
   * @return
   */
  @Override
  public Page<CostTerminalEntity> findByConditions(CostTerminalDto costTerminalDto, Pageable pageable) {
    if(costTerminalDto == null){
      return new Page<>();
    }
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    costTerminalDto.setTenantCode(TenantUtils.getTenantCode());
    return costTerminalRepository.findByConditions(costTerminalDto, pageable);
  }

  /**
   * 根据编码查询
   * @param terminalCode
   * @return
   */
  @Override
  public CostTerminalEntity findByTerminalCode(String terminalCode) {
    if(StringUtils.isBlank(terminalCode)){
      return null;
    }
    return costTerminalRepository.findByTerminalCode(terminalCode);
  }

  /**
   * 创建
   * @param terminal
   * @return
   */
  @Override
  @Transactional
  public CostTerminalEntity create(CostTerminalEntity terminal) {
    this.createValidate(terminal);
    this.costTerminalRepository.save(terminal);
    return terminal;
  }

  /**
   * 创建验证
   * @param terminal
   */
  private void createValidate(CostTerminalEntity terminal) {
    Validate.notNull(terminal, "创建时，数据不能为空");
    Validate.notBlank(terminal.getTerminalCode(), "创建时，经销商编码不能为空");
    Validate.notBlank(terminal.getTerminalName(), "创建时，经销商名称不能为空");
    Validate.notBlank(terminal.getChannel(), "创建时，渠道不能为空");
    CostTerminalEntity costTerminalEntity = this.findByTerminalCode(terminal.getTerminalCode());
    Validate.isTrue(costTerminalEntity == null, "创建时，数据已存在，请检查");

    Validate.isTrue(terminal.getTerminalCode() == null || terminal.getTerminalCode().length() < 128 , "终端编码,在进行创建时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(terminal.getTerminalName() == null || terminal.getTerminalName().length() < 255 , "终端名称,在进行创建时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(terminal.getChannel() == null || terminal.getChannel().length() < 64 , "渠道编码,在进行创建时填入值超过了限定长度(64)，请检查!");
  }

  /**
   * 修改
   * @param terminal
   */
  @Override
  @Transactional
  public CostTerminalEntity update(CostTerminalEntity terminal) {
    this.updateValidate(terminal);
    this.costTerminalRepository.saveOrUpdate(terminal);
    return terminal;
  }

  @Override
  public CostTerminalEntity findTop1ByTerminalCode(String terminalCode) {
    if(StringUtils.isBlank(terminalCode)){
      return null;
    }
    return costTerminalRepository.findTop1ByTerminalCode(terminalCode);
  }

  /**
   * 修改验证
   * @param terminal
   */
  private void updateValidate(CostTerminalEntity terminal) {
    Validate.notNull(terminal, "修改时，数据不能为空");
    Validate.notBlank(terminal.getId(), "修改时，主键不能为空");
    Validate.notBlank(terminal.getTerminalCode(), "修改时，经销商编码不能为空");
    Validate.notBlank(terminal.getTerminalName(), "修改时，经销商名称不能为空");
    Validate.notBlank(terminal.getChannel(), "修改时，渠道不能为空");
    CostTerminalEntity costTerminalEntity = this.findByTerminalCode(terminal.getTerminalCode());
    Validate.isTrue(costTerminalEntity != null, "修改时，未找到对应数据，请检查");

    Validate.isTrue(terminal.getTerminalCode() == null || terminal.getTerminalCode().length() < 128 , "终端编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(terminal.getTerminalName() == null || terminal.getTerminalName().length() < 255 , "终端名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(terminal.getChannel() == null || terminal.getChannel().length() < 64 , "渠道编码,在进行修改时填入值超过了限定长度(64)，请检查!");
  }
}
