package com.biz.crm.cps.business.reward.local.service.observer;

import com.biz.crm.cps.business.activity.sdk.dto.ActivityRewardConditionDto;
import com.biz.crm.cps.business.activity.sdk.observer.ActivityRewardServiceObserver;
import com.biz.crm.cps.business.consumer.sdk.service.ConsumerVoService;
import com.biz.crm.cps.business.consumer.sdk.vo.ConsumerVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.participator.sdk.service.ParticipatorVoService;
import com.biz.crm.cps.business.participator.sdk.vo.ParticipatorVo;
import com.biz.crm.cps.business.reward.sdk.dto.RewardConditionDto;
import com.biz.crm.cps.business.reward.sdk.service.observer.RewardServiceObserver;
import com.biz.crm.cps.external.barcode.sdk.service.ScanCodeVoService;
import com.biz.crm.cps.external.barcode.sdk.vo.ScanCodeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 协议奖励实现
 * <pre>
 * 1将协议奖励条件公式转为奖励计算公式
 * 2填充参与者（获利者信息）
 * 3通知所有奖励模块组观察者
 * </pre>
 * @author yangfan
 */
@Component("ActivityRewardBridgeServiceObserverImpl")
@Slf4j
public class ActivityRewardBridgeServiceObserverImpl implements ActivityRewardServiceObserver {

  @Autowired(required = false)
  private List<RewardServiceObserver> awardServiceObservers;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ParticipatorVoService participatorVoService;
  @Autowired(required = false)
  private ConsumerVoService consumerVoService;
  @Autowired(required = false)
  private ScanCodeVoService scanCodeVoService;

  @Override
  public void createRewardRecord(List<ActivityRewardConditionDto> dtos) {

    /**
     * 作用：
     * 1 桥接（转换活动来的dto，改成 奖励模块组需要的dto）
     *
     * 2 调用参与者接口，补充参与者信息
     *
     * 3 调用所有奖励模块组观察者
     *
     */

    log.debug("createRewardRecord获取入参={}", dtos);
    if (CollectionUtils.isEmpty(dtos)) {
      return;
    }
    //1、2
    List<RewardConditionDto> rewardConditionDtos = this.rewardArgumentTransform(dtos);
    //3
    this.notify(rewardConditionDtos);
  }

  /**
   * 通知 所有奖励模块组观察者
   *
   * @param rewardConditionDtos
   */
  private void notify(List<RewardConditionDto> rewardConditionDtos) {
    if (CollectionUtils.isEmpty(rewardConditionDtos)) {
      return;
    }
    for (RewardConditionDto item : rewardConditionDtos) {
      //3
      if (!CollectionUtils.isEmpty(this.awardServiceObservers)) {
        for (RewardServiceObserver rewardServiceObserver : this.awardServiceObservers) {
          if(StringUtils.isNotBlank(item.getAdditionalParticipatorCode())){
            rewardServiceObserver.onCreate(item);
          }else{
            log.info("{}扫码未找到所属获利者{}跳过本次返利",item.getActualParticipatorName(),item.getAdditionalParticipatorFlag());
          }
        }
        // 如果还有分利传递，继续告知流水模块组的观察者们
        this.notify(item.getRewardConditionDtos());
      }
    }
  }

  /**
   * ActivityRewardConditionDto ->RewardConditionDto 活动模块奖励条件公式-》奖励模块奖励条件公式
   *
   * @param dtos
   * @return
   */
  private List<RewardConditionDto> rewardArgumentTransform(List<ActivityRewardConditionDto> dtos) {
    List<RewardConditionDto> list = Lists.newArrayList();
    for (ActivityRewardConditionDto item : dtos) {
      RewardConditionDto rewardConditionDto = this.nebulaToolkitService.copyObjectByWhiteList(item, RewardConditionDto.class, HashSet.class, ArrayList.class);
      rewardConditionDto = this.buildParticipatorInfo(rewardConditionDto, item);
      if (!CollectionUtils.isEmpty(item.getActivityRewardConditionDtos())) {
        List<RewardConditionDto> child = this.rewardArgumentTransform(item.getActivityRewardConditionDtos());
        rewardConditionDto.setRewardConditionDtos(child);
      }
      list.add(rewardConditionDto);
    }
    return list;
  }

  /**
   * 通过扫码者构建，奖励获利者信息
   *
   * @param target
   * @param source
   */
  private RewardConditionDto buildParticipatorInfo(RewardConditionDto target, ActivityRewardConditionDto source) {
    /**
     * 1。当前扫码人为消费者时
     *  1。1 如果当前获利者类型是终端时，需要去定位当前被扫的码所属终端/经销商，而不是直接去找消费者所属终端，如果码没有所属终端那么不处理
     * 2。当前扫码人为经销商/终端时，去找自己（终端）所属的上级
     */
    Validate.notBlank(source.getParticipatorFlag(), "得到这份奖励的参与者Flag不能为空!");
    List<ParticipatorVo> participatorVos = null;
    //1=========
    if (Objects.equals(ParticipatorTypeEnum.CONSUMER.getDictCode(), source.getActualParticipatorFlag())) {
      String openId = target.getActualParticipatorCode();
      ConsumerVo consumerVo = this.consumerVoService.findByExternalId(openId);
      Validate.notNull(consumerVo, "%s分利时，信息不能为空!", source.getParticipatorName());
      participatorVos = Lists.newArrayList();
      ParticipatorVo actualScan = new ParticipatorVo();
      actualScan.setParticipatorName(consumerVo.getNickname());
      actualScan.setParticipatorCode(consumerVo.getConsumerCode());
      actualScan.setParticipatorType(target.getActualParticipatorFlag());
      participatorVos.add(actualScan);
      //1。1=====
      if (!Objects.equals(ParticipatorTypeEnum.CONSUMER.getDictCode(), source.getParticipatorFlag()) && StringUtils.isNotBlank(source.getRecordCode())) {
        ScanCodeVo scanCodeVo = this.scanCodeVoService.findByRecordCode(source.getRecordCode());
        if (scanCodeVo != null) {
          ParticipatorTypeEnum participatorTypeEnum = ParticipatorTypeEnum.getByKey(scanCodeVo.getParticipatorType());
          List<ParticipatorVo> belong = participatorVoService.findByParticipatorTypeAndParticipatorCode(participatorTypeEnum.getDictCode(), scanCodeVo.getParticipatorCode());
          if (!CollectionUtils.isEmpty(belong)) {
            participatorVos.addAll(belong);
          }
        }
      }

    } else {
      //2=========
      participatorVos = participatorVoService.findByParticipatorTypeAndParticipatorCode(target.getActualParticipatorFlag(), target.getActualParticipatorCode());

    }
    Validate.notEmpty(participatorVos, "分利参与者信息不能为空!");
    ParticipatorVo participatorVo = participatorVos.stream().filter(item -> Objects.equals(item.getParticipatorType(), source.getParticipatorFlag()))
            .findFirst().orElse(null);
    ParticipatorVo actualParticipatorVo = participatorVos.stream().filter(item -> Objects.equals(item.getParticipatorType(), source.getActualParticipatorFlag()))
            .findFirst().orElse(null);
    Validate.notNull(actualParticipatorVo, "%实际扫码者信息不能为空!", source.getActualParticipatorCode());
    target.setActualParticipatorName(actualParticipatorVo.getParticipatorName());
    if (participatorVo != null) {
      target.setOrgCode(participatorVo.getOrgCode());
      target.setOrgName(participatorVo.getOrgName());
      target.setPhone(participatorVo.getPhone());
      target.setAdditionalParticipatorCode(participatorVo.getParticipatorCode());
      target.setAdditionalParticipatorName(participatorVo.getParticipatorName());
      target.setAdditionalParticipatorFlag(participatorVo.getParticipatorType());
      target.setChannel(participatorVo.getChannel());
    }
    return target;
  }
}
