package com.biz.crm.cps.business.reward.local.service.observer;

import cn.hutool.json.JSONUtil;
import com.biz.crm.cps.business.activity.scan.sdk.service.observer.ScanActivityRewardObserver;
import com.biz.crm.cps.business.activity.scan.sdk.vo.RewardValueVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardObserverVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.reward.sdk.dto.RewardConditionDto;
import com.biz.crm.cps.business.reward.sdk.service.observer.RewardServiceObserver;
import com.google.common.collect.Lists;
import java.util.List;
import java.util.Objects;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 扫码活动奖励发放实现
 *
 * @author sunx
 * @date 2022/3/14
 */
@Component
@Slf4j
public class ScanActivityRewardObserverImpl implements ScanActivityRewardObserver {

  @Autowired(required = false)
  private List<RewardServiceObserver> awardServiceObservers;
  /** 扫码活动正向分利 */
  private static final String SCAN_ACTIVITY_FORWARD = "scan_activity_forward";

  private static final String SCAN_ACTIVITY_FORWARD_DESC = "扫码活动分利";
  /** 扫码活动反向分利 */
  private static final String SCAN_ACTIVITY_REVERSE = "scan_activity_reverse";

  private static final String SCAN_ACTIVITY_REVERSE_DESC = "扫码活动返利";
  /** 固定额度 */
  private static final String FIXED_AMOUNT = "fixedAmount";
  /** 定额随机 */
  private static final String RANDOM_AMOUNT = "randomAmount";

  @Override
  public void createRewardRecord(List<ScanActivityRewardObserverVo> data) {
    log.debug("******createRewardRecord获取入参={}", data);
    if (CollectionUtils.isEmpty(data)) {
      return;
    }
    List<RewardConditionDto> dtoList =
        this.rewardArgumentTransform(data, SCAN_ACTIVITY_FORWARD_DESC);
    this.notify(dtoList);
  }

  /**
   * ScanActivityRewardObserverVo ->RewardConditionDto 扫码活动模块奖励条件公式-》奖励模块奖励条件公式
   *
   * @param data
   * @param rebateType
   * @return
   */
  private List<RewardConditionDto> rewardArgumentTransform(
      List<ScanActivityRewardObserverVo> data, String rebateType) {
    if (CollectionUtils.isEmpty(data)) {
      return Lists.newLinkedList();
    }
    List<RewardConditionDto> list = Lists.newArrayList();
    for (ScanActivityRewardObserverVo item : data) {
      RewardConditionDto cur = new RewardConditionDto();
      cur.setRewardMethodFlag(item.getRewardType());
      cur.setAwardConditionFlag(item.getRandomFlag() ? RANDOM_AMOUNT : FIXED_AMOUNT);
      cur.setAdditionalParticipatorCode(item.getUserCode());
      cur.setAdditionalParticipatorName(item.getName());
      cur.setChannel(item.getChannel());
      cur.setOrgCode(item.getOrgCode());
      cur.setOrgName(item.getOrgName());
      ParticipatorTypeEnum type = ParticipatorTypeEnum.getByKey(item.getParticipateObj());
      Validate.notNull(type, "参与类型异常");
      cur.setAdditionalParticipatorFlag(type.getDictCode());
      cur.setTriggerAction(item.getTriggerAction());
      cur.setTriggerObject(item.getMaterialCode());
      cur.setRecordCode(item.getRecordCode());
      cur.setRebateType(rebateType);
      cur.setActCode(item.getScanActCode());
      cur.setAccountCode(item.getAccountCode());
      cur.setRewardData(this.findRewardData(item));
      list.add(cur);
      if (!CollectionUtils.isEmpty(item.getItemList())) {
        list.addAll(this.rewardArgumentTransform(item.getItemList(), SCAN_ACTIVITY_REVERSE_DESC));
      }
    }
    return list;
  }

  /**
   * 获取 固定额度|定额随机配置信息
   *
   * @param item
   * @return
   */
  private String findRewardData(ScanActivityRewardObserverVo item) {
    if (Objects.isNull(item)
        || Objects.isNull(item.getRandomFlag())
        || CollectionUtils.isEmpty(item.getValueList())) {
      return StringUtils.EMPTY;
    }
    if (Boolean.TRUE.equals(item.getRandomFlag())) {
      return JSONUtil.toJsonStr(item.getValueList());
    }
    if (Boolean.FALSE.equals(item.getRandomFlag())) {
      RewardValueVo vo = item.getValueList().stream().findFirst().orElse(null);
      if (Objects.nonNull(vo) || Objects.nonNull(vo.getCash())) {
        return String.valueOf(vo.getCash());
      }
    }
    return StringUtils.EMPTY;
  }

  /**
   * 通知 所有奖励模块组观察者
   *
   * @param dtoList
   */
  private void notify(List<RewardConditionDto> dtoList) {
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    for (RewardConditionDto item : dtoList) {
      if (!CollectionUtils.isEmpty(this.awardServiceObservers)) {
        for (RewardServiceObserver rewardServiceObserver : this.awardServiceObservers) {
          if (StringUtils.isNotBlank(item.getAdditionalParticipatorCode())) {
            rewardServiceObserver.onCreate(item);
          } else {
            log.info(
                "{}扫码未找到所属获利者{}跳过本次返利",
                item.getActualParticipatorName(),
                item.getAdditionalParticipatorFlag());
          }
        }
        // 如果还有分利传递，继续告知流水模块组的观察者们
        this.notify(item.getRewardConditionDtos());
      }
    }
  }
}
