package com.biz.crm.cps.business.reward.redpacket.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketDetailEntity;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketEntity;
import com.biz.crm.cps.business.reward.redpacket.local.repository.RedPacketDetailRepository;
import com.biz.crm.cps.business.reward.redpacket.local.service.KeepBooksService;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketDetailService;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketService;
import com.biz.crm.cps.business.reward.redpacket.sdk.common.constant.RedPacketCodeConstant;
import com.biz.crm.cps.business.reward.redpacket.sdk.dto.RedPacketDetailDto;
import com.biz.crm.cps.business.reward.redpacket.sdk.dto.RedPacketDto;
import com.biz.crm.cps.business.reward.sdk.common.enums.IncomeStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * @ClassName: RedPacketDetailServiceImpl
 * @Author: yangfan
 * @Description: 红包详情service实现类
 * @Date: 2021/8/7 14:54
 * @Version: 1.0
 */
@Service
@Slf4j
public class RedPacketDetailServiceImpl implements RedPacketDetailService {

  @Autowired
  private RedPacketDetailRepository redPacketDetailRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private RedPacketService redPacketService;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private KeepBooksService keepBooksService;


  /**
   * 分页按条件查询 条件： 红包变动流水编码 参与者类型 参与者编码 参与者名称 手机号 触发动作 开始发放时间 结束发放时间 红包id
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<RedPacketDetailEntity> findByConditions(Pageable pageable, RedPacketDetailDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (dto == null) {
      dto = new RedPacketDetailDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.redPacketDetailRepository.findByConditions(pageable, dto);
  }

  /**
   * 昨日发放
   *
   * @param lastDay 格式：YYYY-MM-DD
   * @return
   */
  @Override
  public BigDecimal sumByLastDay(String lastDay) {

    Validate.notBlank(lastDay, "时间不能为空");
    BigDecimal amount = this.redPacketDetailRepository.sumByLastDay(lastDay);
    if (amount == null) {
      return BigDecimal.ZERO;
    }
    return amount;
  }

  @Override
  @Transactional
  public RedPacketDetailEntity create(RedPacketDetailDto redPacketDetailDto) {

    log.debug("新增红包流水传入参数={}", redPacketDetailDto);
    // dto转RedPacketDetailEntity对象
    RedPacketDetailEntity redPacketDetail = this.nebulaToolkitService.copyObjectByWhiteList(redPacketDetailDto, RedPacketDetailEntity.class, HashSet.class, ArrayList.class);
    this.createForm(redPacketDetail);
    BigDecimal balance = BigDecimal.ZERO;
    RedPacketEntity result = null;

    /**
     * 主表里面是否存在此参与者信息,如果没有需要插入一条数据到主表里面,如果有 获取主表的id
     */

    RedPacketEntity old = this.redPacketService.findByParticipatorCode(redPacketDetail.getParticipatorCode());
    if (old != null) {
      RedPacketDto redPacketDto = this.nebulaToolkitService.copyObjectByWhiteList(old, RedPacketDto.class, HashSet.class, ArrayList.class);
      // 如果不为空 更新红包表累计分利金额信息
      this.redPacketService.updateBalance(redPacketDetailDto.getParticipatorCode(), redPacketDetailDto.getAmount(), IncomeStatusEnum.getByDictCode(redPacketDetailDto.getType()));
      RedPacketEntity current = this.redPacketService.findByParticipatorCode(redPacketDetail.getParticipatorCode());
      balance = current.getBalance();
      result = current;
      this.redPacketService.updateBaseInfo(redPacketDto);
    } else {
      /**
       * 此客户没有红包信息 新增一条红包信息
       * redPacketDetailDto对象转RedPacketDto对象
       */
      Validate.isTrue(Objects.equals(IncomeStatusEnum.ADD.getDictCode(), redPacketDetail.getType()), "还未获取到红包");
      RedPacketDto redPacket = this.nebulaToolkitService.copyObjectByWhiteList(redPacketDetailDto, RedPacketDto.class, HashSet.class, ArrayList.class);
      balance = redPacketDetail.getAmount();
      redPacket.setBalance(balance);
      redPacket.setAccruingRedPacket(balance);
      result = this.redPacketService.create(redPacket);

    }
    if (Objects.equals(IncomeStatusEnum.ADD.getDictCode(), redPacketDetail.getType())) {
      // 更新账户余额
      this.keepBooksService.updateTop1SubBalance(redPacketDetail.getAmount());
    }
    Validate.notNull(result, "未找到红包余额");
    redPacketDetail.setBalance(balance);
    redPacketDetail.setRedPacketId(result.getId());
    this.redPacketDetailRepository.saveOrUpdate(redPacketDetail);
    return redPacketDetail;
  }

  @Override
  @Transactional
  public RedPacketDetailEntity update(RedPacketDetailEntity redPacketDetail) {
    RedPacketDetailEntity current = this.updateForm(redPacketDetail);

    return current;
  }

  /**
   * 组装保存entity数据
   *
   * @param redPacketDetail
   */
  private RedPacketDetailEntity updateForm(RedPacketDetailEntity redPacketDetail) {
    /*
     * 对静态模型的修改操作的过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     */
    this.updateValidation(redPacketDetail);
    //这里可根据id或者code更新
    String currentId = redPacketDetail.getId();
    RedPacketDetailEntity current = this.redPacketDetailRepository.getById(currentId);
    current = Validate.notNull(current, "未发现指定的原始模型对象信");
    BeanUtils.copyProperties(redPacketDetail, current, "id", "modifyTime", "createAccount", "createTime", "tenantCode");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    String account = this.loginUserService.getLoginAccountName();
    current.setModifyAccount(account);
    current.setModifyTime(now);
    this.redPacketDetailRepository.saveOrUpdate(current);
    return current;
  }

  /**
   * 组装保存entity数据
   *
   * @param redPacketDetail
   */
  private RedPacketDetailEntity createForm(RedPacketDetailEntity redPacketDetail) {
    if (Objects.nonNull(this.generateCodeService)) {
      redPacketDetail.setCode(this.generateCodeService.generateCode(RedPacketCodeConstant.PACKAGE_FLOW, 1).get(0));
    }
    redPacketDetail.setTenantCode(TenantUtils.getTenantCode());
    redPacketDetail.setCreateTime(new Date());
    redPacketDetail.setCreateAccount(loginUserService.getLoginAccountName());
    redPacketDetail.setModifyTime(new Date());
    redPacketDetail.setModifyAccount(loginUserService.getLoginAccountName());
    redPacketDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    redPacketDetail.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //校验
    this.createValidation(redPacketDetail);
    return redPacketDetail;
  }

  /**
   * 校验红包流水信息
   *
   * @param redPacketDetail
   */
  private void createValidation(RedPacketDetailEntity redPacketDetail) {

    // 验证必填项，以及字段长度
    Validate.notNull(redPacketDetail, "新增操作时，新增对象不能为空！");
    Validate.notBlank(redPacketDetail.getParticipatorType(), "参与者类型不能为空;");
    Validate.notBlank(redPacketDetail.getParticipatorCode(), "参与者编码不能为空;");
    Validate.notBlank(redPacketDetail.getParticipatorName(), "参与者名称不能为空;");
    Validate.isTrue(redPacketDetail.getParticipatorType().length() < 255, "参与者类型，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(redPacketDetail.getParticipatorCode().length() < 255, "参与者编码，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(redPacketDetail.getParticipatorName().length() < 255, "参与者名称，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 在更新一个的material模型对象之前，检查对象各属性的正确性
   *
   * @param redPacketDetail
   */
  private void updateValidation(RedPacketDetailEntity redPacketDetail) {
    Validate.isTrue(!StringUtils.isBlank(redPacketDetail.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    Validate.notBlank(redPacketDetail.getParticipatorType(), "参与者类型不能为空;");
    Validate.notBlank(redPacketDetail.getParticipatorCode(), "参与者编码不能为空;");
    Validate.notBlank(redPacketDetail.getParticipatorName(), "参与者名称不能为空;");
    Validate.isTrue(redPacketDetail.getParticipatorType().length() < 255, "参与者类型，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(redPacketDetail.getParticipatorCode().length() < 255, "参与者编码，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(redPacketDetail.getParticipatorName().length() < 255, "参与者名称，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 根据参与者编码和时间查询
   *
   * @param participatorCode
   * @param yearAndMonth     格式:YYYY-MM
   * @return
   */
  @Override
  public List<RedPacketDetailEntity> findByParticipatorCodeAndCreateTime(String participatorCode, String yearAndMonth) {
    if (StringUtils.isBlank(participatorCode)) {
      return Lists.newArrayList();
    }
    return redPacketDetailRepository.findByParticipatorCodeAndCreateTime(participatorCode, yearAndMonth);
  }

  @Override
  public BigDecimal sumByParticipatorCodeAndRecordCodes(String participatorCode,
          List<String> recordCodes) {
    Validate.notBlank(participatorCode, "参与者不能为空！");
    Validate.notEmpty(recordCodes, "扫码记录不能为空！");
    BigDecimal total = this.redPacketDetailRepository.sumByParticipatorCodeAndRecordCodes(participatorCode, recordCodes);
    return total;
  }

  @Override
  public BigDecimal sumByTenantCodeAndTriggerAction(String tenantCode, String triggerAction) {
    if (StringUtils.isBlank(tenantCode) || StringUtils.isBlank(triggerAction)) {
      return null;
    }
    return this.redPacketDetailRepository.sumByTenantCodeAndTriggerAction(tenantCode, triggerAction);
  }

}
