package com.biz.crm.cps.business.reward.redpacket.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.agreement.sdk.dto.LoginUserAgreementDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementVo;
import com.biz.crm.cps.business.consumer.sdk.service.ConsumerVoService;
import com.biz.crm.cps.business.consumer.sdk.vo.ConsumerVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.participator.sdk.service.ParticipatorTagVoService;
import com.biz.crm.cps.business.participator.sdk.service.ParticipatorVoService;
import com.biz.crm.cps.business.participator.sdk.vo.ParticipatorTagVo;
import com.biz.crm.cps.business.participator.sdk.vo.ParticipatorVo;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketDetailEntity;
import com.biz.crm.cps.business.reward.redpacket.local.repository.RedPacketDetailVoRepository;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketDetailService;
import com.biz.crm.cps.business.reward.redpacket.sdk.dto.RedPacketDetailDto;
import com.biz.crm.cps.business.reward.redpacket.sdk.service.RedPacketDetailVoService;
import com.biz.crm.cps.business.reward.redpacket.sdk.vo.RedPacketDetailVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: zengxingwang
 * @Date: 2021/8/10 10:53
 */
@Service
public class RedPacketDetailVoServiceImpl implements RedPacketDetailVoService {

  @Autowired
  private RedPacketDetailService redPacketDetailService;
  @Autowired
  private AgreementVoService agreementVoService;
  @Autowired
  private RedPacketDetailVoRepository redPacketDetailVoRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private LoginUserService loginUserService;

  @Autowired
  private ConsumerVoService consumerVoService;

  @Autowired
  private ParticipatorVoService participatorVoService;

  @Autowired
  private ParticipatorTagVoService participatorTagVoService;


  /**
   * 终端/经销商红包明细查看 当参数均为空时 查全部
   *
   * @param participatorCode
   * @param yearAndMonth(格式：yyyy-MM)
   * @return
   */
  @Override
  public Page<RedPacketDetailVo> findByParticipatorCodeAndCreateTimeAndType(String participatorCode,
          String yearAndMonth,String type, Pageable pageable) {
    if (StringUtils.isBlank(participatorCode)) {
      return new Page<>();
    }
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    RedPacketDetailDto queryDto = new RedPacketDetailDto();
    queryDto.setParticipatorCode(participatorCode);
    queryDto.setYearAndMonth(yearAndMonth);
    queryDto.setType(type);
    queryDto.setTenantCode(TenantUtils.getTenantCode());
    Page<RedPacketDetailEntity> result = this.redPacketDetailService.findByConditions(pageable,queryDto);
    if(Objects.isNull(result)){
      return new Page<>();
    }
    List<RedPacketDetailEntity> details = result.getRecords();
    List<RedPacketDetailVo> vos = new ArrayList<>();
    for (RedPacketDetailEntity entity : details) {
      RedPacketDetailVo vo = new RedPacketDetailVo();
      BeanUtils.copyProperties(entity,vo);
      LoginUserAgreementDto dto = new LoginUserAgreementDto();
      dto.setTerminalCode(participatorCode);
      dto.setTemplateCodes(Lists.newArrayList(entity.getTemplateCode()));
      List<AgreementVo> agreements = this.agreementVoService.findByConditions(dto);
      if (CollectionUtils.isNotEmpty(agreements)) {
        vo.setTemplateCode(agreements.get(0).getTemplateCode());
        vo.setTemplateName(agreements.get(0).getBelongTemplate());
      }
      vos.add(vo);
    }
    Page<RedPacketDetailVo> page = new Page<>();
    page.setCurrent(result.getCurrent());
    page.setSize(result.getSize());
    page.setTotal(result.getTotal());
    page.setRecords(vos);
    return page;
  }

  /**
   * 分页按条件查询 条件： 红包变动流水编码 参与者类型 参与者编码 参与者名称 手机号 触发动作 开始发放时间 结束发放时间 红包id
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<RedPacketDetailVo> findByConditions(Pageable pageable, RedPacketDetailDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (dto == null) {
      dto = new RedPacketDetailDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    try {
      //格式化时间
      SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
      String startCreateTime = dto.getStartCreateTime();
      if (StringUtils.isNotBlank(startCreateTime)) {
        dto.setStartCreateTime(DateFormatUtils.format(simpleDateFormat.parse(startCreateTime), "yyyy-MM-dd 00:00:00"));
      }
      String endCreateTime = dto.getEndCreateTime();
      if (StringUtils.isNotBlank(endCreateTime)) {
        dto.setEndCreateTime(DateFormatUtils.format(simpleDateFormat.parse(endCreateTime), "yyyy-MM-dd 23:59:59"));
      }
    } catch (ParseException e) {
      e.printStackTrace();
    }
    //消费者
    if(dto.getParticipatorType().equals(ParticipatorTypeEnum.CONSUMER.getDictCode())){
      Page<RedPacketDetailVo> consumerByConditions = this.redPacketDetailVoRepository.findConsumerByConditions(pageable, dto);
      return consumerByConditions;
    }
    //终端
    if(dto.getParticipatorType().equals(ParticipatorTypeEnum.TERMINAL.getDictCode())){
      Page<RedPacketDetailVo> terminalByConditions = this.redPacketDetailVoRepository.findTerminalByConditions(pageable, dto);
      this.buildRedPacketDetailVo(terminalByConditions);
      return terminalByConditions;
    }
    //经销商
    if(dto.getParticipatorType().equals(ParticipatorTypeEnum.DEALER.getDictCode())){
      Page<RedPacketDetailVo> dealerByConditions = this.redPacketDetailVoRepository.findDealerByConditions(pageable, dto);
      this.buildRedPacketDetailVo(dealerByConditions);
      return dealerByConditions;
    }
    Page<RedPacketDetailVo> conditions = this.redPacketDetailVoRepository.findByConditions(pageable, dto);
    this.buildRedPacketDetailVo(conditions);
    return conditions;
  }


  /**
   * 创建
   * @param dto
   * @return
   */
  @Override
  @Transactional
  public RedPacketDetailVo create(RedPacketDetailDto dto) {
    //获取当前参与者信息
    dto.setParticipatorCode(loginUserService.getLoginDetails(LoginUserDetailsForCPS.class).getConsumerCode());
    dto.setParticipatorName(loginUserService.getLoginDetails(LoginUserDetailsForCPS.class).getConsumerName());
    ParticipatorTypeEnum participatorTypeEnum = ParticipatorTypeEnum.getByKey(loginUserService.getLoginDetails(LoginUserDetailsForCPS.class).getUsertype());
    dto.setParticipatorType(participatorTypeEnum.getDictCode());
    Validate.notNull(dto, "新增信息时，入参对象不能为空！");
    Validate.notBlank(dto.getTriggerAction(),"触发动作不能为空");
    Validate.notBlank(dto.getParticipatorCode(),"参与者编码不能为空");
    Validate.notBlank(dto.getParticipatorType(),"参与者类型不能为空");
    Validate.notNull(dto.getAmount(),"红包变动数不能为空");
    if (ParticipatorTypeEnum.CONSUMER.getDictCode().equals(dto.getParticipatorType())) {
      ConsumerVo consumerCode = consumerVoService.findByExternalId(dto.getParticipatorCode());
      Validate.notNull(consumerCode,"未找到指定参与者");
    } else {
      List<ParticipatorVo> list = this.participatorVoService.findByParticipatorTypeAndParticipatorCode(dto.getParticipatorType(),dto.getParticipatorCode());
      Validate.notEmpty(list,"未找到指定参与者");
      ParticipatorVo participatorVo = list.stream().filter(item->Objects.equals(item.getParticipatorType(),dto.getParticipatorType())).findFirst().orElse(null);
      Validate.notNull(participatorVo, "未找到指定参与者！");
      dto.setOrgCode(participatorVo.getOrgCode());
      dto.setOrgName(participatorVo.getOrgName());
      dto.setChannel(participatorVo.getChannel());
      dto.setPhone(participatorVo.getPhone());
      dto.setParticipatorName(participatorVo.getParticipatorName());
    }
    RedPacketDetailEntity redPacketDetailEntity = this.redPacketDetailService.create(dto);
    Validate.notNull(redPacketDetailEntity, "新增信息时，返回对象不能为空！");
    return this.nebulaToolkitService.copyObjectByWhiteList(redPacketDetailEntity, RedPacketDetailVo.class, null, null);
  }

  /**
   * 组装返回数据
   * @param packetDetailVoPage
   */
  private void buildRedPacketDetailVo(Page<RedPacketDetailVo> packetDetailVoPage) {
    if (packetDetailVoPage == null) {
      return;
    }
    List<RedPacketDetailVo> records = packetDetailVoPage.getRecords();
    if (org.springframework.util.CollectionUtils.isEmpty(records)) {
      return;
    }
    List<String> codeList = records.stream()
        .map(redPacketDetailVo -> redPacketDetailVo.getParticipatorCode())
        .collect(Collectors.toList());
    //标签
    Map<String, List<ParticipatorTagVo>> participatorTagMap = new HashMap<>();
    List<ParticipatorTagVo> participatorTagList = this.participatorTagVoService.findByParticipatorCodes(codeList);
    if (!org.springframework.util.CollectionUtils.isEmpty(participatorTagList)) {
      participatorTagMap = participatorTagList.stream()
          .collect(Collectors.groupingBy(ParticipatorTagVo::getParticipatorCode));
    }
    for (RedPacketDetailVo entity : records) {
      //赋值标签
      List<ParticipatorTagVo> tags = participatorTagMap.get(entity.getParticipatorCode());
      if(!org.springframework.util.CollectionUtils.isEmpty(tags)){
        String participatorTag = tags.stream().map(tag -> tag.getTagDescription()).collect(Collectors.joining(";"));
        entity.setTags(participatorTag);
      }
    }
  }
}
