package com.biz.crm.cps.business.reward.redpacket.local.service.internal;

import cn.hutool.core.date.DateUtil;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.reward.redpacket.local.config.CashProperties;
import com.biz.crm.cps.business.reward.redpacket.local.config.RedPacketProperties;
import com.biz.crm.cps.business.reward.redpacket.local.entity.MaxAmountLimitEntity;
import com.biz.crm.cps.business.reward.redpacket.local.entity.MaxCountLimitEntity;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketConfigEntity;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketEntity;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketConfigService;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketService;
import com.biz.crm.cps.business.reward.redpacket.sdk.service.RedPacketRewardCashProcessVoService;
import com.biz.crm.cps.business.reward.sdk.dto.RewardCashConditionDto;
import com.biz.crm.cps.business.reward.sdk.event.RequestCashListener;
import com.biz.crm.cps.business.reward.sdk.service.observer.RewardCashObserver;
import com.biz.crm.cps.business.reward.sdk.vo.RewardCashProcessVo;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description: 红包提现流程实现
 * @date 2021/8/29 上午11:39
 */
@Service("RedPacketRewardCashProcessVoServiceImpl")
public class RedPacketRewardCashProcessVoServiceImpl implements RedPacketRewardCashProcessVoService {

  private static final String DAY = "day";
  private static final String MONTH = "month";

  @Autowired(required = false)
  private List<RewardCashObserver> rewardCashObserverList;
  @Autowired(required = false)
  private RedPacketProperties redPacketProperties;
  @Autowired
  private RedPacketService redPacketService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired
  private RedPacketConfigService redPacketConfigService;
  @Autowired
  private RequestCashListener requestCashListener;

  @Override
  public RewardCashProcessVo findByParticipator(RewardCashConditionDto conditionDto) {
    if (CollectionUtils.isEmpty(this.rewardCashObserverList) || Objects.isNull(conditionDto)) {
      return null;
    }
    for (RewardCashObserver rewardCashObserver : rewardCashObserverList) {
      this.buildCashType(conditionDto);
      RewardCashProcessVo cashProcessVo = rewardCashObserver.onRequestCashProcess(conditionDto);
      return cashProcessVo;
    }
    return null;
  }

  @Override
  public RewardCashProcessVo findByCurrentUser() {
    LoginUserDetailsForCPS userDetails = this.loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    ParticipatorTypeEnum participatorTypeEnum = ParticipatorTypeEnum.getByKey(userDetails.getUsertype());
    RewardCashConditionDto conditionDto = new RewardCashConditionDto();
    conditionDto.setParticipatorCode(userDetails.getConsumerCode());
    conditionDto.setParticipatorName(userDetails.getConsumerName());
    conditionDto.setParticipatorFlag(participatorTypeEnum.getDictCode());
    conditionDto.setPayeeCode(userDetails.getAccount());
    conditionDto.setPayeeName(userDetails.getUsername());
    return this.findByParticipator(conditionDto);
  }

  @Override
  public void validate(RewardCashConditionDto conditionDto) {
    Validate.notNull(conditionDto, "兑付条件不能为空");
    BigDecimal amount = conditionDto.getAmount();
    String participatorCode = conditionDto.getParticipatorCode();
    String participatorName = conditionDto.getParticipatorName();
    String participatorType = conditionDto.getParticipatorFlag();
    Validate.notNull(amount, "减少余额时，金额不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "减少余额时，金额必须大于0");
    Validate.notBlank(participatorType, "参与者类型不能为空");
    Validate.notBlank(participatorName, "参与者名称不能为空");
    Validate.notBlank(participatorCode, "参与者编码不能为空");
    //校验 红包提现规则
    this.cashValidate(conditionDto);
    // 检查是否需要签署服务合同、是否需要认证
    RewardCashProcessVo rewardCashProcessVo = this.findByParticipator(conditionDto);
    Validate.notNull(rewardCashProcessVo, "【%s】：缺少兑现配置", participatorName);
    Validate.isTrue(!rewardCashProcessVo.isNeedSign(), "【%s】：兑现需要签署服务合同", participatorName);
    Validate.isTrue(!rewardCashProcessVo.isNeedVerify(), "【%s】：兑现需要实名认证", participatorName);
    //验证红包余额
    RedPacketEntity redPacket = this.redPacketService.findByParticipatorCode(participatorCode);
    Validate.notNull(redPacket, "【%s】：还未获得红包", participatorName);
    BigDecimal balance = redPacket.getBalance() == null ? BigDecimal.ZERO : redPacket.getBalance();
    Validate.isTrue(balance.compareTo(amount) >= 0, "提现金额大于余额");
  }


  @Override
  public void buildCashType(RewardCashConditionDto conditionDto) {
    Validate.notNull(conditionDto, "兑付条件不能为空");
    List<CashProperties> cashProperties = this.redPacketProperties.getCashs();
    Validate.notEmpty(cashProperties, "系统缺少红包提现配置");
    CashProperties cashProp = cashProperties.stream().filter(item -> Objects.equals(conditionDto.getParticipatorFlag(), item.getParticipatorType())).findAny().orElse(null);
    Validate.notNull(cashProp, "系统缺少【%s】类型的红包提现配置", conditionDto.getParticipatorFlag());
    conditionDto.setCashKey(cashProp.getCashKey());
  }

  /**
   * 此私有方法是为了验证红包配置是否满足提现要求
   */
  private void cashValidate(RewardCashConditionDto conditionDto) {
    List<RedPacketConfigEntity> configAll = this.redPacketConfigService.findAll();
    if (CollectionUtils.isEmpty(configAll)) {
      return;
    }
    RedPacketConfigEntity configEntity = configAll.stream()
        .filter(redPacketConfigEntity -> redPacketConfigEntity.getType().equals(conditionDto.getParticipatorFlag()))
        .findFirst()
        .orElse(null);
    if (configEntity == null) {
      return;
    }
    //验证余额是否达到可提现额度
    if (configEntity.getIsAmountLimit() == 1) {
      BigDecimal balanceByParticipatorCode = this.redPacketService.findBalanceByParticipatorCode(conditionDto.getParticipatorCode());
      balanceByParticipatorCode = balanceByParticipatorCode == null ? BigDecimal.ZERO : balanceByParticipatorCode;
      Validate.isTrue(balanceByParticipatorCode.compareTo(configEntity.getAmountLimit()) >= 0, "当前余额不满足提现门槛！");
    }
    //验证单次红包提现金额
    if (configEntity.getIsOneCountLimit() == 1) {
      Validate.isTrue(conditionDto.getAmount().compareTo(configEntity.getOneAmountLimit()) < 1, "提现金额超过单次最高提现金额！");
    }
    //验证红包提现金额上限设置
    if (configEntity.getIsMaxAmountLimit() == 1) {
      Set<MaxAmountLimitEntity> amountLimitDetails = configEntity.getAmountLimitDetails();
      Validate.isTrue(!CollectionUtils.isEmpty(amountLimitDetails), "未查询到提现金额上限设置");
      Map<String, List<MaxAmountLimitEntity>> map = amountLimitDetails.stream().collect(Collectors.groupingBy(MaxAmountLimitEntity::getTimeType));
      List<MaxAmountLimitEntity> days = map.get(DAY);
      List<MaxAmountLimitEntity> months = map.get(MONTH);
      Date date = new Date();
      if (!CollectionUtils.isEmpty(days)) {
        MaxAmountLimitEntity day = days.get(0);
        BigDecimal bigDecimal = this.requestCashListener.onRequestAmountByParticipatorAndDateTime(conditionDto.getParticipatorCode(), conditionDto.getParticipatorFlag(), DateUtil.beginOfDay(date), DateUtil.endOfDay(date));
        Validate.isTrue(bigDecimal.add(conditionDto.getAmount()).compareTo(day.getAmountLimit()) < 1, "累计提现金额已超过上限！");
      }
      if (!CollectionUtils.isEmpty(months)) {
        MaxAmountLimitEntity month = months.get(0);
        BigDecimal bigDecimal = this.requestCashListener.onRequestAmountByParticipatorAndDateTime(conditionDto.getParticipatorCode(), conditionDto.getParticipatorFlag(), DateUtil.beginOfMonth(date), DateUtil.endOfMonth(date));
        Validate.isTrue(bigDecimal.add(conditionDto.getAmount()).compareTo(month.getAmountLimit()) < 1, "累计提现金额已超过上限！");
      }
    }
    //验证红包提现次数上限设置
    if (configEntity.getIsMaxCountLimit() == 1) {
      Set<MaxCountLimitEntity> countLimitDetails = configEntity.getCountLimitDetails();
      Validate.isTrue(!CollectionUtils.isEmpty(countLimitDetails), "未查询红包提现次数上限设置");
      Map<String, List<MaxCountLimitEntity>> map = countLimitDetails.stream().collect(Collectors.groupingBy(MaxCountLimitEntity::getTimeType));
      List<MaxCountLimitEntity> days = map.get(DAY);
      List<MaxCountLimitEntity> months = map.get(MONTH);
      Date date = new Date();
      if (!CollectionUtils.isEmpty(days)) {
        MaxCountLimitEntity day = days.get(0);
        Integer integer = this.requestCashListener.onRequestCountByParticipatorAndDateTime(conditionDto.getParticipatorCode(), conditionDto.getParticipatorFlag(), DateUtil.beginOfDay(date), DateUtil.endOfDay(date));
        Validate.isTrue((integer +1) <= day.getCountLimit(), "累计提现次数已超过上限！");
      }
      if (!CollectionUtils.isEmpty(months)) {
        MaxCountLimitEntity month = months.get(0);
        Integer integer = this.requestCashListener.onRequestCountByParticipatorAndDateTime(conditionDto.getParticipatorCode(), conditionDto.getParticipatorFlag(), DateUtil.beginOfMonth(date), DateUtil.endOfMonth(date));
        Validate.isTrue((integer +1) <= month.getCountLimit(), "累计提现次数已超过上限！");
      }
    }

  }
}
