package com.biz.crm.cps.business.reward.redpacket.local.service.internal;

import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.consumer.sdk.service.ConsumerVoService;
import com.biz.crm.cps.business.consumer.sdk.vo.ConsumerVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketDetailEntity;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketDetailService;
import com.biz.crm.cps.business.reward.redpacket.sdk.common.constant.RedPacketRedisKeys;
import com.biz.crm.cps.business.reward.redpacket.sdk.dto.RedPacketDetailDto;
import com.biz.crm.cps.business.reward.redpacket.sdk.service.RedPacketRewardCashProcessVoService;
import com.biz.crm.cps.business.reward.sdk.common.enums.CashActionEnum;
import com.biz.crm.cps.business.reward.sdk.common.enums.IncomeStatusEnum;
import com.biz.crm.cps.business.reward.sdk.dto.RewardCashConditionDto;
import com.biz.crm.cps.business.reward.sdk.service.RewardCashVoService;
import com.biz.crm.cps.business.reward.sdk.service.observer.RewardCashObserver;
import com.biz.crm.cps.business.reward.sdk.service.observer.RewardMountRegister;
import com.biz.crm.cps.business.reward.sdk.service.observer.RewardTypeStatisticsServiceObserver;
import com.biz.crm.cps.business.reward.sdk.vo.RewardCashRecordVo;
import com.biz.crm.cps.business.reward.sdk.vo.RewardTypeStatisticsVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author hecheng
 * @description: 红包兑现实现类
 * @date 2021/8/26 下午9:36
 */
@Service("RedPacketRewardCashVoServiceImpl")
public class RedPacketRewardCashVoServiceImpl implements RewardCashVoService {

  private static final String WEIXINPAY = "weixinpay";

  @Autowired(required = false)
  private List<RewardCashObserver> rewardCashObserverList;
  @Autowired
  private RedPacketRewardCashProcessVoService redPacketRewardCashProcessVoService;
  @Autowired
  private RedPacketDetailService redPacketDetailService;
  @Autowired
  @Qualifier("RedpacketRewardMountRegisterServiceImpl")
  private RewardMountRegister rewardMountRegister;
  @Autowired
  @Qualifier("RedPacketRewardTypeStatisticsServiceObserverImpl")
  private RewardTypeStatisticsServiceObserver rewardTypeStatisticsServiceObserver;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired
  private ConsumerVoService consumerVoService;
  @Autowired
  private RedisMutexService redisMutexService;

  @Override
  @Transactional
  public void cash(RewardCashConditionDto conditionDto) {
    /**
     * 1校验参数、校验提现前置条件，余额等
     * 2扣减红包余额、生成红包流水记录
     * 3通知兑现模块，完成发放
     */
    //====1
    Validate.notNull(conditionDto, "兑付条件不能为空");
    Validate.notNull(conditionDto.getAmount(), "提现金额不能为空");
    Validate.notBlank(conditionDto.getParticipatorFlag(), "参与者类型不能为空");
    Validate.notBlank(conditionDto.getParticipatorCode(), "参与者编码不能为空");
    Validate.notEmpty(this.rewardCashObserverList, "兑现模块不存在");
    Validate.isTrue(conditionDto.getAmount().compareTo(BigDecimal.ZERO) > 0, "提现金额必须大于0");
    String tenantCode = TenantUtils.getTenantCode();
    String redisKey = String.format(RedPacketRedisKeys.TENANT_CASH_BUSINESS_LOCK_PREFIX, tenantCode, conditionDto.getParticipatorCode());
    boolean locked = false;
    try {
      locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
      Validate.isTrue(locked, "红包提现繁忙请稍后再试！");
      this.redPacketRewardCashProcessVoService.buildCashType(conditionDto);
      //1校验是否满足提现前置条件
      this.redPacketRewardCashProcessVoService.validate(conditionDto);
      //====2
      RedPacketDetailDto redPacketDetailDto = new RedPacketDetailDto();
      redPacketDetailDto.setAmount(conditionDto.getAmount());
      redPacketDetailDto.setParticipatorCode(conditionDto.getParticipatorCode());
      redPacketDetailDto.setParticipatorType(conditionDto.getParticipatorFlag());
      redPacketDetailDto.setParticipatorName(conditionDto.getParticipatorName());
      redPacketDetailDto.setTriggerAction(CashActionEnum.WITHDRAW.getValue());
      redPacketDetailDto.setTriggerObject("");
      redPacketDetailDto.setType(IncomeStatusEnum.SUBTRACT.getDictCode());
      redPacketDetailDto.setRebateType(CashActionEnum.WITHDRAW.getValue());
      RedPacketDetailEntity redPacketDetail = this.redPacketDetailService.create(redPacketDetailDto);
      conditionDto.setRewardId(redPacketDetail.getId());
      conditionDto.setRewardKey(this.rewardMountRegister.getKey());
      conditionDto.setRewardName(this.rewardMountRegister.getName());
      //====3
      for (RewardCashObserver rewardCashObserver : rewardCashObserverList) {
        RewardCashRecordVo rewardCashRecordVo = rewardCashObserver.onCash(conditionDto);
        if (rewardCashRecordVo != null) {
          redPacketDetail.setTriggerObject(rewardCashRecordVo.getCashCode());
          this.redPacketDetailService.update(redPacketDetail);
          return;
        }
      }
    } finally {
      if (locked) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  @Transactional
  public void currentUserCash(BigDecimal amount, String operatorOpenId, String caseKey) {
    Validate.notNull(amount, "提现金额不能为空");
    Validate.notNull(caseKey, "提现类型不能为空");
    LoginUserDetailsForCPS userDetails = this.loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    ParticipatorTypeEnum participatorTypeEnum = ParticipatorTypeEnum.getByKey(userDetails.getUsertype());
    RewardCashConditionDto conditionDto = new RewardCashConditionDto();
    conditionDto.setParticipatorCode(userDetails.getConsumerCode());
    conditionDto.setParticipatorName(userDetails.getConsumerName());
    conditionDto.setParticipatorFlag(participatorTypeEnum.getDictCode());
    conditionDto.setPayeeCode(userDetails.getAccount());
    conditionDto.setPayeeName(userDetails.getUsername());
    conditionDto.setPayeePhone(userDetails.getPhone());
    conditionDto.setAmount(amount);
    conditionDto.setOpenId(userDetails.getOpenId());
    if (caseKey.equals(WEIXINPAY)) {
      Validate.isTrue(Objects.equals(operatorOpenId, userDetails.getOpenId()), "提现操作者openid和当前账户openid不匹配，认定为非本人操作");
    }
    this.cash(conditionDto);
  }

  @Override
  @Transactional
  public void consumerCash(BigDecimal amount, String openId) {
    Validate.notNull(amount, "提现金额不能为空");
    Validate.notBlank(openId, "消费者提现时openid不能为空");
    ConsumerVo consumerVo = this.consumerVoService.findByExternalId(openId);
    Validate.notNull(consumerVo, "消费者提现时openid未查询到消费者信息");
    RewardCashConditionDto conditionDto = new RewardCashConditionDto();
    conditionDto.setParticipatorCode(consumerVo.getConsumerCode());
    conditionDto.setParticipatorName(consumerVo.getNickname());
    conditionDto.setParticipatorFlag(ParticipatorTypeEnum.CONSUMER.getDictCode());
    conditionDto.setPayeeCode(consumerVo.getExternalId());
    conditionDto.setPayeeName(consumerVo.getNickname());
    conditionDto.setPayeePhone(consumerVo.getPhone());
    conditionDto.setAmount(amount);
    conditionDto.setOpenId(consumerVo.getExternalId());
    this.cash(conditionDto);
  }

  @Override
  public BigDecimal findByConsumerOpenIdRecordCodes(String openId, List<String> recordCodes) {
    ConsumerVo consumerVo = this.consumerVoService.findByExternalId(openId);
    if (Objects.isNull(consumerVo)) {
      return null;
    }
    RewardTypeStatisticsVo statisticsVo = this.rewardTypeStatisticsServiceObserver.onRequestRewardTypeStatisticsVo(consumerVo.getConsumerCode(), recordCodes);
    if (Objects.isNull(statisticsVo)) {
      return null;
    }
    return statisticsVo.getAmount();
  }
}
