package com.biz.crm.cps.business.reward.redpacket.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.reward.redpacket.local.entity.RedPacketEntity;
import com.biz.crm.cps.business.reward.redpacket.local.repository.RedPacketRepository;
import com.biz.crm.cps.business.reward.redpacket.local.service.RedPacketService;
import com.biz.crm.cps.business.reward.redpacket.sdk.dto.CustomerRedPacketBalanceDto;
import com.biz.crm.cps.business.reward.redpacket.sdk.dto.RedPacketDto;
import com.biz.crm.cps.business.reward.sdk.common.enums.IncomeStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;

/** @Author: zengxingwang @Date: 2021/8/7 14:29 */
@Service
@Slf4j
@ConditionalOnMissingBean(name = "RedPacketServiceExpandImpl")
public class RedPacketServiceImpl implements RedPacketService {

  @Autowired private RedPacketRepository redPacketRepository;

  @Autowired private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  /**
   * 条件查询分客户红包列表
   *
   * @param balanceDto
   * @param pageable
   * @return
   */
  @Override
  public Page<RedPacketEntity> findByConditions(
      CustomerRedPacketBalanceDto balanceDto, Pageable pageable) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (balanceDto == null) {
      balanceDto = new CustomerRedPacketBalanceDto();
    }
    balanceDto.setTenantCode(TenantUtils.getTenantCode());
    return redPacketRepository.findByConditions(pageable, balanceDto);
  }

  @Override
  public RedPacketEntity findByParticipatorCode(String participatorCode) {
    if (StringUtils.isBlank(participatorCode)) {
      return null;
    }
    return redPacketRepository.findByParticipatorCode(participatorCode);
  }

  @Override
  @Transactional
  public RedPacketEntity create(RedPacketDto redPacketDto) {
    log.debug("保存红包信息入参={}", redPacketDto);
    this.createValidation(redPacketDto);
    RedPacketEntity redPacketEntity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            redPacketDto, RedPacketEntity.class, HashSet.class, ArrayList.class);

    // 组装保存字段
    this.createForm(redPacketEntity);
    this.redPacketRepository.saveOrUpdate(redPacketEntity);
    return redPacketEntity;
  }

  @Override
  @Transactional
  public RedPacketEntity update(RedPacketDto redPacketDto) {
    this.updateValidation(redPacketDto);
    RedPacketEntity redPacketEntity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            redPacketDto, RedPacketEntity.class, HashSet.class, ArrayList.class);
    RedPacketEntity old = this.redPacketRepository.getById(redPacketEntity.getId());
    Validate.notNull(old, "未发现指定的原始模型对象信");
    BeanUtils.copyProperties(
        redPacketEntity,
        old,
        "id",
        "modifyTime",
        "createAccount",
        "createTime",
        "tenantCode",
        "balance",
        "accruingRedPacket");
    // 开始赋值——更新时间与更新人
    if (redPacketDto.getAccruingRedPacket() != null) {
      old.setAccruingRedPacket(redPacketDto.getAccruingRedPacket());
    }
    if (redPacketDto.getBalance() != null) {
      old.setBalance(redPacketDto.getBalance());
    }
    Date now = new Date();
    String account = this.loginUserService.getLoginAccountName();
    old.setModifyAccount(account);
    old.setModifyTime(now);
    this.redPacketRepository.saveOrUpdate(old);
    return old;
  }

  @Override
  @Transactional
  public void updateBalance(
      String participatorCode, BigDecimal amount, IncomeStatusEnum incomeStatus) {
    Validate.notBlank(participatorCode, "参与者不能为空");
    Validate.notNull(amount, "金额不能为空");
    Validate.notNull(incomeStatus, "收益状态不能为空");
    if (IncomeStatusEnum.ADD == incomeStatus) {
      long count = this.redPacketRepository.addBalance(participatorCode, amount);
      Validate.isTrue(count > 0, "红包余额新增失败");
    } else if (IncomeStatusEnum.SUBTRACT == incomeStatus) {
      long count = this.redPacketRepository.subtractBalance(participatorCode, amount);
      Validate.isTrue(count > 0, "红包余额扣减失败");
    }
  }

  @Transactional
  @Override
  public void updateBaseInfo(RedPacketDto redPacketDto) {
    Validate.notBlank(redPacketDto.getParticipatorCode(), "参与者不能为空");
    if (org.apache.commons.lang3.StringUtils.isAllBlank(
        redPacketDto.getChannel(),
        redPacketDto.getOrgCode(),
        redPacketDto.getOrgName(),
        redPacketDto.getTag())) {
      return;
    }
    this.redPacketRepository.updateBaseInfo(redPacketDto);
  }

  @Override
  public BigDecimal findBalanceByParticipatorCode(String participatorCode) {
    if (StringUtils.isBlank(participatorCode)) {
      return BigDecimal.ZERO;
    }
    return this.redPacketRepository.findBalanceByParticipatorCode(participatorCode);
  }

  /**
   * 校验红包流水信息
   *
   * @param redPacketDto
   */
  private void updateValidation(RedPacketDto redPacketDto) {
    // 验证必填项，以及字段长度
    Validate.notNull(redPacketDto, "修改信息时，修改对象不能为空！");
    Validate.notBlank(redPacketDto.getId(), "修改信息时，id不能为空;");
    Validate.notBlank(redPacketDto.getParticipatorType(), "修改信息时，参与者类型不能为空;");
    Validate.notBlank(redPacketDto.getParticipatorCode(), "修改信息时，参与者编码不能为空;");
    Validate.notBlank(redPacketDto.getParticipatorName(), "修改信息时，参与者名称不能为空;");
    Validate.isTrue(
        redPacketDto.getParticipatorType().length() < 255, "参与者类型，在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(
        redPacketDto.getParticipatorCode().length() < 255, "参与者编码，在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(
        redPacketDto.getParticipatorName().length() < 255, "参与者名称，在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 校验红包流水信息
   *
   * @param redPacketDto
   */
  private void createValidation(RedPacketDto redPacketDto) {
    // 验证必填项，以及字段长度
    Validate.notNull(redPacketDto, "新增操作时，新增对象不能为空！");
    Validate.notBlank(redPacketDto.getParticipatorType(), "参与者类型不能为空;");
    Validate.notBlank(redPacketDto.getParticipatorCode(), "参与者编码不能为空;");
    Validate.notBlank(redPacketDto.getParticipatorName(), "参与者名称不能为空;");
    Validate.isTrue(
        redPacketDto.getParticipatorType().length() < 255, "参与者类型，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(
        redPacketDto.getParticipatorCode().length() < 255, "参与者编码，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(
        redPacketDto.getParticipatorName().length() < 255, "参与者名称，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 组装保存entity数据
   *
   * @param redPacketEntity
   */
  private void createForm(RedPacketEntity redPacketEntity) {
    redPacketEntity.setId(null);
    redPacketEntity.setTenantCode(TenantUtils.getTenantCode());
    redPacketEntity.setCreateTime(new Date());
    redPacketEntity.setCreateAccount(loginUserService.getLoginAccountName());
    redPacketEntity.setModifyTime(new Date());
    redPacketEntity.setModifyAccount(loginUserService.getLoginAccountName());
    redPacketEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    redPacketEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
  }
}
