package com.biz.crm.cps.business.signtask.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.constant.CommonConstant;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskActivity;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskActivityDetail;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskActivityDetailRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskActivityPictureRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskActivityRepository;
import com.biz.crm.cps.business.signtask.local.service.SignTaskActivityService;
import com.biz.crm.cps.business.signtask.sdk.constant.SignTaskCodeConstant;
import com.biz.crm.cps.business.signtask.sdk.dto.SignTaskActivityDto;
import com.biz.crm.cps.business.signtask.sdk.enums.SignTaskStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @description: 活动签收任务主表service
 * @author: rentao
 * @date: 2021/12/23 17:26
 */
@Service
@Slf4j
public class SignTaskActivityServiceImpl implements SignTaskActivityService {

  @Autowired
  private SignTaskActivityRepository signTaskActivityRepository;
  @Autowired
  private SignTaskActivityDetailRepository signTaskActivityDetailRepository;
  @Autowired
  private SignTaskActivityPictureRepository signTaskActivityPictureRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private LoginUserService loginUserService;

  /**
   * 分页条件查询活动签收任务
   *
   * @param pageable            分页信息
   * @param signTaskActivityDto 分页查询dto
   * @return 分页返回信息
   */
  @Override
  public Page<SignTaskActivity> findByConditions(Pageable pageable,
      SignTaskActivityDto signTaskActivityDto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if(Objects.isNull(signTaskActivityDto)){
      signTaskActivityDto = new SignTaskActivityDto();
    }
    if(CommonConstant.QUERY_FLAG_H5.equals(signTaskActivityDto.getQueryFlag())){
      //判断用户是否登录
      LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
      log.info("当前登录用户信息：" + loginUser);
      if (!Objects.isNull(loginUser) && StringUtils.isNotBlank(loginUser.getConsumerCode())) {
        signTaskActivityDto.setClientCode(loginUser.getConsumerCode());
      }else {
        return null;
      }
    }
    signTaskActivityDto.setTenantCode(TenantUtils.getTenantCode());
    Page<SignTaskActivity> signTaskPage = this.signTaskActivityRepository
        .findByConditions(pageable, signTaskActivityDto);
    this.setSignTaskActivitySignResourceName(signTaskPage.getRecords());
    return signTaskPage;
  }

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   * @return 活动签收任务
   */
  @Override
  public SignTaskActivity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SignTaskActivity signTaskActivity = this.signTaskActivityRepository.getById(id);
    signTaskActivity.setSignTaskActivityDetails(signTaskActivityDetailRepository
        .findBySignTaskActivityCode(signTaskActivity.getSignTaskActivityCode()));
    signTaskActivity.setSignTaskActivityPictures(signTaskActivityPictureRepository
        .findBySignTaskActivityCode(signTaskActivity.getSignTaskActivityCode()));
    return signTaskActivity;
  }

  /**
   * 按照签约配置编码查询(不包含明细)
   *
   * @param signTaskCodes
   * @return 活动签收任务
   */
  @Override
  public List<SignTaskActivity> findBySignTaskCodes(List<String> signTaskCodes) {
    if (CollectionUtils.isEmpty(signTaskCodes)) {
      return null;
    }
    return this.signTaskActivityRepository.findBySignTaskCodes(signTaskCodes);
  }

  /**
   * 签约
   *
   * @param signTaskActivity 签约
   * @return
   */
  @Override
  @Transactional
  public void updateSign(SignTaskActivity signTaskActivity) {
    Validate.notNull(signTaskActivity, "签约时数据不能为空");
    Validate.notEmpty(signTaskActivity.getId(), "签约时活动签约任务id不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(signTaskActivity.getSignTaskActivityDetails()),
        "签约时活动签约任务详情不能为空");
    if (signTaskActivity.getPhotograph()) {
      Validate.isTrue(!CollectionUtils.isEmpty(signTaskActivity.getSignTaskActivityPictures()),
          "签约时活动签约任务照片不能为空");
    } else {
      signTaskActivity.setSignTaskActivityPictures(null);
    }
    SignTaskActivity taskActivity = signTaskActivityRepository.getById(signTaskActivity.getId());
    Validate.notNull(taskActivity, "签约时数据不能为空");
    Validate.isTrue(taskActivity.getEnableStatus().equals(EnableStatusEnum.ENABLE.getCode()),
        "数据禁用不能签约");
    Validate
        .isTrue(SignTaskStatusEnum.STATUS_WAIT.getKey().equals(taskActivity.getSignTaskStatus()),
            "数据状态只能为未签约");
    taskActivity.setSignTaskStatus(SignTaskStatusEnum.STATUS_COMPLETE.getKey());
    taskActivity.setSignDate(new Date());
    signTaskActivityRepository.updateById(taskActivity);
    //
    signTaskActivityPictureRepository
        .deleteBySignTaskActivityCode(signTaskActivity.getSignTaskActivityCode());
    if (signTaskActivity.getPhotograph()) {
      signTaskActivity.getSignTaskActivityPictures().forEach(
          signTaskActivityPicture -> signTaskActivityPicture
              .setSignTaskActivityCode(taskActivity.getSignTaskActivityCode()));
      this.signTaskActivityPictureRepository
          .saveBatch(signTaskActivity.getSignTaskActivityPictures());
    }
  }

  /**
   * 活动签收任务新增
   *
   * @param signTaskActivity
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public SignTaskActivity create(SignTaskActivity signTaskActivity) {
    SignTaskActivity task = this.createForm(signTaskActivity);
    this.signTaskActivityRepository.save(task);
    //保存明细
    if (!CollectionUtils.isEmpty(signTaskActivity.getSignTaskActivityDetails())) {
      task.getSignTaskActivityDetails()
          .forEach(signTaskActivityDetail -> signTaskActivityDetail
              .setSignTaskActivityCode(task.getSignTaskActivityCode()));
      this.signTaskActivityDetailRepository.saveBatch(task.getSignTaskActivityDetails());
    }
    return task;
  }

  /**
   * 活动签收批量任务新增
   *
   * @param signTaskActivitys
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public void createBatch(List<SignTaskActivity> signTaskActivitys) {
    if (CollectionUtils.isEmpty(signTaskActivitys)) {
      return;
    }
    signTaskActivitys.forEach(signTaskActivity -> {
      this.create(signTaskActivity);
    });
  }

  /**
   * 通过id删除签约任务
   *
   * @param signTaskActivities 活动任务集合
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public void deleteBatch(List<SignTaskActivity> signTaskActivities) {
    if (CollectionUtils.isEmpty(signTaskActivities)) {
      return;
    }
    List<String> ids = signTaskActivities.stream().map(SignTaskActivity::getId)
        .collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(signTaskActivities)) {
      List<SignTaskActivity> collect = signTaskActivities.stream().filter(
          signTaskActivity -> !SignTaskStatusEnum.STATUS_WAIT.getKey()
              .equals(signTaskActivity.getSignTaskStatus())).collect(
          Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(collect), "存在已过期或者已签约活动任务，不能删除或编辑");
      List<String> signTaskActivityCodes = signTaskActivities.stream()
          .map(SignTaskActivity::getSignTaskActivityCode).collect(Collectors.toList());
      signTaskActivityRepository.removeByIds(ids);
      signTaskActivityDetailRepository.deleteBySignTaskActivityCodes(signTaskActivityCodes);
    }
  }

  /**
   * 触发活动签约任务方法(定时器) step1:过期结束时间为前一天的活动签约任务
   *
   * @author rentao
   */
  @Override
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "过期活动签约任务")
  @Transactional
  public void handleSignTaskActivity() {
    //过期结束时间为前一天得任务
    Date now = new Date();
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(now);
    int nowDay = calendar.get(Calendar.DAY_OF_MONTH);
    //今天开始时间
    Date startTime = DateUtils.ceiling(DateUtils.setDays(now, nowDay - 1), Calendar.DAY_OF_MONTH);
    //昨天结束时间
    Date endTime = DateUtils.addSeconds(startTime, -1);
    //查询之前所有得活动签约任务
    List<SignTaskActivity> signTaskActivities = this.signTaskActivityRepository
        .findBySignTaskEndTime(endTime);
    if (!CollectionUtils.isEmpty(signTaskActivities)) {
      signTaskActivities.forEach(s -> {
        s.setSignTaskStatus(SignTaskStatusEnum.STATUS_EXPIRED.getKey());
      });
      List<List<SignTaskActivity>> partition = Lists.partition(signTaskActivities, 500);
      partition.forEach(s -> {
        this.signTaskActivityRepository.updateBatchById(s);
      });
    }
  }

  /**
   * 批量启用
   *
   * @param ids 主键列表
   */
  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.signTaskActivityRepository.updateEnableStatusByIds(EnableStatusEnum.ENABLE, ids);
  }

  /**
   * 批量禁用
   *
   * @param ids 主键列表
   */
  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.signTaskActivityRepository.updateEnableStatusByIds(EnableStatusEnum.DISABLE, ids);
  }

  /**
   * 公共字段填充 验证
   *
   * @param signTaskActivity
   * @author rentao
   * @date
   */
  private SignTaskActivity createForm(SignTaskActivity signTaskActivity) {
    Validate.notNull(signTaskActivity, "进行当前操作时，信息对象必须传入!!");
    this.createValidation(signTaskActivity);
    Date now = new Date();
    signTaskActivity.setCreateTime(now);
    signTaskActivity.setModifyTime(now);
    if (Objects.nonNull(generateCodeService)) {
      signTaskActivity.setSignTaskActivityCode(
          generateCodeService.generateCode(SignTaskCodeConstant.SIGN_TASK_ACTIVITY_CODE, 1).get(0));
    }
    signTaskActivity.setSignTaskStatus(SignTaskStatusEnum.STATUS_WAIT.getKey());
    signTaskActivity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    signTaskActivity.setTenantCode(TenantUtils.getTenantCode());
    signTaskActivity.setCreateAccount(loginUserService.getLoginAccountName());
    signTaskActivity.setModifyAccount(loginUserService.getLoginAccountName());
    return signTaskActivity;
  }

  /**
   * 新增验证
   *
   * @param signTaskActivity
   * @author rentao
   * @date
   */
  private void createValidation(SignTaskActivity signTaskActivity) {
    Validate.isTrue(StringUtils.isBlank(signTaskActivity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    signTaskActivity.setId(null);
    Validate.notBlank(signTaskActivity.getSignTaskCode(), "做数据操作时，签收活动配置编码不能为空！");
  }

  /**
   * 组装签收资源名称
   *
   * @param signTaskActivities
   * @author rentao
   * @date
   */
  private void setSignTaskActivitySignResourceName(List<SignTaskActivity> signTaskActivities) {
    if (!CollectionUtils.isEmpty(signTaskActivities)) {
      signTaskActivities.forEach(signTaskActivity -> {
        List<SignTaskActivityDetail> signTaskActivityDetails = signTaskActivity
            .getSignTaskActivityDetails();
        if (!CollectionUtils.isEmpty(signTaskActivityDetails)) {
          StringBuilder stb = new StringBuilder();
          List<String> signResourceNames = signTaskActivityDetails.stream().distinct()
              .map(SignTaskActivityDetail::getSignResourceName).collect(
                  Collectors.toList());
          signResourceNames.forEach(signResourceName -> {
            if (StringUtils.isEmpty(stb.toString())) {
              stb.append(signResourceName);
            } else {
              stb.append("," + signResourceName);
            }
          });
          signTaskActivity.setSignResourceName(stb.toString());
        }
      });
    }
  }
}
