package com.biz.crm.cps.external.capital.tax.raise.local.service.observer;

import com.biz.crm.cps.business.capital.sdk.dto.CapitalRechargeDto;
import com.biz.crm.cps.business.capital.sdk.dto.CapitalRechargeInfoDto;
import com.biz.crm.cps.business.capital.sdk.dto.RechargeConditionDto;
import com.biz.crm.cps.business.capital.sdk.service.observer.RechargeMountRegister;
import com.biz.crm.cps.business.capital.sdk.service.observer.RechargeServiceObserver;
import com.biz.crm.cps.business.capital.sdk.vo.CapitalRechargeVo;
import com.biz.crm.cps.business.capital.sdk.vo.RechargeProcessVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.external.capital.tax.raise.local.config.CapitalTaxRaiseProperties;
import com.biz.crm.cps.external.tax.raise.sdk.dto.base.MerchantAccountDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.capital.TaxRaiseRechargeDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.recharge.TaxRaiseRechargeAccountDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.recharge.TaxRaiseRechargeContractSignDto;
import com.biz.crm.cps.external.tax.raise.sdk.enums.recharge.TaxRaiseRechargeAccountTypeEnum;
import com.biz.crm.cps.external.tax.raise.sdk.enums.recharge.TaxRaiseRechargeCompanyVerifyStatusEnum;
import com.biz.crm.cps.external.tax.raise.sdk.enums.recharge.TaxRaiseRechargeContractSignStatus;
import com.biz.crm.cps.external.tax.raise.sdk.enums.recharge.TaxRaiseRechargePersonVerifyStatusEnum;
import com.biz.crm.cps.external.tax.raise.sdk.service.capital.TaxRaiseRechargeVoService;
import com.biz.crm.cps.external.tax.raise.sdk.service.recharge.TaxRaiseRechargeAccountVoService;
import com.biz.crm.cps.external.tax.raise.sdk.service.recharge.TaxRaiseRechargeContractSignVoService;
import com.biz.crm.cps.external.tax.raise.sdk.service.recharge.TaxRaiseRechargeContractTemplateVoService;
import com.biz.crm.cps.external.tax.raise.sdk.vo.recharge.TaxRaiseRechargeAccountVo;
import com.biz.crm.cps.external.tax.raise.sdk.vo.recharge.TaxRaiseRechargeContractSignVo;
import com.biz.crm.cps.external.tax.raise.sdk.vo.recharge.TaxRaiseRechargeContractTemplateVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

/**
 * 税筹充值 观察者实现
 *
 * @author ning.zhang
 * @date 2022/3/30
 */
@Service
public class TaxRechargeServiceObserverImpl implements RechargeServiceObserver {

  @Autowired
  @Qualifier("TaxRechargeMountRegisterImpl")
  private RechargeMountRegister rechargeMountRegister;
  @Autowired
  private TaxRaiseRechargeVoService taxRaiseRechargeVoService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private TaxRaiseRechargeAccountVoService taxRaiseRechargeAccountVoService;
  @Autowired
  private TaxRaiseRechargeContractSignVoService taxRaiseRechargeContractSignVoService;
  @Autowired
  private TaxRaiseRechargeContractTemplateVoService taxRaiseRechargeContractTemplateVoService;
  @Autowired
  private CapitalTaxRaiseProperties capitalTaxRaiseProperties;

  /**
   * 获取充值流程 是否需要认证 是否需要签署协议 1.判断是否该本模块处理该提现
   *
   * @param rechargeCondition 充值条件
   * @return 充值流程Vo
   */
  @Override
  public RechargeProcessVo onRequestRechargeProcess(RechargeConditionDto rechargeCondition) {
    if (Objects.isNull(rechargeCondition) || !Objects.equals(rechargeCondition.getRechargeKey(), this.rechargeMountRegister.getKey())) {
      return null;
    }
    RechargeProcessVo vo = new RechargeProcessVo();
    vo.setFlag(this.rechargeMountRegister.getFlag());
    vo.setKey(this.rechargeMountRegister.getKey());
    vo.setName(this.rechargeMountRegister.getName());
    // 2、判断是否进行实名认证
    MerchantAccountDto merchantAccountDto = new MerchantAccountDto();
    merchantAccountDto.setMerchantUserAccount(rechargeCondition.getMerchantUserAccount());
    TaxRaiseRechargeAccountVo accountVo = this.taxRaiseRechargeAccountVoService.findByMerchantAccountDto(merchantAccountDto);
    boolean needVerify = this.requiredAgainVerify(accountVo);
    vo.setNeedVerify(needVerify);
    if (!needVerify) {
      this.requiredContractSign(vo, rechargeCondition);
    } else {
      if (Objects.isNull(accountVo)) {
        accountVo = this.createVerifyUrl(rechargeCondition);
      }
      vo.setVerifyUrl(accountVo.getVerifyUrl());
      vo.setCertStatus(accountVo.getCertStatus());
      vo.setCompanyAuditFailReason(accountVo.getCompanyAuditFailReason());
      vo.setCompanyStatus(accountVo.getCompanyStatus());
      vo.setPersonAuditFailReason(accountVo.getPersonAuditFailReason());
      vo.setPersonStatus(accountVo.getPersonStatus());
      vo.setNeedSign(true);
    }
    return vo;
  }

  @Override
  @Transactional
  public CapitalRechargeVo onRecharge(CapitalRechargeInfoDto capitalRechargeInfoDto) {
    Validate.notNull(capitalRechargeInfoDto, "充值信息不能为空");
    Validate.notBlank(capitalRechargeInfoDto.getRechargeKey(), "充值类型不能为空");
    Validate.notEmpty(capitalRechargeInfoDto.getCapitalRechargeDtos(), "充值信息集合不能为空");
    if (Objects.equals(capitalRechargeInfoDto.getRechargeKey(), this.rechargeMountRegister.getKey())) {
      //cps新增充值同步到税筹
      List<TaxRaiseRechargeDto> taxRaiseRechargeDtos = (List<TaxRaiseRechargeDto>) this.nebulaToolkitService
          .copyCollectionByBlankList(capitalRechargeInfoDto.getCapitalRechargeDtos(), CapitalRechargeDto.class, TaxRaiseRechargeDto.class, HashSet.class, LinkedList.class);
      taxRaiseRechargeVoService.pushRechargeBatch(taxRaiseRechargeDtos);
    }
    return null;
  }

  /**
   * 是否需要认证
   *
   * @param accountVo 认证账号
   * @return 是否需要认证标识
   */
  private boolean requiredAgainVerify(TaxRaiseRechargeAccountVo accountVo) {
    if (Objects.isNull(accountVo)) {
      return true;
    }
    //如果实名认证已经通过代表不需要在进行认证
    return !((Objects.equals(accountVo.getAccountType(), TaxRaiseRechargeAccountTypeEnum.PERSON.getDictCode()) && Objects.equals(accountVo.getPersonStatus(), TaxRaiseRechargePersonVerifyStatusEnum.PASSED.getDictCode())) ||
        (Objects.equals(accountVo.getAccountType(), TaxRaiseRechargeAccountTypeEnum.COMPANY.getDictCode()) && Objects.equals(accountVo.getCompanyStatus(), TaxRaiseRechargeCompanyVerifyStatusEnum.PASSED.getDictCode())));
  }

  /**
   * 获取实名认证的url
   *
   * @param rechargeCondition 充值条件
   * @return 实名认证的url
   */
  private TaxRaiseRechargeAccountVo createVerifyUrl(RechargeConditionDto rechargeCondition) {
    TaxRaiseRechargeAccountDto dto = new TaxRaiseRechargeAccountDto();
    dto.setAccountType(TaxRaiseRechargeAccountTypeEnum.COMPANY.getDictCode());
    dto.setMerchantUserAccount(rechargeCondition.getMerchantUserAccount());
    if (ParticipatorTypeEnum.DEALER.getDictCode().equals(rechargeCondition.getParticipatorFlag())) {
      dto.setReturnUrl(capitalTaxRaiseProperties.getDealerRechargeReturnUrl());
    }
    this.taxRaiseRechargeAccountVoService.create(dto);
    return this.taxRaiseRechargeAccountVoService.findByMerchantAccountDto(dto);
  }

  /**
   * 获取手动签署地址
   *
   * @param rechargeCondition  充值条件
   * @param contractTemplateVo 合同模板
   * @return 手动签署地址
   */
  private String findContractSignUrl(RechargeConditionDto rechargeCondition, TaxRaiseRechargeContractTemplateVo contractTemplateVo) {
    TaxRaiseRechargeContractSignDto dto = new TaxRaiseRechargeContractSignDto();
    dto.setMerchantUserAccount(rechargeCondition.getMerchantUserAccount());
    dto.setTemplateCode(contractTemplateVo.getTemplateCode());
    if (ParticipatorTypeEnum.DEALER.getDictCode().equals(rechargeCondition.getParticipatorFlag())) {
      dto.setReturnUrl(capitalTaxRaiseProperties.getDealerRechargeReturnUrl());
    }
    TaxRaiseRechargeContractSignVo contractSignVo = this.taxRaiseRechargeContractSignVoService.create(dto);
    return contractSignVo.getSignUrl();
  }

  /**
   * 是否进行合同签署
   *
   * @param vo                流程vo
   * @param rechargeCondition 充值条件
   */
  private void requiredContractSign(RechargeProcessVo vo, RechargeConditionDto rechargeCondition) {
    //查询是否已签署合同
    MerchantAccountDto dto = new MerchantAccountDto();
    dto.setMerchantUserAccount(rechargeCondition.getMerchantUserAccount());
    TaxRaiseRechargeContractTemplateVo contractTemplateVo = this.taxRaiseRechargeContractTemplateVoService.findByMerchantAccountDto(dto);
    Validate.notNull(contractTemplateVo, "未查询到合同模板！");
    TaxRaiseRechargeContractSignDto signDto = new TaxRaiseRechargeContractSignDto();
    signDto.setMerchantUserAccount(rechargeCondition.getMerchantUserAccount());
    signDto.setTemplateCode(contractTemplateVo.getTemplateCode());
    TaxRaiseRechargeContractSignVo contractSignVo = this.taxRaiseRechargeContractSignVoService.findByTaxRaiseRechargeContractSignDto(signDto);
    vo.setNeedSign(false);
    if (Objects.nonNull(contractSignVo)) {
      if (!TaxRaiseRechargeContractSignStatus.SIGN.getDictCode().equals(contractSignVo.getSignStatus())) {
        vo.setContractSignUrl(contractSignVo.getSignUrl());
        vo.setNeedSign(true);
      } else {
        vo.setContractViewUrl(contractSignVo.getViewUrl());
      }
    } else {
      //获取手动签署合同地址
      String contractSignUrl = this.findContractSignUrl(rechargeCondition, contractTemplateVo);
      vo.setContractSignUrl(contractSignUrl);
      vo.setNeedSign(true);
    }
  }
}
