package com.biz.crm.cps.external.cash.tax.raise.local.service.observer;

import com.biz.crm.cps.business.capital.sdk.common.enums.DataSourceEnum;
import com.biz.crm.cps.business.cash.sdk.common.enums.CashPushStatusEnum;
import com.biz.crm.cps.business.cash.sdk.common.enums.CashStatusEnum;
import com.biz.crm.cps.business.cash.sdk.dto.CashConditionDto;
import com.biz.crm.cps.business.cash.sdk.dto.CashRecordDto;
import com.biz.crm.cps.business.cash.sdk.service.CashRecordVoService;
import com.biz.crm.cps.business.cash.sdk.service.observer.CashMountRegister;
import com.biz.crm.cps.business.cash.sdk.service.observer.CashServiceObserver;
import com.biz.crm.cps.business.cash.sdk.vo.CashProcessVo;
import com.biz.crm.cps.business.cash.sdk.vo.CashRecordVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.external.cash.tax.raise.local.config.CashTaxRaiseProperties;
import com.biz.crm.cps.external.tax.raise.sdk.dto.base.MerchantAccountDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.capital.TaxRaisePaymentDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.withdrawal.TaxRaiseWithdrawalAccountDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.withdrawal.TaxRaiseWithdrawalContractSignDto;
import com.biz.crm.cps.external.tax.raise.sdk.enums.withdrawal.TaxRaiseWithdrawalAccountTypeEnum;
import com.biz.crm.cps.external.tax.raise.sdk.enums.withdrawal.TaxRaiseWithdrawalCompanyVerifyStatusEnum;
import com.biz.crm.cps.external.tax.raise.sdk.enums.withdrawal.TaxRaiseWithdrawalContractSignStatus;
import com.biz.crm.cps.external.tax.raise.sdk.enums.withdrawal.TaxRaiseWithdrawalPersonVerifyStatusEnum;
import com.biz.crm.cps.external.tax.raise.sdk.service.capital.TaxRaisePaymentVoService;
import com.biz.crm.cps.external.tax.raise.sdk.service.withdrawal.TaxRaiseWithdrawalAccountVoService;
import com.biz.crm.cps.external.tax.raise.sdk.service.withdrawal.TaxRaiseWithdrawalContractSignVoService;
import com.biz.crm.cps.external.tax.raise.sdk.service.withdrawal.TaxRaiseWithdrawalContractTemplateVoService;
import com.biz.crm.cps.external.tax.raise.sdk.vo.withdrawal.TaxRaiseWithdrawalAccountVo;
import com.biz.crm.cps.external.tax.raise.sdk.vo.withdrawal.TaxRaiseWithdrawalContractSignVo;
import com.biz.crm.cps.external.tax.raise.sdk.vo.withdrawal.TaxRaiseWithdrawalContractTemplateVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

/**
 * 税筹兑现 观察者实现
 *
 * @author ning.zhang
 * @date 2022/3/28
 */
@Service
public class TaxRaiseCashServiceObserverImpl implements CashServiceObserver {

  @Autowired
  @Qualifier("TaxRaiseCashMountRegisterImpl")
  private CashMountRegister cashMountRegister;
  @Autowired
  private TaxRaiseWithdrawalAccountVoService taxRaiseWithdrawalAccountVoService;
  @Autowired(required = false)
  private CashRecordVoService cashRecordVoService;
  @Autowired
  private TaxRaiseWithdrawalContractTemplateVoService taxRaiseWithdrawalContractTemplateVoService;
  @Autowired
  private TaxRaiseWithdrawalContractSignVoService taxRaiseWithdrawalContractSignVoService;
  @Autowired
  private TaxRaisePaymentVoService taxRaisePaymentVoService;
  @Autowired
  private CashTaxRaiseProperties cashTaxRaiseProperties;

  /**
   * 获取提现流程 是否需要认证 是否需要签署协议 1.判断是否该本模块处理该提现
   *
   * @param cashCondition 提现条件
   * @return 提现流程Vo
   */
  @Override
  public CashProcessVo onRequestCashProcess(CashConditionDto cashCondition) {
    // 1.判断是否该本模块处理该提现
    if (Objects.isNull(cashCondition) || !Objects.equals(cashCondition.getCashKey(), this.cashMountRegister.getKey())) {
      return null;
    }
    CashProcessVo vo = new CashProcessVo();
    vo.setFlag(this.cashMountRegister.getFlag());
    vo.setKey(this.cashMountRegister.getKey());
    vo.setName(this.cashMountRegister.getName());
    // 2、判断是否进行实名认证
    MerchantAccountDto merchantAccountDto = new MerchantAccountDto();
    merchantAccountDto.setMerchantUserAccount(cashCondition.getPayeeCode());
    TaxRaiseWithdrawalAccountVo accountVo = this.taxRaiseWithdrawalAccountVoService.findByMerchantAccountDto(merchantAccountDto);
    boolean needVerify = this.requiredAgainVerify(accountVo);
    vo.setNeedVerify(needVerify);
    if (!needVerify) {
      this.requiredContractSign(vo, cashCondition);
    } else {
      if (Objects.isNull(accountVo)) {
        accountVo = this.createVerifyUrl(cashCondition);
      }
      vo.setVerifyUrl(accountVo.getVerifyUrl());
      vo.setCertStatus(accountVo.getCertStatus());
      vo.setCompanyAuditFailReason(accountVo.getCompanyAuditFailReason());
      vo.setPersonAuditFailReason(accountVo.getPersonAuditFailReason());
      vo.setCompanyStatus(accountVo.getCompanyStatus());
      vo.setPersonStatus(accountVo.getPersonStatus());
      vo.setNeedSign(true);
    }
    return vo;
  }

  /**
   * 暂时用不到
   *
   * @param cashCondition 提现条件
   * @return 提现流程Vo
   */
  @Override
  @Transactional
  public CashProcessVo onCreateContract(CashConditionDto cashCondition) {
    //TODO创建合同
    return null;
  }

  /**
   * 生成提现流水
   *
   * @param cashCondition 提现条件
   * @return 提现流水
   */
  @Override
  @Transactional
  public CashRecordVo onCash(CashConditionDto cashCondition) {
    Validate.notNull(cashCondition, "兑付条件不能为空");
    Validate.notNull(cashCondition.getAmount(), "兑付金额不能为空");
    Validate.isTrue(cashCondition.getAmount().compareTo(BigDecimal.ZERO) > 0, "兑付金额必须大于0");
    if (Objects.equals(cashCondition.getCashKey(), this.cashMountRegister.getKey())) {
      // 1 生成支付流水
      CashRecordDto cashRecordDto = new CashRecordDto();
      cashRecordDto.setAmount(cashCondition.getAmount());
      cashRecordDto.setParticipatorCode(cashCondition.getParticipatorCode());
      cashRecordDto.setParticipatorType(cashCondition.getParticipatorFlag());
      cashRecordDto.setParticipatorName(cashCondition.getParticipatorName());
      cashRecordDto.setCashMethodKey(cashMountRegister.getKey());
      cashRecordDto.setCashMethodName(cashMountRegister.getName());
      cashRecordDto.setCashStatus(CashStatusEnum.NO_PAID.getDictCode());
      cashRecordDto.setRewardKey(cashCondition.getRewardKey());
      cashRecordDto.setRewardName(cashCondition.getRewardName());
      cashRecordDto.setPayeePhone(cashCondition.getPayeePhone());
      cashRecordDto.setPayeeCode(cashCondition.getPayeeCode());
      cashRecordDto.setPayeeName(cashCondition.getPayeeName());
      //创建后立刻推送到税筹
      cashRecordDto.setPushStatus(CashPushStatusEnum.PUSH_SUCCESS.getDictCode());
      //TODO
      //cashRecordDto.setPayeeBank();
      cashRecordDto.setPushStatus(CashPushStatusEnum.PUSH_SUCCESS.getDictCode());
      CashRecordVo cashRecordVo = this.cashRecordVoService.create(cashRecordDto);
      this.pushPaymentBatch(cashRecordVo);
      return cashRecordVo;
    }
    return null;
  }

  /**
   * 是否需要认证
   *
   * @param accountVo 认证账号
   * @return 是否需要认证标识
   */
  private boolean requiredAgainVerify(TaxRaiseWithdrawalAccountVo accountVo) {
    if (Objects.isNull(accountVo)) {
      return true;
    }
    //如果实名认证已经通过代表不需要在进行认证
    return !((Objects.equals(accountVo.getAccountType(), TaxRaiseWithdrawalAccountTypeEnum.PERSON.getDictCode()) && Objects.equals(accountVo.getPersonStatus(), TaxRaiseWithdrawalPersonVerifyStatusEnum.PASSED.getDictCode())) ||
        (Objects.equals(accountVo.getAccountType(), TaxRaiseWithdrawalAccountTypeEnum.COMPANY.getDictCode()) && Objects.equals(accountVo.getCompanyStatus(), TaxRaiseWithdrawalCompanyVerifyStatusEnum.PASSED.getDictCode())));
  }

  /**
   * 获取实名认证的url
   *
   * @param cashCondition 提现条件
   * @return 实名认证的url
   */
  private TaxRaiseWithdrawalAccountVo createVerifyUrl(CashConditionDto cashCondition) {
    TaxRaiseWithdrawalAccountDto dto = new TaxRaiseWithdrawalAccountDto();
    dto.setAccountType(TaxRaiseWithdrawalAccountTypeEnum.PERSON.getDictCode());
    dto.setMerchantUserAccount(cashCondition.getPayeeCode());
    if (ParticipatorTypeEnum.DEALER.getDictCode().equals(cashCondition.getParticipatorFlag())) {
      dto.setReturnUrl(cashTaxRaiseProperties.getDealerWithdrawalReturnUrl());
    }
    if (ParticipatorTypeEnum.TERMINAL.getDictCode().equals(cashCondition.getParticipatorFlag())) {
      dto.setReturnUrl(cashTaxRaiseProperties.getTerminalWithdrawalReturnUrl());
    }
    this.taxRaiseWithdrawalAccountVoService.create(dto);
    return this.taxRaiseWithdrawalAccountVoService.findByMerchantAccountDto(dto);
  }

  /**
   * 获取手动签署地址
   *
   * @param cashCondition      提现条件
   * @param contractTemplateVo 合同模板
   * @return 手动签署地址
   */
  private String findContractSignUrl(CashConditionDto cashCondition, TaxRaiseWithdrawalContractTemplateVo contractTemplateVo) {
    TaxRaiseWithdrawalContractSignDto dto = new TaxRaiseWithdrawalContractSignDto();
    dto.setMerchantUserAccount(cashCondition.getPayeeCode());
    dto.setTemplateCode(contractTemplateVo.getTemplateCode());
    if (ParticipatorTypeEnum.DEALER.getDictCode().equals(cashCondition.getParticipatorFlag())) {
      dto.setReturnUrl(cashTaxRaiseProperties.getDealerWithdrawalReturnUrl());
    }
    if (ParticipatorTypeEnum.TERMINAL.getDictCode().equals(cashCondition.getParticipatorFlag())) {
      dto.setReturnUrl(cashTaxRaiseProperties.getTerminalWithdrawalReturnUrl());
    }
    TaxRaiseWithdrawalContractSignVo contractSignVo = this.taxRaiseWithdrawalContractSignVoService.create(dto);
    return contractSignVo.getSignUrl();
  }

  /**
   * 是否进行合同签署
   *
   * @param vo            流程vo
   * @param cashCondition 提现条件
   */
  private void requiredContractSign(CashProcessVo vo, CashConditionDto cashCondition) {
    //查询是否已签署合同
    MerchantAccountDto dto = new MerchantAccountDto();
    dto.setMerchantUserAccount(cashCondition.getPayeeCode());
    TaxRaiseWithdrawalContractTemplateVo contractTemplateVo = this.taxRaiseWithdrawalContractTemplateVoService.findByMerchantAccountDto(dto);
    Validate.notNull(contractTemplateVo, "未查询到合同模板！");
    TaxRaiseWithdrawalContractSignDto signDto = new TaxRaiseWithdrawalContractSignDto();
    signDto.setMerchantUserAccount(cashCondition.getPayeeCode());
    signDto.setTemplateCode(contractTemplateVo.getTemplateCode());
    TaxRaiseWithdrawalContractSignVo contractSignVo = this.taxRaiseWithdrawalContractSignVoService.findByTaxRaiseWithdrawalContractSignDto(signDto);
    vo.setNeedSign(false);
    if (Objects.nonNull(contractSignVo)) {
      if (!TaxRaiseWithdrawalContractSignStatus.SIGN.getDictCode().equals(contractSignVo.getSignStatus())) {
        vo.setContractSignUrl(contractSignVo.getSignUrl());
        vo.setNeedSign(true);
      } else {
        vo.setContractViewUrl(contractSignVo.getViewUrl());
      }
    } else {
      //获取手动签署合同地址
      String contractSignUrl = this.findContractSignUrl(cashCondition, contractTemplateVo);
      vo.setContractSignUrl(contractSignUrl);
      vo.setNeedSign(true);
    }
  }

  /**
   * 推送数据到税筹平台
   * @param cashRecordVo
   */
  private void pushPaymentBatch (CashRecordVo cashRecordVo){
    TaxRaisePaymentDto taxRaisePaymentDto = new TaxRaisePaymentDto();
    taxRaisePaymentDto.setCashCode(cashRecordVo.getCashCode());
    taxRaisePaymentDto.setRemark(cashRecordVo.getRemark());
    taxRaisePaymentDto.setCustomerCode(cashRecordVo.getPayeeCode());
    taxRaisePaymentDto.setCustomerName(cashRecordVo.getPayeeName());
    taxRaisePaymentDto.setPaymentAmount(cashRecordVo.getAmount());
    taxRaisePaymentDto.setAccountCode(cashRecordVo.getPayeeCode());
    taxRaisePaymentDto.setPaymentTime(cashRecordVo.getPayDate());
    taxRaisePaymentDto.setDataSource(DataSourceEnum.CPS_PUSH.getDictCode());
    List<TaxRaisePaymentDto> taxRaisePaymentDtos = Lists.newArrayList();
    taxRaisePaymentDtos.add(taxRaisePaymentDto);
    taxRaisePaymentVoService.pushPaymentBatch(taxRaisePaymentDtos);
  }
}
