package com.biz.crm.cps.external.cash.weixinpay.local.service.observer;

import com.alibaba.fastjson.JSON;
import com.biz.crm.cps.business.cash.sdk.common.enums.CashStatusEnum;
import com.biz.crm.cps.business.cash.sdk.dto.CashConditionDto;
import com.biz.crm.cps.business.cash.sdk.dto.CashConfigurationDto;
import com.biz.crm.cps.business.cash.sdk.dto.CashRecordDto;
import com.biz.crm.cps.business.cash.sdk.service.CashConfigurationVoService;
import com.biz.crm.cps.business.cash.sdk.service.CashRecordVoService;
import com.biz.crm.cps.business.cash.sdk.service.observer.CashMountRegister;
import com.biz.crm.cps.business.cash.sdk.service.observer.CashServiceObserver;
import com.biz.crm.cps.business.cash.sdk.vo.CashProcessVo;
import com.biz.crm.cps.business.cash.sdk.vo.CashRecordVo;
import com.biz.crm.cps.external.cash.weixinpay.local.config.WXPayProperties;
import com.biz.crm.cps.external.cash.weixinpay.sdk.utils.WXPay;
import com.biz.crm.cps.external.cash.weixinpay.sdk.utils.WXPayConfig;
import com.biz.crm.cps.external.cash.weixinpay.sdk.utils.WXPayConstants;
import com.biz.crm.cps.external.cash.weixinpay.sdk.utils.WXPayUtil;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;

/**
 * @author hecheng
 * @description: 微信兑付 观察者实现
 * @date 2021/8/27 下午10:53
 */
@Service("WinXinPayServiceObserverImpl")
@Slf4j
public class WinXinPayServiceObserverImpl implements CashServiceObserver {

  @Autowired
  @Qualifier("WeiXinPayCashMountRegisterImpl")
  private CashMountRegister cashMountRegister;
  @Autowired(required = false)
  private CashRecordVoService cashRecordVoService;
  @Autowired(required = false)
  private CashConfigurationVoService cashConfigurationVoService;
  @Autowired(required = false)
  private WXPayConfig wxPayConfig;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private WXPayProperties wxPayProperties;
  @Override
  @Transactional
  public CashProcessVo onRequestCashProcess(CashConditionDto cashCondition) {
    // 1.判断是否该本模块处理该提现
    if (Objects.isNull(cashCondition) || !Objects.equals(cashCondition.getCashKey(), this.cashMountRegister.getKey())) {
      return null;
    }
    CashConfigurationDto config = new CashConfigurationDto();
    config.setCashMethodKey(this.cashMountRegister.getKey());
    config.setCashMethodName(this.cashMountRegister.getName());
    config.setSignatoryCode(cashCondition.getPayeeCode());
    config.setSignatoryName(cashCondition.getPayeeName());
    config.setSignatoryType(cashCondition.getParticipatorFlag());
    this.cashConfigurationVoService.create(config);

    CashProcessVo vo = new CashProcessVo();
    vo.setFlag(this.cashMountRegister.getFlag());
    vo.setKey(this.cashMountRegister.getKey());
    vo.setName(this.cashMountRegister.getName());
    vo.setNeedVerify(cashMountRegister.needSignVerify());
    vo.setNeedSign(cashMountRegister.needSign());
    return vo;
  }

  @Override
  @Transactional
  public CashProcessVo onCreateContract(CashConditionDto cashCondition) {
    return null;
  }

  @Override
  @Transactional
  public CashRecordVo onCash(CashConditionDto cashCondition) {
    Validate.notNull(cashCondition, "兑付条件不能为空");
    Validate.notNull(cashCondition.getAmount(), "兑付金额不能为空");
    Validate.isTrue(cashCondition.getAmount().compareTo(BigDecimal.ZERO) > 0, "兑付金额必须大于0");
    if (!Objects.equals(cashCondition.getCashKey(), this.cashMountRegister.getKey())) {
      return null;
    }
    // 1 生成支付流水
    CashRecordDto cashRecordDto = new CashRecordDto();
    cashRecordDto.setAmount(cashCondition.getAmount());
    cashRecordDto.setParticipatorCode(cashCondition.getParticipatorCode());
    cashRecordDto.setParticipatorType(cashCondition.getParticipatorFlag());
    cashRecordDto.setParticipatorName(cashCondition.getParticipatorName());
    cashRecordDto.setCashMethodKey(cashMountRegister.getKey());
    cashRecordDto.setCashMethodName(cashMountRegister.getName());
    cashRecordDto.setCashStatus(CashStatusEnum.PAIDING.getDictCode());
    cashRecordDto.setRewardKey(cashCondition.getRewardKey());
    cashRecordDto.setRewardName(cashCondition.getRewardName());
    cashRecordDto.setPayeeName(cashCondition.getPayeeName());
    cashRecordDto.setPayeePhone(cashCondition.getPayeePhone());
    cashRecordDto.setPayeeCode(cashCondition.getPayeeCode());
    Validate.notBlank(cashCondition.getOpenId(), "兑付时，openid不能为空");
    cashRecordDto.setPayeeBank(cashCondition.getOpenId());
    CashRecordVo cashRecordVo = this.cashRecordVoService.create(cashRecordDto);
    // 2。调用微信支付
    this.pay(cashRecordVo);
    //更新支付状态
    CashRecordDto cashRecordStatus = this.nebulaToolkitService.copyObjectByWhiteList(cashRecordVo, CashRecordDto.class, HashSet.class, ArrayList.class);
    this.cashRecordVoService.updateCashStatus(cashRecordStatus);
    return cashRecordVo;
  }

  /**
   * 调用微信支付
   *
   * @param cashRecordVo
   */
  private void pay(CashRecordVo cashRecordVo) {
    if(this.wxPayProperties.isSkipPay()){
      log.info("=========已开启跳过支付配置=============");
      cashRecordVo.setCashStatus(CashStatusEnum.PAID.getDictCode());
      cashRecordVo.setRemark("支付成功!");
      cashRecordVo.setReceiveDate(new Date());
      return;
    }
    WXPay wxpay = null;
    try {
      wxpay = new WXPay(this.wxPayConfig);
    } catch (Exception e) {
      log.error("微信支付初始化失败！{}", e, e.getMessage());
      cashRecordVo.setCashStatus(CashStatusEnum.PAID_FAIL.getDictCode());
      cashRecordVo.setRemark("微信支付初始化失败!");
      Validate.isTrue(false,"红包提现初始化失败!");
      return;
    }
    // 获取随机字符串
    String nonceStr = WXPayUtil.generateNonceStr();
    int amountFen = cashRecordVo.getAmount().multiply(new BigDecimal(100)).intValue();
    String openid = cashRecordVo.getPayeeBank();
    String cashCode = cashRecordVo.getCashCode();
    Map<String, String> params = new HashMap<>();
    params.put("mch_appid", this.wxPayConfig.getAppID());
    //商户号
    params.put("mchid", this.wxPayConfig.getMchID());
    params.put("nonce_str", nonceStr);
    //商户订单号
    params.put("partner_trade_no", cashCode);
    //用户openid
    params.put("openid", openid);
    //校验用户姓名选项
    params.put("check_name", "NO_CHECK");
    //金额 分
    params.put("amount", String.valueOf(amountFen));
    //付款备注
    params.put("desc", "红包提现");
    //Ip地址
    //params.put("spbill_create_ip","");
    // 签名需要其它数据信息，最后设置
    String sign = null;
    try {
      sign = WXPayUtil.generateSignature(params, this.wxPayConfig.getKey());
      // 签名
      params.put("sign", sign);
    } catch (Exception e) {
      log.error("微信支付sign失败！{}" + e.getMessage(), e);
      cashRecordVo.setCashStatus(CashStatusEnum.PAID_FAIL.getDictCode());
      cashRecordVo.setRemark("微信支付签名失败!");
      Validate.isTrue(false,"红包提现签名失败!");

      return;
    }
    //
    Map<String, String> result = null;
    try {
      log.info("微信支付参数=:" + JSON.toJSONString(params));
      String resp = wxpay.requestWithCert(WXPayConstants.MMPAYMKTTRANSFERS_URL_SUFFIX, params, this.wxPayConfig.getHttpConnectTimeoutMs(), this.wxPayConfig.getHttpReadTimeoutMs());
      result = WXPayUtil.xmlToMap(resp);
      log.info("微信支付结果=:" + JSON.toJSONString(result));
      //返回状态码此字段是通信标识，非交易标识，交易是否成功需要查看result_code来判断
      String returnCode = result.get("return_code");
      String returnMsg = result.get("return_msg");
      //业务结果
      String resultCode = result.get("result_code");
      String errCodeDes = result.get("err_code_des");
      //成功
      if (Objects.equals(returnCode, WXPayConstants.SUCCESS) && Objects.equals(resultCode, WXPayConstants.SUCCESS)) {
        cashRecordVo.setCashStatus(CashStatusEnum.PAID.getDictCode());
        cashRecordVo.setRemark("支付成功!");
        cashRecordVo.setReceiveDate(new Date());
      } else {
        //失败
        String errCode = result.get("err_code");
        cashRecordVo.setCashStatus(CashStatusEnum.PAID_FAIL.getDictCode());
        String remark = Objects.equals(returnMsg, errCodeDes) ? errCodeDes : returnMsg + errCodeDes;
        cashRecordVo.setRemark(remark);
        //不需要查询以查单结果为准/如果要继续付款必须使用原商户订单号重试的情况
        if(Objects.equals(errCode,"NO_AUTH")||
                Objects.equals(errCode,"AMOUNT_LIMIT")||
                Objects.equals(errCode,"PARAM_ERROR")||
                Objects.equals(errCode,"OPENID_ERROR")||
                Objects.equals(errCode,"XML_ERROR")||
                Objects.equals(errCode,"RECV_ACCOUNT_NOT_ALLOWED")||
                Objects.equals(errCode,"PAY_CHANNEL_NOT_ALLOWED")){
          Validate.isTrue(false,"红包提现失败，请联系业务员!"+remark);
        }
      }
    } catch (Exception e) {
      log.error("微信支付失败！" + e.getMessage(), e);
      cashRecordVo.setCashStatus(CashStatusEnum.PAID_FAIL.getDictCode());
      cashRecordVo.setRemark("微信支付失败!");
      Validate.isTrue(false,"红包提现失败，请联系业务员!");
    }
  }
}
