package com.biz.crm.cps.external.tax.raise.local.client.base;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.cps.business.common.local.model.Result;
import com.biz.crm.cps.external.tax.raise.local.config.TaxRaiseProperties;
import com.biz.crm.cps.external.tax.raise.sdk.dto.base.BaseTaxRaiseDto;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.net.URI;
import java.net.URLEncoder;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;

/**
 * 税筹对接抽象client
 *
 * @author ning.zhang
 * @date 2022/3/18
 */
@Slf4j
public abstract class AbstractTaxRaiseClient {

  @Autowired
  protected TaxRaiseProperties properties;
  @Autowired
  private RestTemplate restTemplate;

  /**
   * 执行http get请求
   *
   * @param paramObject 参数对象
   * @param pageable    分页参数
   * @param <T>         参数泛型
   * @return 返回结果
   */
  protected <T extends BaseTaxRaiseDto> Result doGet(String url, T paramObject, Pageable pageable) {
    Result result;
    try {
      UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(String.format("%s/%s", properties.getHost(), url));
      JSONObject paramJson = new JSONObject();
      if (Objects.nonNull(paramObject)) {
        paramJson.putAll(JSON.parseObject(JSON.toJSONStringWithDateFormat(paramObject, JSON.DEFFAULT_DATE_FORMAT)));
      }
      if (Objects.nonNull(pageable)) {
        paramJson.putAll(JSON.parseObject(JSON.toJSONString(pageable)));
      }
      paramJson.put("merchantId", properties.getMerchantId());
      if (!paramJson.isEmpty()) {
        for (Map.Entry<String, Object> entry : paramJson.entrySet()) {
          if (Objects.nonNull(entry.getValue()) && entry.getValue() instanceof Collection) {
            builder.queryParam(entry.getKey(), String.join(",", (Collection) entry.getValue()));
          } else {
            builder.queryParam(entry.getKey(), URLEncoder.encode(entry.getValue().toString(), "UTF-8"));
          }
        }
      }
      UriComponents components = builder.build(true);
      URI uri = components.toUri();
      result = restTemplate.getForObject(uri, Result.class);
      Validate.notNull(result, "结果异常");
    } catch (Exception e) {
      log.error("税筹平台请求错误", e);
      throw new IllegalArgumentException("系统繁忙,请稍后重试");
    }
    Validate.isTrue(result.isSuccess(), result.getMessage());
    return result;
  }

  /**
   * 执行http post请求
   *
   * @param paramBody 请求Body对象
   * @param <T>       参数泛型
   * @return 返回结果
   */
  protected <T extends BaseTaxRaiseDto> Result doPost(String url, T paramBody) {
    Result result;
    try {
      UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(String.format("%s/%s", properties.getHost(), url));
      UriComponents components = builder.build(true);
      URI uri = components.toUri();
      JSONObject bodyJson = new JSONObject();
      if (Objects.nonNull(paramBody)) {
        bodyJson.putAll(JSON.parseObject(JSON.toJSONStringWithDateFormat(paramBody, JSON.DEFFAULT_DATE_FORMAT)));
      }
      bodyJson.put("merchantId", properties.getMerchantId());
      result = restTemplate.postForEntity(uri, bodyJson, Result.class).getBody();
      Validate.notNull(result, "结果异常");
    } catch (Exception e) {
      log.error("税筹平台请求错误", e);
      throw new IllegalArgumentException("系统繁忙,请稍后重试");
    }
    Validate.isTrue(result.isSuccess(), result.getMessage());
    return result;
  }

}
