package com.biz.crm.cps.external.weixinsign.local.controller;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.cps.external.weixinsign.sdk.service.WXOpenVoService;
import com.biz.crm.cps.external.weixinsign.sdk.vo.WxUserInfoVo;
import com.biz.crm.cps.external.weixinsign.sdk.vo.WxWebViewAccessTokenRespVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;

/**
 * 微信网页授权接口
 * <pre>
 *   方式1：https://domain/cps-mobile/v1/wx/wxSso/authorize?redirectUri=https://domain/cps-mobile/v1/wx/wxSso/findOpenId?redirectUri=https://www.baidu.com
 *   第一个redirectUri为授权后获取openId的操作，第二个redirectUri为整个授权成功后重定向的地址，例如成功后回百度
 *   最终回到第二个redirectUri时 https://www.baidu.com?openid=123456
 *
 *   方式2：https://domain/cps-mobile/v1/wx/wxSso/authorize?redirectUri=https://domain/xxxx?code=1234
 *   redirectUri为授权后获取网页授权code的页面，可以在此页面调用/v1/wx/wxOpen/findWebViewAccessToken?code=1234,获取openid
 * </pre>
 *
 * @author hecheng
 */
@Slf4j
@Controller
@RequestMapping("/v1/wx/wxSso")
@Api(tags = "微信网页授权接口")
public class  WXOpenSsoController extends WxBaseController {

  @Autowired(required = false)
  private WXOpenVoService wxOpenService;


  /**
   * 获取微信网页授权code
   *
   * @param redirectUri
   * @param appType
   * @return
   */
  @ApiOperation(value = "获取微信网页授权code")
  @GetMapping("/authorize")
  public ModelAndView authorize(@ApiParam("重定向url") @RequestParam String redirectUri, @RequestParam(defaultValue = "snsapi_base") String scope,
          @ApiParam("应用类型") @RequestParam(required = false) String appType, HttpServletRequest request) {
    if (!this.isWeChat(request)) {
      ModelAndView mv = new ModelAndView("redirect:" + redirectUri);
      return mv;
    }
    String authorizeUrl = wxOpenService.getAuthorizeUrl(redirectUri, scope, appType);
    ModelAndView mv = new ModelAndView("redirect:" + authorizeUrl);
    return mv;
  }

  @ApiOperation(value = "获取openId")
  @GetMapping("/findOpenId")
  public ModelAndView findOpenId(@ApiParam("重定向url") @RequestParam String redirectUri, @RequestParam(required = false) String code,
          @ApiParam("应用类型") @RequestParam(required = false) String appType, HttpServletRequest request) {
    if (this.isWeChat(request)) {
      Validate.notBlank(code, "微信环境下code不能为空");
    }
    WxWebViewAccessTokenRespVo accessTokenRespVo = wxOpenService.findWebViewAccessToken(code, appType);
    String openId = null;
    if (accessTokenRespVo != null) {
      openId = accessTokenRespVo.getOpenid();
    }
    ModelAndView mv = new ModelAndView("redirect:" + redirectUri + "?openid=" + openId);
    return mv;
  }


  @ApiOperation(value = "获取微信用户信息")
  @GetMapping("/findWxUserInfo")
  public ModelAndView findWxUserInfo(@ApiParam("重定向url") @RequestParam String redirectUri, @RequestParam(required = false) String code,
                                     @ApiParam("应用类型") @RequestParam(required = false) String appType, HttpServletRequest request) {
    if (this.isWeChat(request)) {
      Validate.notBlank(code, "微信环境下code不能为空");
    }
    WxWebViewAccessTokenRespVo accessTokenRespVo = wxOpenService.findWebViewAccessToken(code, appType);
    String wxUserInfoStr = null;
    if (accessTokenRespVo != null) {
      WxUserInfoVo wxUserInfo = this.wxOpenService.findWxUserInfo(accessTokenRespVo.getOpenid(), accessTokenRespVo.getAccessToken(), null);
      if (wxUserInfo != null) {
        wxUserInfoStr = JSONObject.toJSONString(wxUserInfo);
      }
    }
    ModelAndView mv = new ModelAndView("redirect:" + redirectUri + "?wxUserInfo=" + wxUserInfoStr);
    return mv;
  }

}
