package com.biz.crm.business.common.sdk.utils;

import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * 树形结构降维编码工具
 */
public class TreeRuleCode {

  /**
   * 降维编码单层长度
   */
  private int ruleCodeSplitLength = 3;

  //构造方法
  public TreeRuleCode(int ruleCodeSplitLength) {
    this.ruleCodeSplitLength = ruleCodeSplitLength;
  }
  
  /**
   * 数字转换成一层降维编码 如（3->003）
   *
   * @param num
   * @return
   */
  public String generateByNum(Integer num) {
    int length = getRuleCodeSplitLength();
    return String.format("%0" + length + "d", num);
  }

  /**
   * 根据降维编码分离出上级降维编码集合
   *
   * @param ruleCode
   * @return
   */
  public Set<String> findParentRuleCodesByRuleCode(String ruleCode) {
    int length = getRuleCodeSplitLength();
    Set<String> set = new LinkedHashSet<>();
    if (StringUtils.isNotEmpty(ruleCode)) {
      set.add(ruleCode);
      while (ruleCode.length() > length) {
        ruleCode = ruleCode.substring(0, ruleCode.length() - length);
        set.add(ruleCode);
      }
    }
    return set;
  }

  /**
   * 根据降维编码分离出上级降维编码集合
   *
   * @param ruleCodeList
   * @return
   */
  public Set<String> findParentRuleCodesByRuleCodes(List<String> ruleCodeList) {
    Set<String> set = new LinkedHashSet<>();
    for (String ruleCode : ruleCodeList) {
      set.addAll(this.findParentRuleCodesByRuleCode(ruleCode));
    }
    return set;
  }

  /**
   * 根据降维编码分离出上级降维编码集合(不包含自己)
   *
   * @param ruleCode
   * @return
   */
  public Set<String> findParentRuleCodesByRuleCodeExcludeSelf(String ruleCode) {
    int length = getRuleCodeSplitLength();
    Set<String> set = new LinkedHashSet<>();
    if (StringUtils.isNotEmpty(ruleCode)) {
      while (ruleCode.length() > length) {
        ruleCode = ruleCode.substring(0, ruleCode.length() - length);
        set.add(ruleCode);
      }
    }
    return set;
  }

  /**
   * 根据降维编码分离出上级降维编码集合(不包含自己，如果本身编码集合已经有包含关系也会返回)
   *
   * @param ruleCodeList
   * @return
   */
  public Set<String> findParentRuleCodesByRuleCodesExcludeSelf(List<String> ruleCodeList) {
    Set<String> set = new LinkedHashSet<>();
    if (CollectionUtils.isNotEmpty(ruleCodeList)) {
      for (String ruleCode : ruleCodeList) {
        set.addAll(this.findParentRuleCodesByRuleCodeExcludeSelf(ruleCode));
      }
    }
    return set;
  }

  /**
   * 根据降维编码分离出上级降维编码集合(返回值不包含任何一个参数里的编码)
   *
   * @param ruleCodeList
   * @return
   */
  public Set<String> findParentRuleCodesByRuleCodesExcludeAnySelf(List<String> ruleCodeList) {
    Set<String> set = new LinkedHashSet<>();
    if (CollectionUtils.isNotEmpty(ruleCodeList)) {
      final Set<String> collect = new HashSet<>(ruleCodeList);
      for (String ruleCode : collect) {
        set.addAll(this.findParentRuleCodesByRuleCodeExcludeSelf(ruleCode));
      }
      return set.stream().filter(item -> !collect.contains(item)).collect(Collectors.toSet());
    }
    return set;
  }

  /**
   * 分离出当前降维最后一层对应的数（即降维编码在当前层级的序号）
   *
   * @param ruleCode
   * @return
   */
  public int findLevelNumByRuleCode(String ruleCode) {
    int length = getRuleCodeSplitLength();
    if (StringUtils.isNotBlank(ruleCode) && ruleCode.length() >= length) {
      try {
        if (ruleCode.length() % length != 0) {
          throw new RuntimeException();
        }
        return Integer.valueOf(ruleCode.substring(ruleCode.length() - length));
      } catch (Exception e) {
        throw new RuntimeException("存在不规范的降维编码,或者编码规则已变更，请先重置降维编码或者联系管理员处理");
      }
    }
    return 0;
  }

  /**
   * 获取降维编码单层长度
   *
   * @return
   */
  public int getRuleCodeSplitLength() {
    return this.ruleCodeSplitLength;
  }

  /**
   * 获取降维编码单层长度
   *
   * @return
   */
  public int getMaxRuleCode() {
    int ruleCodeSplitLength = getRuleCodeSplitLength();
    int max = 1;
    for (int i = 1; i <= ruleCodeSplitLength; i++) {
      max *= 10;
    }
    return (max - 1);
  }
}
