package com.biz.crm.business.common.local.interceptor;
/**
 * Created by Bao Hongbin on 2021-12-06 09:52.
 */

import com.alibaba.druid.sql.SQLUtils;
import com.alibaba.druid.sql.ast.SQLExpr;
import com.alibaba.druid.sql.ast.SQLStatement;
import com.alibaba.druid.sql.ast.expr.SQLCharExpr;
import com.alibaba.druid.sql.ast.expr.SQLIdentifierExpr;
import com.alibaba.druid.sql.ast.expr.SQLMethodInvokeExpr;
import com.alibaba.druid.sql.ast.expr.SQLNullExpr;
import com.alibaba.druid.sql.ast.statement.SQLInsertStatement;
import com.alibaba.druid.sql.ast.statement.SQLSelect;
import com.alibaba.druid.sql.ast.statement.SQLUpdateSetItem;
import com.alibaba.druid.sql.ast.statement.SQLUpdateStatement;
import com.alibaba.druid.sql.dialect.oracle.ast.expr.OracleDatetimeExpr;
import com.alibaba.druid.util.JdbcConstants;

import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.baomidou.mybatisplus.core.metadata.TableInfoHelper;
import com.biz.crm.business.common.local.entity.UuidOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.bizunited.nebula.common.repository.interceptor.NebulaRepositoryInterceptorStrategy;
import com.bizunited.nebula.common.repository.interceptor.SqlCommandType;


import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.orm.jpa.JpaProperties;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.stereotype.Component;

import java.sql.Connection;
import java.sql.SQLSyntaxErrorException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Objects;

/**
 * @program: crm
 * @description: mybatis基础字段设置增强拦截器，@Order注解可以指定其执行的顺序，数值越小越晚执行；
 * 本拦截器拦截"不"通过实体进行的创建和更新操作，"不强制"设置基础字段，如果有对基础字段手动赋值，则采用手动设置的值</p>
 * <p>
 * 注意，该拦截器从 2022-03-25开始，使用nebula-mars模块提供的拦截器策略
 * @author: Bao Hongbin   yinwenjie
 * @create: 2021-12-06 09:52
 **/
@Slf4j
@Component
public class DefaultMybatisBaseFieldEnhanceInterceptor implements NebulaRepositoryInterceptorStrategy {
  private static final String CREATE_TIME = "create_time";
  private static final String CREATE_NAME = "create_name";
  private static final String CREATE_ACCOUNT = "create_account";

  private static final String MODIFY_TIME = "modify_time";
  private static final String MODIFY_NAME = "modify_name";
  private static final String MODIFY_ACCOUNT = "modify_account";

  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private JpaProperties jpaProperties;

  /**
   * 步骤：
   * 1 获取mappedStatement对象，修改sql语句是基于该对象进行的
   * 2 判断是否为新增操作
   * 2.1 构建新增语句的Statement
   * 2.1.1 如果存在字段但未设值，则进行赋值
   * 2.1.2 如果不存在字段，则新增对应字段并进行赋值
   * 2.2 回写修改后的SQL到mappedStatement对象中
   * 3 判断是否为更新操作
   * 3.1 构建更新语句的Statement
   * 3.1.1 如果存在字段但未设值，则进行赋值
   * 3.1.2 如果不存在字段，则新增对应字段并进行赋值
   * 3.2 回写修改后的SQL到mappedStatement对象中
   *
   * @param sqlCommandType
   * @return
   * @throws Throwable
   */
  @Override
  public String intercept(SqlCommandType sqlCommandType, String sql, Connection currentConnection) {
    if (sqlCommandType != SqlCommandType.INSERT && sqlCommandType != SqlCommandType.UPDATE) {
      return sql;
    }

    try {
      switch (sqlCommandType) {
        case INSERT:
          return handleInsertSql(sql);
        case UPDATE:
          return handleUpdateSql(sql);
        default:
          break;
      }
      return sql;
    } catch (Throwable e) {
      log.error(e.getMessage(), e);
      throw new IllegalArgumentException(e);
    }
  }

  /**
   * 处理updateSql，如果满足条件则对其设值
   *
   * @param sql
   * @throws Throwable
   */
  private String handleUpdateSql(String sql) throws SQLSyntaxErrorException, NoSuchFieldException, IllegalAccessException {
    SQLUpdateStatement sqlUpdateStatement = (SQLUpdateStatement) parser(sql, JdbcConstants.MYSQL);
    if (Objects.isNull(sqlUpdateStatement.getTableName())) {
      // 获取不到表名直接跳过设置
      return sql;
    }
    // 获取表名，判断该表对应的实体是否需要设置字段
    String tableName = sqlUpdateStatement.getTableName().getSimpleName();
    log.debug("获取的表名为：{}", tableName);
    TableInfo tableInfo = TableInfoHelper.getTableInfo(tableName);
    if (tableInfo == null || !UuidOpEntity.class.isAssignableFrom(tableInfo.getEntityType())) {
      // 只对UuidOpEntity类型及其子类进行拦截设值
      return sql;
    }
    buildUpdateStatement(sqlUpdateStatement);
    //设置新sql
    String newSql = sqlUpdateStatement.toString();
    return newSql;
  }

  private void buildUpdateStatement(SQLUpdateStatement sqlUpdateStatement) {
    //判断是否存在对应的字段
    List<SQLUpdateSetItem> items = sqlUpdateStatement.getItems();
    boolean existModifyTime = false;
    boolean existModifyName = false;
    boolean existModifyAccount = false;
    AbstractCrmUserIdentity context = loginUserService.getAbstractLoginUser();
    String account = context != null && StringUtils.isNotBlank(context.getAccount()) ? context.getAccount() : "admin";
    String name = context != null && StringUtils.isNotBlank(context.getRealName()) ? context.getRealName() : "系统用户或系统定时任务";
    for (SQLUpdateSetItem item : items) {
      if (item.columnMatch(MODIFY_TIME)) {
        SQLExpr value = item.getValue();
        if (value instanceof SQLNullExpr) {
          //如果更新时间，在存在字段但没有预设值时，则更改
          String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(Calendar.getInstance().getTime());
          SQLCharExpr sqlCharExpr = new SQLCharExpr(now);
          item.setValue(sqlCharExpr);
        }
        existModifyTime = true;
      } else if (item.columnMatch(MODIFY_NAME)) {
        SQLExpr value = item.getValue();
        if (value instanceof SQLNullExpr) {
          //如果更新人名称，在存在字段但没有预设值时，则更改
          SQLCharExpr sqlCharExpr = new SQLCharExpr(name);
          item.setValue(sqlCharExpr);
        }
        existModifyName = true;
      } else if (item.columnMatch(MODIFY_ACCOUNT)) {
        SQLExpr value = item.getValue();
        if (value instanceof SQLNullExpr) {
          //如果更新人账号，在存在字段但没有预设值时，则更改
          SQLCharExpr sqlCharExpr = new SQLCharExpr(account);
          item.setValue(sqlCharExpr);
        }
        existModifyAccount = true;
      }
    }
    if (!existModifyTime) {
      SQLUpdateSetItem sqlUpdateSetItem = new SQLUpdateSetItem();
      sqlUpdateSetItem.setColumn(new SQLIdentifierExpr(MODIFY_TIME));
      SQLExpr sqlExpr = null;
      String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(Calendar.getInstance().getTime());
      if (this.jpaProperties.getDatabase() == Database.ORACLE) {
        List<SQLExpr> params = new ArrayList<>();
        params.add(new SQLCharExpr(now));
        params.add(new SQLCharExpr("yyyy-MM-dd HH24:mi:ss"));
        SQLMethodInvokeExpr methodInvokeExpr = new SQLMethodInvokeExpr("to_date", null, params);
        sqlUpdateSetItem.setValue(methodInvokeExpr);
      } else {
        sqlExpr = new SQLCharExpr(now);
        sqlUpdateSetItem.setValue(sqlExpr);
      }
      sqlUpdateStatement.addItem(sqlUpdateSetItem);
    }
    if (!existModifyName) {
      SQLUpdateSetItem sqlUpdateSetItem = new SQLUpdateSetItem();
      sqlUpdateSetItem.setColumn(new SQLIdentifierExpr(MODIFY_NAME));
      SQLCharExpr sqlCharExpr = new SQLCharExpr(name);
      sqlUpdateSetItem.setValue(sqlCharExpr);
      sqlUpdateStatement.addItem(sqlUpdateSetItem);
    }
    if (!existModifyAccount) {
      SQLUpdateSetItem sqlUpdateSetItem = new SQLUpdateSetItem();
      sqlUpdateSetItem.setColumn(new SQLIdentifierExpr(MODIFY_ACCOUNT));
      SQLCharExpr sqlCharExpr = new SQLCharExpr(account);
      sqlUpdateSetItem.setValue(sqlCharExpr);
      sqlUpdateStatement.addItem(sqlUpdateSetItem);
    }
  }

  /**
   * 处理insertSql，如果满足条件则对其设值
   *
   * @param sql
   * @throws Throwable
   */
  private String handleInsertSql(String sql) throws SQLSyntaxErrorException, NoSuchFieldException, IllegalAccessException {
    SQLInsertStatement sqlInsertStatement = (SQLInsertStatement) parser(sql, JdbcConstants.MYSQL);
    // 获取表名，判断该表对应的实体是否需要设置字段
    String tableName = sqlInsertStatement.getTableName().getSimpleName();
    log.debug("获取的表名为：{}", tableName);
    TableInfo tableInfo = TableInfoHelper.getTableInfo(tableName);
    if (tableInfo == null || !UuidOpEntity.class.isAssignableFrom(tableInfo.getEntityType())) {
      // 只对UuidOpEntity类型及其子类进行拦截设值
      return sql;
    }
    SQLSelect query = sqlInsertStatement.getQuery();
    if (Objects.nonNull(query)) {
      // 此类型的语句不做解析：INSERT INTO table2 SELECT * FROM table1;
      return sql;
    }
    List<SQLExpr> columns = sqlInsertStatement.getColumns();
    if (CollectionUtils.isEmpty(columns)) {
      // 没有列字段，不做解析
      return sql;
    }
    //构建Statement
    buildInsertStatement(sqlInsertStatement, columns);
    //设置新sql
    String newSql = sqlInsertStatement.toString();
    return newSql;
  }

  /**
   * 构建插入语句的Statement
   *
   * @param sqlInsertStatement
   * @param columns
   */
  private void buildInsertStatement(SQLInsertStatement sqlInsertStatement, List<SQLExpr> columns) {
    //判断是否已经存在对应的字段
    boolean existCreateTime = false;
    boolean existCreateName = false;
    boolean existCreateAccount = false;
    boolean existModifyTime = false;
    boolean existModifyName = false;
    boolean existModifyAccount = false;
    AbstractCrmUserIdentity context = this.loginUserService.getAbstractLoginUser();
    String account = context != null && StringUtils.isNotBlank(context.getAccount()) ? context.getAccount() : "admin";
    String name = context != null && StringUtils.isNotBlank(context.getRealName()) ? context.getRealName() : "系统用户或系统定时任务";
    List<SQLInsertStatement.ValuesClause> valuesList = sqlInsertStatement.getValuesList();
    for (int i = 0; i < columns.size(); i++) {
      //如果存在对应的基础字段，则对其设值
      SQLExpr sqlExpr = columns.get(i);
      switch (sqlExpr.toString()) {
        case CREATE_TIME:
          setInsertExistCreateOrModifyTimeValues(valuesList, i);
          existCreateTime = true;
          break;
        case CREATE_NAME:
          setInsertExistCreateOrModifyNameValues(valuesList, name, i);
          existCreateName = true;
          break;
        case CREATE_ACCOUNT:
          setInsertExistCreateOrModifyAccountValues(valuesList, account, i);
          existCreateAccount = true;
          break;
        case MODIFY_TIME:
          setInsertExistCreateOrModifyTimeValues(valuesList, i);
          existModifyTime = true;
          break;
        case MODIFY_NAME:
          setInsertExistCreateOrModifyNameValues(valuesList, account, i);
          existModifyName = true;
          break;
        case MODIFY_ACCOUNT:
          setInsertExistCreateOrModifyAccountValues(valuesList, account, i);
          existModifyAccount = true;
          break;
      }
    }
    //如果不存在对应字段，则设置字段与值
    setInsertCreateOrModifyTimeValues(sqlInsertStatement, valuesList, existCreateTime, CREATE_TIME);
    setInsertCreateOrModifyNameValues(sqlInsertStatement, valuesList, existCreateName, name, CREATE_NAME);
    setInsertCreateOrModifyAccountValues(sqlInsertStatement, valuesList, existCreateAccount, account, CREATE_ACCOUNT);
    setInsertCreateOrModifyTimeValues(sqlInsertStatement, valuesList, existModifyTime, MODIFY_TIME);
    setInsertCreateOrModifyNameValues(sqlInsertStatement, valuesList, existModifyName, name, MODIFY_NAME);
    setInsertCreateOrModifyAccountValues(sqlInsertStatement, valuesList, existModifyAccount, account, MODIFY_ACCOUNT);
  }

  /**
   * 设置创建语句不存在的创建人帐号和更新人帐号字段的值
   *
   * @param sqlInsertStatement
   * @param valuesList
   * @param existCreateAccount
   * @param account
   * @param createAccount
   */
  private void setInsertCreateOrModifyAccountValues(SQLInsertStatement sqlInsertStatement, List<SQLInsertStatement.ValuesClause> valuesList
      , boolean existCreateAccount, String account, String createAccount) {
    if (existCreateAccount) {
      return;
    }
    sqlInsertStatement.addColumn(new SQLIdentifierExpr(createAccount));
    for (SQLInsertStatement.ValuesClause valuesClause : valuesList) {
      List<SQLExpr> values = valuesClause.getValues();
      SQLCharExpr sqlCharExpr = new SQLCharExpr(account);
      values.add(sqlCharExpr);
    }
  }

  /**
   * 设置创建语句不存在的创建人名称和更新人名称字段的值
   *
   * @param sqlInsertStatement
   * @param valuesList
   * @param existCreateName
   * @param name
   * @param createName
   */
  private void setInsertCreateOrModifyNameValues(SQLInsertStatement sqlInsertStatement, List<SQLInsertStatement.ValuesClause> valuesList
      , boolean existCreateName, String name, String createName) {
    if (existCreateName) {
      return;
    }
    sqlInsertStatement.addColumn(new SQLIdentifierExpr(createName));
    for (SQLInsertStatement.ValuesClause valuesClause : valuesList) {
      List<SQLExpr> values = valuesClause.getValues();
      SQLCharExpr sqlCharExpr = new SQLCharExpr(name);
      values.add(sqlCharExpr);
    }
  }

  /**
   * 设置创建语句已经存在的创建人帐号和更新人帐号字段的值
   *
   * @param valuesList
   * @param account
   * @param i
   */
  private void setInsertExistCreateOrModifyAccountValues(List<SQLInsertStatement.ValuesClause> valuesList, String account, int i) {
    for (SQLInsertStatement.ValuesClause valuesClause : valuesList) {
      List<SQLExpr> values = valuesClause.getValues();
      if (!(values.get(i) instanceof SQLNullExpr)) {
        //在存在字段且有预设值时，则不做更改
        return;
      }
      SQLCharExpr sqlCharExpr = new SQLCharExpr(account);
      values.set(i, sqlCharExpr);
    }
  }

  /**
   * 设置创建语句不存在的创建时间和更新时间字段的值
   *
   * @param sqlInsertStatement
   * @param valuesList
   * @param existCreateTime
   * @param createTime
   */
  private void setInsertCreateOrModifyTimeValues(SQLInsertStatement sqlInsertStatement, List<SQLInsertStatement.ValuesClause> valuesList, boolean existCreateTime, String createTime) {
    if (existCreateTime) {
      return;
    }

    sqlInsertStatement.addColumn(new SQLIdentifierExpr(createTime));
    for (SQLInsertStatement.ValuesClause valuesClause : valuesList) {
      List<SQLExpr> values = valuesClause.getValues();
      SQLExpr sqlExpr = null;
      if (this.jpaProperties.getDatabase() == Database.ORACLE) {
        String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(Calendar.getInstance().getTime());
        SQLCharExpr nowExpr = new SQLCharExpr(now);
        SQLCharExpr formatExpr = new SQLCharExpr("yyyy-MM-dd HH24:mi:ss");
        sqlExpr = new OracleDatetimeExpr(nowExpr, formatExpr);
      } else {
        String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(Calendar.getInstance().getTime());
        if (this.jpaProperties.getDatabase() == Database.ORACLE) {
          List<SQLExpr> params = new ArrayList<>();
          params.add(new SQLCharExpr(now));
          params.add(new SQLCharExpr("yyyy-MM-dd HH24:mi:ss"));
          sqlExpr = new SQLMethodInvokeExpr("to_date", null, params);
        } else {
          sqlExpr = new SQLCharExpr(now);
        }
      }
      values.add(sqlExpr);
    }
  }

  /**
   * 设置创建语句已经存在的创建人名称和更新人名称字段的值
   *
   * @param valuesList
   * @param name
   * @param i
   */
  private void setInsertExistCreateOrModifyNameValues(List<SQLInsertStatement.ValuesClause> valuesList, String name, int i) {
    for (SQLInsertStatement.ValuesClause valuesClause : valuesList) {
      List<SQLExpr> values = valuesClause.getValues();
      if (!(values.get(i) instanceof SQLNullExpr)) {
        //在存在字段且有预设值时，则不做更改
        return;
      }
      SQLCharExpr sqlCharExpr = new SQLCharExpr(name);
      values.set(i, sqlCharExpr);
    }
  }

  /**
   * 设置创建语句已经存在的创建时间和更新时间字段的值
   *
   * @param valuesList
   * @param i
   */
  private void setInsertExistCreateOrModifyTimeValues(List<SQLInsertStatement.ValuesClause> valuesList, int i) {
    for (SQLInsertStatement.ValuesClause valuesClause : valuesList) {
      List<SQLExpr> values = valuesClause.getValues();
      if (!(values.get(i) instanceof SQLNullExpr)) {
        //在存在字段且有预设值时，则不做更改
        return;
      }

      SQLExpr sqlExpr = null;
      if (this.jpaProperties.getDatabase() == Database.ORACLE) {
        String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(Calendar.getInstance().getTime());
        SQLCharExpr nowExpr = new SQLCharExpr(now);
        SQLCharExpr formatExpr = new SQLCharExpr("yyyy-MM-dd HH24:mi:ss");
        sqlExpr = new OracleDatetimeExpr(nowExpr, formatExpr);
      } else {
        String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(Calendar.getInstance().getTime());
        sqlExpr = new SQLCharExpr(now);
      }
      values.set(i, sqlExpr);
    }
  }

  public SQLStatement parser(String sql, String dbType) throws SQLSyntaxErrorException {
    List<SQLStatement> list = SQLUtils.parseStatements(sql, dbType);
    if (list.size() > 1) {
      throw new SQLSyntaxErrorException("MultiQueries is not supported,use single query instead ");
    }
    return list.get(0);
  }

  @Override
  public int sort() {
    return 2;
  }


}
