package com.biz.crm.business.common.sdk.utils;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

/**
 * @author ning.zhang
 * @description 登录用户基础工具类(可用于获取用户信息)
 * @date 2025/03/12
 */
@Component
public class LoginUserContextUtil implements ApplicationContextAware {

  private static ApplicationContext applicationContext;

  @Override
  public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
    LoginUserContextUtil.applicationContext = applicationContext;
  }

  /**
   * 获取用户身份服务实例
   *
   * @return 用户身份服务实例
   */
  public static LoginUserService getLoginUserService() {
    LoginUserService loginUserService = applicationContext.getBean(LoginUserService.class);
    Validate.notNull(loginUserService, "未获取到用户身份服务实例!");
    return loginUserService;
  }

  /**
   * 得到抽象登录用户,是本系统登录用户都会继承的对象
   * @return {@link AbstractCrmUserIdentity}
   */
  public static AbstractCrmUserIdentity getAbstractLoginUser() {
    return LoginUserContextUtil.getLoginUserService().getAbstractLoginUser();
  }

  /**
   * 获取登录用户，并将登录信息转成json对象
   *
   * @return {@link JSONObject}
   */
  public static JSONObject getLoginUserJson() {
    return LoginUserContextUtil.getLoginUserService().getLoginUserJson();
  }

  /**
   * 获取当前线程的用户身份上下文会话
   * @return
   */
  public static Authentication getAuthentication() {
    return LoginUserContextUtil.getLoginUserService().getAuthentication();
  }
  /**
   * 获取登录用户名
   * @return 登录用户名
   */
  public static String getLoginAccountName() {
    return LoginUserContextUtil.getLoginUserService().getLoginAccountName();
  }

  /**
   * 获取登录用户信息
   * LoginDetails为全局登录对象用户信息，但不包含子系统扩展字段信息
   * @return 全局登录对象用户信息
   */
  public static UserIdentity getLoginUser() {
    return LoginUserContextUtil.getLoginUserService().getLoginUser();
  }
  /**
   * 获取当前子系统登录对象，注意，该方法不推荐直接使用
   * mdm模块登录对象：LoginUserDetails
   * dms模块登录对象：LoginUserDetails
   * ...
   * 其他模块登录对象：LoginXxxDetails
   *
   * @param targetClass 子系统登录对象class
   * @return 子系统登录对象
   */
  public static <T extends UserIdentity> T getLoginDetails(Class<T> targetClass) {
    return LoginUserContextUtil.getLoginUserService().getLoginDetails(targetClass);
  }
  /**
   * 在CRM系统开发过程中，经常有模块根据自身要求重新初始化当前线程的用户上下文信息，例如导入导出中启动一个新线程的时候，需要为这个线程初始化操作者身份信息；
   * 再例如CRM中切换当前登录人的“当前岗位”时，要重新初始化当前线程中，用户上下文中用户信息的“currentPositionCode”值。</p>
   *
   * 但是由于CRM标品研发规范要求，除boot启动器以外，其它业务模块均不能直接依赖identity.distributor、identity.facturer等具体的用户身份模块，所以世界上业务模块是不知道如何初始化/刷新当前线程中securityContext内容的。</p>
   *
   * 这个时候就需要调用AuthenticationUserService.refreshAuthentication方法，将初始化/刷新securityContext内容的逻辑交给上层的boot启动器。</br>
   *
   * @param userInfo 这个可以是任意入参，例如如果是导入导出功能要求初始化/刷新securityContext内容，那么这个参数可能是导入导出任务的vo对象；</br>
   * 再例如；如果是优惠政策模块要求初始化/刷新securityContext内容，那么这个参数可能是当前优惠政策的vo对象。</br>
   * 该参数也可以为null，如果是这样，那么进行当前线程初始化/刷新securityContext内容所写入的用户身份，可能是就是一种固定的实现（例如直接采用当前系统管理员用户身份）</p>
   *
   * 该方法调用后，最终将触发某一种AuthenticationRefreshStrategy测了的实现逻辑（请参见AuthenticationRefreshStrategy接口本身的描述）
   *
   * @return 如果初始化成功，初始化完成的新的用户身份认证信息将会被返回；如果失败将会返回null
   */
  public static Authentication refreshAuthentication(Object userInfo) {
    return LoginUserContextUtil.getLoginUserService().refreshAuthentication(userInfo);
  }

  /**
   * 该方法负责从spring-sercutiy组件中获取当前登录人信息
   * 注意：没有登录信息，则默认返回admin
   * @return String
   */
  public static String findCurrentAccount(){
    return LoginUserContextUtil.getLoginUserService().findCurrentAccount();
  }

}
