package com.biz.crm.common.personalized.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.common.personalized.local.entity.PersonalizedTheme;
import com.biz.crm.common.personalized.local.repository.PersonalizedThemeRepository;
import com.biz.crm.common.personalized.local.service.PersonalizedThemeService;
import com.biz.crm.common.personalized.sdk.constant.PersonalizedThemeConstant;
import com.biz.crm.common.personalized.sdk.dto.PersonalizedThemePaginationDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author hejun
 * @date 2023/2/13 14:54
 * @description personalized_theme 个性化主题PersonalizedTheme表服务实现类
 */
@Service("personalizedThemeService")
public class PersonalizedThemeServiceImpl implements PersonalizedThemeService {

  @Autowired(required = false)
  private PersonalizedThemeRepository personalizedThemeRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Override
  public Page<PersonalizedTheme> findByConditions(Pageable pageable, PersonalizedThemePaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0,50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new PersonalizedThemePaginationDto());
    paginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    Page<PersonalizedTheme>page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<PersonalizedTheme>pageResult = this.personalizedThemeRepository.findByConditions(page,paginationDto);
    if (CollectionUtils.isEmpty(pageResult.getRecords())){
      return null;
    }
    return pageResult;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.personalizedThemeRepository.updateEnableStatusByIds(ids,EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.personalizedThemeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public PersonalizedTheme create(PersonalizedTheme personalizedTheme) {
    this.validation(personalizedTheme);
    if (StringUtils.isEmpty(personalizedTheme.getThemeCode())){
      String themeCode =  this.generateCodeService.generateCode(PersonalizedThemeConstant.THEME_CODE,1).get(0);
      personalizedTheme.setThemeCode(themeCode);
    }else{
      Integer count = this.personalizedThemeRepository.CountByThemeCode(personalizedTheme.getThemeCode());
      Validate.isTrue(null == count || 1 > count, personalizedTheme.getThemeCode() + "系统主题编码已存在");
    }
    personalizedTheme.setTenantCode(TenantUtils.getTenantCode());
    personalizedTheme.setCreateName(personalizedTheme.getUserName());
    personalizedTheme.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    personalizedTheme.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    personalizedTheme.setCreateTime(new Date());
    personalizedThemeRepository.saveOrUpdate(personalizedTheme);
    return personalizedTheme;
  }

  @Override
  @Transactional
  public PersonalizedTheme update(PersonalizedTheme personalizedTheme) {
    this.validation(personalizedTheme);
    personalizedTheme.setTenantCode(TenantUtils.getTenantCode());
    personalizedTheme.setModifyName(personalizedTheme.getUserName());
    personalizedTheme.setModifyTime(new Date());
    PersonalizedTheme theme = this.personalizedThemeRepository.findById(personalizedTheme.getId(),personalizedTheme.getTenantCode());
    Validate.isTrue(theme.getThemeCode().equals(personalizedTheme.getThemeCode()),"系统主题编码不能修改");
    this.personalizedThemeRepository.saveOrUpdate(personalizedTheme);
    return theme;
  }

  @Override
  public PersonalizedTheme findDetailById(String id) {
    return this.personalizedThemeRepository.findById(id,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<PersonalizedTheme> themes = this.personalizedThemeRepository.findThemesByIds(ids);
    if (CollectionUtils.isEmpty(themes)){
      return;
    }
    this.personalizedThemeRepository.updateBatchByIds(ids);
  }


  private void validation(PersonalizedTheme personalizedTheme) {
    Validate.notNull(personalizedTheme, "主题信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(personalizedTheme.getThemeName()), "系统主题名称不能为空");
    Validate.isTrue(StringUtils.isNotBlank(personalizedTheme.getUserName()), "用户名称不能为空");
  }
}
