package com.biz.crm.common.personalized.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.personalized.local.entity.ScreenButton;
import com.biz.crm.common.personalized.local.repository.ScreenButtonRepository;
import com.biz.crm.common.personalized.local.service.ScreenButtonService;
import com.biz.crm.common.personalized.sdk.constant.ScreenButtonConstant;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.util.List;
import java.util.Optional;

/**
 * @author XXLsansui
 * @date 2023-02-13 16:38
 * @description：列表模板-常用筛选服务实现类
 */
@Service("screenButtonService")
public class ScreenButtonServiceImpl implements ScreenButtonService {

  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private ScreenButtonRepository screenButtonRepository;
  @Autowired
  private GenerateCodeService generateCodeService;

  @Override
  public Page<ScreenButton> findByConditions(Pageable pageable, ScreenButton screenButton) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    screenButton = Optional.ofNullable(screenButton).orElse(new ScreenButton());
    /**
     * 当前用户
     * 租户编码
     * 未删除状态
     * 数据视图
     */
    screenButton.setCurrentAccount(loginUserService.getLoginAccountName());
    screenButton.setTenantCode(TenantUtils.getTenantCode());
    screenButton.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

    Page<ScreenButton> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<ScreenButton> result = this.screenButtonRepository.findByConditions(page, screenButton);
    return result;
  }

  /**
   * 新建配置
   *
   * @param screenButton 配置实体
   * @return 新建后的实体类
   */
  @Override
  @Transactional
  public ScreenButton create(ScreenButton screenButton) {
    checkData(screenButton);
    //校验
    String screenButtonCode = this.generateCodeService.generateCode(ScreenButtonConstant.SCREEN_BUTTON, 1).get(0);
    screenButton.setScreenButtonCode(screenButtonCode);
    screenButton.setCurrentAccount(loginUserService.getLoginAccountName());
    screenButton.setTenantCode(TenantUtils.getTenantCode());
    screenButton.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    screenButton.setCreateName(loginUserService.getLoginAccountName());
    screenButton.setCreateTime(new Date());
    this.screenButtonRepository.saveOrUpdate(screenButton);
    return screenButton;
  }

  /**
   * 更新配置
   *
   * @param screenButton 配置实体
   * @return 更新后的实体类
   */
  @Override
  public ScreenButton update(ScreenButton screenButton) {
    //校验
    checkData(screenButton);
    Validate.isTrue(!ObjectUtils.isEmpty(screenButton.getScreenButtonCode()), "常用筛选模板编码缺失");
    //找出旧数据
    ScreenButton screenButtonOld = this.screenButtonRepository.getById(screenButton.getId());
    screenButtonOld.setScreenButtonName(screenButton.getScreenButtonName());
    screenButtonOld.setIsDefault(screenButton.getIsDefault());
    screenButtonOld.setDataJson(screenButton.getDataJson());
    screenButtonOld.setModifyName(loginUserService.getLoginAccountName());
    screenButtonOld.setModifyTime(new Date());
    screenButtonRepository.saveOrUpdate(screenButtonOld);
    return screenButtonOld;
  }

  /**
   * 批量删除
   *
   * @param ids 主键列表
   */
  @Override
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    screenButtonRepository.lambdaUpdate()
            .eq(ScreenButton::getTenantCode, TenantUtils.getTenantCode())
            .in(ScreenButton::getId, ids)
            .set(ScreenButton::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
            .update();
  }

  public void checkData(ScreenButton screenButton){
    Validate.isTrue(!ObjectUtils.isEmpty(screenButton.getScreenButtonName()), "请填写筛选模板名称");
    Validate.isTrue(!ObjectUtils.isEmpty(screenButton.getDataJson()), "请选择筛选字段");
  }

}
