package com.biz.crm.dms.admin.web.report.mall.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.constant.LoginUserConstant;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.admin.web.report.mall.service.ProductWithMiniAndStockVoService;
import com.biz.crm.dms.admin.web.report.mall.service.assist.PolicyAssist;
import com.biz.crm.dms.admin.web.report.mall.vo.HomePageSkuPolicyVo;
import com.biz.crm.dms.admin.web.report.mall.vo.ProductSkuWithMiniAndStockVo;
import com.biz.crm.dms.admin.web.report.mall.vo.ProductWithMiniAndStockVo;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListDmsPaginationDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.service.AllowSaleListDmsVoService;
import com.biz.crm.dms.business.allow.sale.sdk.list.vo.AllowSaleListDmsVo;
import com.biz.crm.dms.business.favorite.sdk.service.FavoriteProductService;
import com.biz.crm.dms.business.order.minimum.local.entity.DmsOrderMinimum;
import com.biz.crm.dms.business.order.minimum.local.service.DmsOrderMinimumService;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductStockVo;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressQueryDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerAddressVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerAddressVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;

import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuVo;
import com.biz.crm.mdm.business.warehouse.sdk.service.WarehouseVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 根据spuCode查询
 *
 * @author hefan
 * @date 2022/04/22
 */
@Service
public class ProductWithMiniAndStockVoServiceImpl implements ProductWithMiniAndStockVoService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private WarehouseVoService warehouseVoService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  @Autowired(required = false)
  private CustomerAddressVoService customerAddressVoService;

  @Autowired(required = false)
  private DmsOrderMinimumService dmsOrderMinimumService;

  /**
   * - 商品的详细信息
   * - 促销政策
   * - 库存
   * - 起订量
   */
  @Override
  public ProductWithMiniAndStockVo findBySpuCode(String spuCode,Boolean isFuzzy) {
    /**
     * - 要查询商品的详细信息
     * - 库存
     * - 起订量
     * - 不含促销政策
     */
    ProductWithMiniAndStockVo productWithMiniAndStockVo = this.findDetailsBySpuCode(spuCode,isFuzzy);
    // - 查询仓库
    WarehouseVo warehouse = this.findWarehouse();
    // - SKU集合
    List<ProductSkuWithMiniAndStockVo> productList = productWithMiniAndStockVo.getProductList();
    Set<String> skuCodes = productList.stream()
        .map(ProductSkuWithMiniAndStockVo::getProductCode)
        .collect(Collectors.toSet());
    // - 查询库存
    List<ProductStockVo> productStockVos = productStockVoService.findByWarehouseAndProducts(warehouse.getWarehouseCode(), skuCodes);
    Map<String, BigDecimal> skuToAvailableMap = new HashMap<>(productStockVos.size());
    for (ProductStockVo stockVo : productStockVos) {
      skuToAvailableMap.put(stockVo.getProductCode(), stockVo.getAvailableStock());
    }
    // - 校验每个sku都有库存;
    // 前端说不校验，返回0; this.validateSkuStock(warehouse, skuCodes, skuToAvailableMap);
    // - 起订量
    JSONObject loginUserJson = loginUserService.getLoginUserJson();
    String orgCode = loginUserJson.getString(LoginUserConstant.FIELD_ORG_CODE);
    String customerCode = loginUserJson.getString(LoginUserConstant.FIELD_CUSTOMER_CODE);
    List<DmsOrderMinimum> minimums = dmsOrderMinimumService.findByCustomerCodeOrOrgCodeAndStatusInProductCodes(customerCode, orgCode, EnableStatusEnum.ENABLE.getCode(), skuCodes);
    Map<String, BigDecimal> skuToMinimunMap = new HashMap<>(minimums.size());
    for (DmsOrderMinimum minimum : minimums) {
      skuToMinimunMap.put(minimum.getProductCode(), minimum.getMinimum());
    }
    // - 整合返回数据
    for (ProductSkuWithMiniAndStockVo sku : productList) {
      String productCode = sku.getProductCode();
      sku.setMinimum(skuToMinimunMap.get(productCode));
      BigDecimal availableStock = skuToAvailableMap.get(productCode);
      sku.setAvailableStock(availableStock != null ? availableStock : BigDecimal.ZERO);
    }
    return productWithMiniAndStockVo;
  }

  /**
   * 校验每个sku都有库存
   *
   * @param warehouse         仓库
   * @param skuCodes          sku码
   * @param skuToAvailableMap sku可用地图
   */
  private void validateSkuStock(WarehouseVo warehouse, Set<String> skuCodes, Map<String, BigDecimal> skuToAvailableMap) {
    for (String skuCode : skuCodes) {
      boolean contain = skuToAvailableMap.containsKey(skuCode);
      Validate.isTrue(contain, "该商品【%s】在仓库【%s】暂未维护库存信息，请联系管理员。", skuCode, warehouse.getWarehouseCode());
    }
  }

  @Autowired(required = false)
  private PolicyAssist policyAssist;

  @Autowired(required = false)
  private FavoriteProductService favoriteProductService;

  @Override
  public ProductWithMiniAndStockVo findJoinPolicyBySpuCode(String spuCode) {
    // false代表在查询可够清单时不进行模糊搜索
    ProductWithMiniAndStockVo productWithMiniAndStockVo = this.findBySpuCode(spuCode,false);
    JSONObject loginUserJson = loginUserService.getLoginUserJson();
    String customerCode = loginUserJson.getString(LoginUserConstant.FIELD_CUSTOMER_CODE);
    // -查询收藏
    List<String> favoriteProduct = favoriteProductService.filterFavoriteSpuCodes(customerCode, Lists.newArrayList(spuCode));
    if (!CollectionUtils.isEmpty(favoriteProduct)) {
      productWithMiniAndStockVo.setFavorite(favoriteProduct.contains(spuCode));
    }
    // - SKU集合
    List<ProductSkuWithMiniAndStockVo> productList = productWithMiniAndStockVo.getProductList();
    List<String> productCodes = productList.stream().map(ProductSkuWithMiniAndStockVo::getProductCode).collect(Collectors.toList());
    // - 促销政策
    Map<String, List<HomePageSkuPolicyVo>> skuCodeToVoMap = policyAssist.find(customerCode, productCodes);
    // - 填充返回数据
    for (ProductSkuWithMiniAndStockVo sku : productList) {
      sku.setPolicyVoList(skuCodeToVoMap.get(sku.getProductCode()));
    }
    return productWithMiniAndStockVo;
  }

  /**
   * 查询仓库
   *
   * @return {@link WarehouseVo}
   */
  private WarehouseVo findWarehouse() {
    /**
     * - 查询当前登录经销商的默认收货地址认
     * - 如果没有收货地址就用默认仓库
     * - 如果有说货地址还是没有查到仓库，就用默认仓库
     */
    JSONObject loginUserJson = loginUserService.getLoginUserJson();
    String customerCode = loginUserJson.getString(LoginUserConstant.FIELD_CUSTOMER_CODE);
    // - 查询客户的默认的有效的收货地址
    String cityCode = this.findCityCode(customerCode);
    WarehouseVo warehouse = null;
    if (StringUtils.isBlank(cityCode)) {
      warehouse = warehouseVoService.findDetailsByWarehouseDefault(true);
    } else {
      warehouse = warehouseVoService.findDetailsByCityCode(cityCode);
      if (warehouse == null) {
        warehouse = warehouseVoService.findDetailsByWarehouseDefault(true);
      }
    }
    Validate.notNull(warehouse, "没有找到仓库信息");
    return warehouse;
  }

  /**
   * 查询当前经销商登录者的默认地址的cityCode
   *
   * @param customerCode 客户代码
   * @return {@link String}
   */
  private String findCityCode(String customerCode) {
    CustomerAddressQueryDto customerAddressQueryDto = new CustomerAddressQueryDto();
    customerAddressQueryDto.setDefaultAddress(true);
    customerAddressQueryDto.setCustomerCode(customerCode);
    customerAddressQueryDto.setTenantCode(TenantUtils.getAppCode());
    List<CustomerAddressVo> customerAddressList = customerAddressVoService.findByCustomerAddressQueryDto(customerAddressQueryDto);
    if (CollectionUtils.isEmpty(customerAddressList)) {
      return null;
    }
    for (CustomerAddressVo addressVo : customerAddressList) {
      return addressVo.getCityCode();
    }
    return null;
  }

  @Autowired(required = false)
  private ProductVoService productVoService;
  @Autowired(required = false)
  private AllowSaleListDmsVoService allowSaleListDmsVoService;

  /**
   * 根据spu编码查询一条信息
   * - 为了与列表信息一致，这里应该接入允销查询
   *
   * @param spuCode spu代码
   * @param isFuzzy true代表在查询可购清单时进行模糊搜索 false代表在查询可够清单时不进行模糊搜索
   * @return {@link ProductSpuVo}
   */
  private ProductWithMiniAndStockVo findDetailsBySpuCode(String spuCode,Boolean isFuzzy) {
    JSONObject loginUserJson = loginUserService.getLoginUserJson();
    String customerCode = loginUserJson.getString(LoginUserConstant.FIELD_CUSTOMER_CODE);
    AllowSaleListDmsPaginationDto allowSaleDto = new AllowSaleListDmsPaginationDto();
    allowSaleDto.setKeyword(spuCode);
    allowSaleDto.setBusinessCode(customerCode);
    allowSaleDto.setListType(AllowSaleListTypeEnums.CUSTOMER.getCode());
    Pageable pageable = PageRequest.of(0, 1);
    Page<AllowSaleListDmsVo> allowPage =null;
    if (isFuzzy){
      //模糊搜索
       allowPage = allowSaleListDmsVoService.findByConditions(pageable, allowSaleDto);
    }else{
      //非模糊搜索
      allowPage = allowSaleListDmsVoService.findByConditionsPrecise(pageable, allowSaleDto);
    }
    Validate.notNull(allowPage, "没有该商品的的信息");
    List<AllowSaleListDmsVo> records = allowPage.getRecords();
    Validate.isTrue(!CollectionUtils.isEmpty(records), "没有该商品的的信息");
    AllowSaleListDmsVo allowSaleListDmsVo = records.get(0);
    ProductWithMiniAndStockVo productWithMiniAndStockVo = nebulaToolkitService.copyObjectByBlankList(allowSaleListDmsVo, ProductWithMiniAndStockVo.class, HashSet.class, LinkedList.class, "priceShowList");
    productWithMiniAndStockVo.setPriceShowList(allowSaleListDmsVo.getPriceShowList());
    // - 规格详情
    List<ProductSkuWithMiniAndStockVo> productList = productWithMiniAndStockVo.getProductList();
    List<String> skuCodeList = productList.stream()
        .map(ProductSkuWithMiniAndStockVo::getProductCode)
        .collect(Collectors.toList());
    List<ProductVo> skuList = productVoService.findDetailsByIdsOrProductCodes(Lists.newLinkedList(), skuCodeList);
    Map<String, ProductVo> mapSku = skuList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
        .collect(Collectors.toMap(ProductVo::getProductCode, Function.identity(), (a, b) -> a));
    for (ProductSkuWithMiniAndStockVo vo : productList) {
      ProductVo productVo = mapSku.get(vo.getProductCode());
      if (Objects.isNull(productVo)) {
        continue;
      }
      vo.setIsShelf(productVo.getIsShelf());
      vo.setProductLevelName(productVo.getProductLevelName());
      vo.setProductName(productVo.getProductName());
      vo.setProductType(productVo.getProductType());
      vo.setSpec(productVo.getSpec());
      vo.setUnite(productVo.getSaleUnit());
      vo.setBeginDateTime(productVo.getBeginDateTime());
      vo.setEndDateTime(productVo.getEndDateTime());
      vo.setPictureMediaList(productVo.getPictureMediaList());
      vo.setVideoMediaList(productVo.getVideoMediaList());
      vo.setIntroduction(productVo.getIntroduction());
    }
    return productWithMiniAndStockVo;
  }
}
