package com.biz.crm.dms.business.allow.sale.local.dimension.service.register;

import com.biz.crm.dms.business.allow.sale.local.dimension.detail.entity.AllowSaleRuleDetail;
import com.biz.crm.dms.business.allow.sale.local.dimension.detail.model.DetailRelateVo;
import com.biz.crm.dms.business.allow.sale.local.dimension.detail.service.AllowSaleRuleDetailService;
import com.biz.crm.dms.business.allow.sale.local.util.AllowSaleUtil;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.Dimension;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.register.AllowSaleRuleDimensionRegister;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.org.OrgDimensionItemVo;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.org.OrgDimensionVo;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleDimensionTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.org.sdk.dto.RelateOrgCodeQueryDto;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/**
 * 客户维度配置注册器实现
 *
 * @author sunx
 * @date 2021/11/23
 */
@Component
@Slf4j
public class OrgCustomerDimensionVoRegisterImpl
    implements AllowSaleRuleDimensionRegister<OrgDimensionVo> {

  @Autowired(required = false)
  private AllowSaleRuleDetailService allowSaleRuleDetailService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Override
  public String getType() {
    return AllowSaleDimensionTypeEnums.CUSTOMER_ORG.getCode();
  }

  @Override
  public String getTypeDesc() {
    return AllowSaleDimensionTypeEnums.CUSTOMER_ORG.getDesc();
  }

  @Override
  public int sort() {
    return 2;
  }

  @Override
  public Class<OrgDimensionVo> getDimensionClass() {
    return OrgDimensionVo.class;
  }

  @Override
  public String getListType() {
    return AllowSaleListTypeEnums.CUSTOMER.getCode();
  }

  @Override
  @Transactional
  public void saveDimensionDetail(Dimension dimension) {
    Validate.notNull(dimension, "客户组织维度规则信息不能为空");
    OrgDimensionVo vo = (OrgDimensionVo) dimension;
    Validate.notBlank(vo.getRuleCode(), "允销规则编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(vo.getList()), "客户组织维度规则明细不能为空");
    List<AllowSaleRuleDetail> list = Lists.newArrayList();
    vo.getList()
        .forEach(
            a -> {
              AllowSaleRuleDetail cur = new AllowSaleRuleDetail();
              cur.setRelateCode(a.getOrgCode());
              cur.setRuleCode(vo.getRuleCode());
              list.add(cur);
            });
    allowSaleRuleDetailService.batchSave(list);
  }

  @Override
  public OrgDimensionVo findByRuleCode(String ruleCode) {
    if (StringUtils.isBlank(ruleCode)) {
      return null;
    }
    List<AllowSaleRuleDetail> list =
        allowSaleRuleDetailService.findByRuleCodes(Lists.newArrayList(ruleCode));
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }

    List<String> codeList =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRelateCode()))
            .map(AllowSaleRuleDetail::getRelateCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(codeList)) {
      return null;
    }
    List<OrgVo> orgVoList = orgVoService.findByOrgCodes(codeList);
    if (CollectionUtils.isEmpty(orgVoList)) {
      return null;
    }
    Map<String, String> nameMap =
        orgVoList.stream()
            .collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (a, b) -> a));

    OrgDimensionVo vo = new OrgDimensionVo();
    List<OrgDimensionItemVo> subList = Lists.newArrayList();
    list.forEach(
        a -> {
          OrgDimensionItemVo cur = new OrgDimensionItemVo();
          cur.setOrgCode(a.getRelateCode());
          cur.setOrgName(nameMap.getOrDefault(a.getRelateCode(), StringUtils.EMPTY));
          subList.add(cur);
        });
    vo.setList(subList);
    return vo;
  }

  @Override
  public Map<String, Set<String>> findRelateRuleCodesMapByRelateCodes(List<String> relateCodeList) {
    // 根据组织编码获取当前组织及其关联的上级组织配置的可购规则信息，
    // 如 001001-规则A、规则B，001-规则C
    // 按照001001获取数据，会获取到001001-A、B、C
    if (CollectionUtils.isEmpty(relateCodeList)) {
      return Maps.newHashMap();
    }
    final RelateOrgCodeQueryDto relateOrgCodeQueryDto = new RelateOrgCodeQueryDto();
    relateOrgCodeQueryDto.setSearchType(1);
    relateOrgCodeQueryDto.setOrgCodeSet(Sets.newHashSet(relateCodeList));
    // k-orgCode,v-降维码
    final Map<String, String> orgRuleCodeMap =
        this.orgVoService.findByRelateOrgCodeQueryDto(relateOrgCodeQueryDto);
    if (orgRuleCodeMap == null || orgRuleCodeMap.isEmpty()) {
      return Maps.newHashMap();
    }
    List<DetailRelateVo> list =
        this.allowSaleRuleDetailService.findByDimensionTypeAndRelateCodes(
            this.getType(), Lists.newArrayList(orgRuleCodeMap.keySet()));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    // k-orgCode,v-规则编码集合
    final Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
            .collect(
                Collectors.groupingBy(
                    DetailRelateVo::getRelateCode,
                    Collectors.mapping(DetailRelateVo::getRuleCode, Collectors.toSet())));
    return AllowSaleUtil.findRelateRule(relateCodeList, orgRuleCodeMap, map);
  }

  @Override
  public Map<String, Set<String>> findRuleRelateBusinessCodesMapByRuleCodes(
      List<String> ruleCodeList) {
    // 根据规则编码获取规则配置的组织对应的所有子级经销商编码信息
    // 如 规则A配置的组织是001001、002001
    // 按照A去获取，会获取到001001下级所有经销商编码+002001下级所有经销商编码
    if (CollectionUtils.isEmpty(ruleCodeList)) {
      return Maps.newHashMap();
    }
    List<AllowSaleRuleDetail> list = this.allowSaleRuleDetailService.findByRuleCodes(ruleCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    // k-规则编码，v-组织编码集合
    Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleRuleDetail::getRuleCode,
                    Collectors.mapping(AllowSaleRuleDetail::getRelateCode, Collectors.toSet())));
    Set<String> orgCodeSet = Sets.newHashSet();
    for (Set<String> item : map.values()) {
      orgCodeSet.addAll(item);
    }
    final RelateOrgCodeQueryDto relateOrgCodeQueryDto = new RelateOrgCodeQueryDto();
    relateOrgCodeQueryDto.setSearchType(0);
    relateOrgCodeQueryDto.setOrgCodeSet(orgCodeSet);
    // orgCodeSet 查询到的对应的编码-降维码map
    final Map<String, String> orgCurRuleMap =
        this.orgVoService.findByRelateOrgCodeQueryDto(relateOrgCodeQueryDto);
    // k-经销商编码，v-关联组织对应的降维码集合
    final Map<String, Set<String>> mapCusRule =
        this.customerVoService.findAllowSaleCustomerByOrgCodes(orgCodeSet);
    if (mapCusRule == null || mapCusRule.isEmpty()) {
      return Maps.newHashMap();
    }

    // k-组织编码，v-经销商编码集合
    Map<String, Set<String>> orgCodeRelateRuleMap =
        AllowSaleUtil.findRelateBusiness(orgCurRuleMap, mapCusRule);
    if (orgCodeRelateRuleMap.isEmpty()) {
      return Maps.newHashMap();
    }

    Map<String, Set<String>> re = Maps.newHashMap();
    for (Entry<String, Set<String>> item : map.entrySet()) {
      final Set<String> cusCodeSet = Sets.newHashSet();
      for (String orgCode : item.getValue()) {
        final Set<String> curSet = orgCodeRelateRuleMap.get(orgCode);
        if (CollectionUtils.isEmpty(curSet)) {
          continue;
        }
        cusCodeSet.addAll(curSet);
      }
      if (CollectionUtils.isEmpty(cusCodeSet)) {
        continue;
      }
      re.put(item.getKey(), cusCodeSet);
    }
    return re;
  }
}
