package com.biz.crm.dms.business.allow.sale.local.list.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleList;
import com.biz.crm.dms.business.allow.sale.local.list.mapper.AllowSaleListMapper;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListPaginationDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.ValidateAllowSaleProductDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

/**
 * 允销清单信息(repository)
 *
 * @author sunx
 * @date 2021-11-19 15:15:41
 */
@Component
public class AllowSaleListRepository extends ServiceImpl<AllowSaleListMapper, AllowSaleList> {

  private static final Integer SIZE = 1000;

  /**
   * 分页
   *
   * @param page 分页
   * @param dto 查询参数
   * @return
   */
  public Page<AllowSaleList> findByConditions(
      Page<AllowSaleList> page, AllowSaleListPaginationDto dto) {
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.baseMapper.findByConditions(page, dto);
  }

  /**
   * 物理删除
   *
   * @param ids 主键集合
   */
  public void deleteByIds(List<String> ids) {
    for (List<String> list : Lists.partition(ids, SIZE)) {
      LambdaQueryWrapper<AllowSaleList> lambdaQuery = Wrappers.lambdaQuery();
      lambdaQuery.in(AllowSaleList::getId, list);
      baseMapper.delete(lambdaQuery);
    }
  }

  /**
   * 获取可购商品编码集合(验证商品是否可购)
   *
   * @param dto 验证可购商品请求dto
   * @return
   */
  public List<String> findAllowSaleProductCodes(ValidateAllowSaleProductDto dto) {
    List<AllowSaleList> list =
        this.lambdaQuery()
            .eq(AllowSaleList::getTenantCode, TenantUtils.getTenantCode())
            .eq(AllowSaleList::getNotAllow, 0)
            .ge(AllowSaleList::getAllow, 1)
            .eq(AllowSaleList::getListType, dto.getListType())
            .eq(AllowSaleList::getBusinessCode, dto.getBusinessCode())
            .in(
                CollectionUtils.isNotEmpty(dto.getProductCodeList()),
                AllowSaleList::getProductCode,
                dto.getProductCodeList())
            .select(AllowSaleList::getProductCode)
            .list();
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return list.stream().map(AllowSaleList::getProductCode).collect(Collectors.toList());
  }

  /**
   * 根据业务编码集合+清单类型获取明细数据
   *
   * @param businessCodes 业务编码集合
   * @param listType 清单类型
   * @param ruleType 规则类型
   * @return
   */
  @Deprecated
  public List<AllowSaleList> findByBusinessCodesAndListTypeAndRuleType(
      List<String> businessCodes, String listType, String ruleType) {
    return this.lambdaQuery()
        .eq(AllowSaleList::getListType, listType)
        .in(AllowSaleList::getBusinessCode, businessCodes)
        .select(AllowSaleList::getProductCode, AllowSaleList::getBusinessCode)
        .list();
  }

  /**
   * 根据清单类型获取业务编码集合
   *
   * @param listType 清单类型
   * @param ruleType 规则类型
   * @return
   */
  public Set<String> findBusinessCodesByListTypeAndRuleType(String listType, String ruleType) {
    if (StringUtils.isBlank(ruleType)
        || AllowSaleRuleTypeEnums.ALLOW.getDictCode().equals(ruleType)) {
      ruleType = AllowSaleRuleTypeEnums.ALLOW.getDictCode();
    } else {
      ruleType = AllowSaleRuleTypeEnums.NOT_ALLOW.getDictCode();
    }
    return this.baseMapper.findBusinessCodesByListTypeAndRuleType(listType, ruleType);
  }

  /**
   * 刷新关联可购规则数量
   *
   * @param ruleCode 规则编码
   * @param offset 偏移量
   */
  public void updateByAllowRule(String ruleCode, Integer offset) {
    this.baseMapper.updateByAllowRule(ruleCode, offset, TenantUtils.getTenantCode());
  }

  /**
   * 刷新关联不可购规则数量
   *
   * @param ruleCode 规则编码
   * @param offset 偏移量
   */
  public void updateByNotAllowRule(String ruleCode, Integer offset) {
    this.baseMapper.updateByNotAllowRule(ruleCode, offset, TenantUtils.getTenantCode());
  }

  /**
   * 获取已经存在的itemKey集合
   *
   * @param itemKeys 行记录编码
   * @return
   */
  public Set<String> findExistsKeys(Set<String> itemKeys) {
    Set<String> set = Sets.newHashSet();
    for (List<String> item : Lists.partition(Lists.newArrayList(itemKeys), SIZE)) {
      final Set<String> existsKeys =
          this.baseMapper.findExistsKeys(Sets.newHashSet(item), TenantUtils.getTenantCode());
      if (CollectionUtils.isEmpty(existsKeys)) {
        continue;
      }
      set.addAll(existsKeys);
    }
    return set;
  }

  /** 清理作废的清单数据（allow=0&&not_allow=0） */
  public void cleanListData() {
    LambdaQueryWrapper<AllowSaleList> lambdaQuery = Wrappers.lambdaQuery();
    lambdaQuery.eq(AllowSaleList::getAllow, 0).eq(AllowSaleList::getNotAllow, 0);
    this.baseMapper.delete(lambdaQuery);
  }

  /**
   * 根据itemKeys删除
   *
   * @param itemKeys 行记录编码
   */
  public void deleteByItemKeys(Set<String> itemKeys) {
    String tenantCode = TenantUtils.getTenantCode();
    for (List<String> list : Lists.partition(Lists.newArrayList(itemKeys), SIZE)) {
      LambdaQueryWrapper<AllowSaleList> lambdaQuery = Wrappers.lambdaQuery();
      lambdaQuery.eq(AllowSaleList::getTenantCode, tenantCode);
      lambdaQuery.in(AllowSaleList::getItemKey, list);
      baseMapper.delete(lambdaQuery);
    }
  }
}
