package com.biz.crm.dms.business.allow.sale.local.list.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleList;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleRuleRelateList;
import com.biz.crm.dms.business.allow.sale.local.list.model.RelateRule;
import com.biz.crm.dms.business.allow.sale.local.list.repository.AllowSaleListRepository;
import com.biz.crm.dms.business.allow.sale.local.list.repository.AllowSaleRuleRelateListRepository;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleListService;
import com.biz.crm.dms.business.allow.sale.local.util.AllowSaleUtil;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListPaginationDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.ValidateAllowSaleProductDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 允销清单信息(AllowSaleList)表服务实现类
 *
 * <p>可购清单是确定人货关系的唯一凭证
 *
 * <p>主要属性businessCode+listType+productCode
 *
 * <p>productCode确定货
 *
 * <p>businessCode+listType确定人
 *
 * <p>businessCode 根据业务不同可以是终端、客户、用户等
 *
 * <p>listType 为业务区分，和businessCode业务匹配
 *
 * @author sunx
 * @date 2021-11-19 15:15:42
 */
@Slf4j
@Service("allowSaleListService")
public class AllowSaleListServiceImpl implements AllowSaleListService {

  @Autowired(required = false)
  private AllowSaleListRepository allowSaleListRepository;

  @Autowired(required = false)
  private AllowSaleRuleRelateListRepository allowSaleRuleRelateListRepository;

  @Override
  public Page<AllowSaleList> findByConditions(Pageable pageable, AllowSaleListPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new AllowSaleListPaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    if (StringUtils.isBlank(dto.getRuleType())
        || AllowSaleRuleTypeEnums.ALLOW.getDictCode().equals(dto.getRuleType())) {
      dto.setRuleType(AllowSaleRuleTypeEnums.ALLOW.getDictCode());
    } else {
      dto.setRuleType(AllowSaleRuleTypeEnums.NOT_ALLOW.getDictCode());
    }
    Page<AllowSaleList> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.allowSaleListRepository.findByConditions(page, dto);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.allowSaleListRepository.deleteByIds(ids);
  }

  @Override
  public List<String> findAllowSaleProductCodes(ValidateAllowSaleProductDto dto) {
    if (Objects.isNull(dto)
        || StringUtils.isBlank(dto.getListType())
        || StringUtils.isBlank(dto.getBusinessCode())) {
      return Lists.newLinkedList();
    }
    return this.allowSaleListRepository.findAllowSaleProductCodes(dto);
  }

  // ==========================上面的是历史接口===============================================

  @Override
  @Transactional
  public void createByItemKeys(Set<String> itemKeys) {
    if (CollectionUtils.isEmpty(itemKeys)) {
      return;
    }
    this.allowSaleListRepository.deleteByItemKeys(itemKeys);
    // 获取规则清单关联数据
    final List<AllowSaleRuleRelateList> itemData =
        this.allowSaleRuleRelateListRepository.findByItemKeys(itemKeys);
    if (CollectionUtils.isEmpty(itemData)) {
      return;
    }
    Map<String, List<AllowSaleRuleRelateList>> map =
        itemData.stream()
            .filter(a -> StringUtils.isNotBlank(a.getItemKey()))
            .collect(Collectors.groupingBy(AllowSaleRuleRelateList::getItemKey));
    List<AllowSaleList> addDataList = Lists.newLinkedList();
    for (Entry<String, List<AllowSaleRuleRelateList>> item : map.entrySet()) {
      String itemKey = item.getKey();
      final AllowSaleList cur = AllowSaleUtil.findByItemKey(itemKey);
      Map<String, Long> curMap =
          item.getValue().stream()
              .filter(a -> StringUtils.isNotBlank(a.getRuleType()))
              .collect(
                  Collectors.groupingBy(
                      AllowSaleRuleRelateList::getRuleType, Collectors.counting()));
      Integer allow =
          curMap.getOrDefault(AllowSaleRuleTypeEnums.ALLOW.getDictCode(), 0L).intValue();
      Integer notAllow =
          curMap.getOrDefault(AllowSaleRuleTypeEnums.NOT_ALLOW.getDictCode(), 0L).intValue();
      if (Objects.isNull(cur) || (allow == 0 && notAllow == 0)) {
        continue;
      }
      cur.setTenantCode(TenantUtils.getTenantCode());
      cur.setItemKey(itemKey);
      cur.setAllow(allow);
      cur.setNotAllow(notAllow);
      addDataList.add(cur);
    }
    if (CollectionUtils.isEmpty(addDataList)) {
      return;
    }
    this.allowSaleListRepository.saveBatch(addDataList);
  }

  @Override
  @Transactional
  public void createByRule(RelateRule rule, List<AllowSaleList> list) {
    this.updateAllowNumByRule(rule, 1);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    for (AllowSaleList item : list) {
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    // 去重
    List<AllowSaleList> data = list.stream().distinct().collect(Collectors.toList());
    this.saveByList(data);
  }

  @Override
  @Transactional
  public void saveByList(List<AllowSaleList> saleList) {
    if (CollectionUtils.isEmpty(saleList)) {
      return;
    }
    final Set<String> allItemKeys =
        saleList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getItemKey()))
            .map(AllowSaleList::getItemKey)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(allItemKeys)) {
      return;
    }
    // 需要新增的明细的itemKey的集合
    Set<String> addKeys = this.findAddItemKeys(allItemKeys);
    if (CollectionUtils.isEmpty(addKeys)) {
      return;
    }
    // 需要新增的清单数据
    List<AllowSaleList> addList =
        saleList.stream()
            .filter(a -> addKeys.contains(a.getItemKey()))
            .collect(Collectors.toList());
    this.createItemsByList(addList);
  }

  @Override
  @Transactional
  public void updateAllowNumByRule(RelateRule rule, Integer offset) {
    Validate.isTrue(
        Objects.nonNull(rule) && StringUtils.isNoneBlank(rule.getRuleCode(), rule.getRuleType()),
        "规则编码、规则类型不能为空");
    if (AllowSaleRuleTypeEnums.ALLOW.getDictCode().equals(rule.getRuleType())) {
      this.allowSaleListRepository.updateByAllowRule(rule.getRuleCode(), offset);
    } else if (AllowSaleRuleTypeEnums.NOT_ALLOW.getDictCode().equals(rule.getRuleType())) {
      this.allowSaleListRepository.updateByNotAllowRule(rule.getRuleCode(), offset);
    }
    this.cleanListData();
  }

  @Override
  @Transactional
  public void cleanListData() {
    this.allowSaleListRepository.cleanListData();
  }

  /**
   * 分组新增到可购清单
   *
   * @param list
   */
  private void createItemsByList(List<AllowSaleList> list) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Set<String> itemKeys =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getItemKey()))
            .map(AllowSaleList::getItemKey)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(itemKeys)) {
      return;
    }
    List<AllowSaleRuleRelateList> data =
        this.allowSaleRuleRelateListRepository.findByItemKeys(itemKeys);
    if (CollectionUtils.isEmpty(data)) {
      return;
    }
    // k-itemKey,v-List
    Map<String, List<AllowSaleRuleRelateList>> map =
        data.stream().collect(Collectors.groupingBy(AllowSaleRuleRelateList::getItemKey));
    List<AllowSaleList> addList = Lists.newLinkedList();
    for (AllowSaleList item : list) {
      final List<AllowSaleRuleRelateList> curList = map.get(item.getItemKey());
      if (CollectionUtils.isEmpty(curList)) {
        continue;
      }
      Map<String, Long> mapNum =
          curList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getRuleType()))
              .collect(
                  Collectors.groupingBy(
                      AllowSaleRuleRelateList::getRuleType, Collectors.counting()));
      Integer allow =
          mapNum.getOrDefault(AllowSaleRuleTypeEnums.ALLOW.getDictCode(), 0L).intValue();
      Integer notAllow =
          mapNum.getOrDefault(AllowSaleRuleTypeEnums.NOT_ALLOW.getDictCode(), 0L).intValue();
      item.setAllow(allow);
      item.setNotAllow(notAllow);
      addList.add(item);
    }
    if (CollectionUtils.isEmpty(addList)) {
      return;
    }
    this.allowSaleListRepository.saveBatch(addList);
  }

  /**
   * 获取需要新增的itemKey集合
   *
   * @param itemKeys
   * @return
   */
  private Set<String> findAddItemKeys(Set<String> itemKeys) {
    if (CollectionUtils.isEmpty(itemKeys)) {
      return Sets.newHashSet();
    }
    Set<String> existsKeys = this.allowSaleListRepository.findExistsKeys(itemKeys);
    existsKeys = Optional.ofNullable(existsKeys).orElse(Sets.newHashSet());
    return Sets.newHashSet(Sets.difference(itemKeys, existsKeys));
  }
}
