package com.biz.crm.dms.business.allow.sale.local.list.service.internal;

import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleList;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleRuleRelateList;
import com.biz.crm.dms.business.allow.sale.local.list.model.RelateRule;
import com.biz.crm.dms.business.allow.sale.local.list.repository.AllowSaleRuleRelateListRepository;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleListService;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleRuleRelateListService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 可购规则与可购清单关系服务实现类
 *
 * @author sunx
 * @date 2022-06-24 15:29:39
 */
@Slf4j
@Service("allowSaleDetailRelateRuleService")
public class AllowSaleRuleRelateListServiceImpl implements AllowSaleRuleRelateListService {

  @Autowired(required = false)
  private AllowSaleRuleRelateListRepository allowSaleRuleRelateListRepository;

  @Autowired(required = false)
  private AllowSaleListService allowSaleListService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void createByRule(RelateRule rule, List<AllowSaleRuleRelateList> list) {
    boolean f =
        Objects.nonNull(rule)
            && StringUtils.isNoneBlank(rule.getRuleCode(), rule.getRuleType(), rule.getListType());
    Validate.isTrue(f, "规则编码、规则类型、清单类型不能为空");
    this.deleteByRule(rule);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    for (AllowSaleRuleRelateList item : list) {
      item.setRuleCode(rule.getRuleCode());
      item.setRuleType(rule.getRuleType());
      item.setListType(rule.getListType());
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    List<AllowSaleRuleRelateList> data = this.distinctData(list);
    // 注：需要先执行该操作，才能去调用清单接口
    this.allowSaleRuleRelateListRepository.saveBatch(data);
    // 刷新可购清单数据
    List<AllowSaleList> saleList =
        (List<AllowSaleList>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                data,
                AllowSaleRuleRelateList.class,
                AllowSaleList.class,
                HashSet.class,
                ArrayList.class);
    this.allowSaleListService.createByRule(rule, saleList);
  }

  @Override
  @Transactional
  public void deleteByRule(RelateRule rule) {
    Validate.isTrue(
        Objects.nonNull(rule) && StringUtils.isNoneBlank(rule.getRuleCode(), rule.getRuleType()),
        "规则编码及规则类型不能为空");
    this.allowSaleListService.updateAllowNumByRule(rule, -1);
    this.allowSaleRuleRelateListRepository.deleteByRuleCode(rule.getRuleCode());
  }

  @Override
  @Transactional
  public void createByList(List<AllowSaleRuleRelateList> list) {
    // 根据subItemKeys删除关联数据
    // 新增关联数据
    // 根据itemKeys全量刷新清单最终数据
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    for (AllowSaleRuleRelateList item : list) {
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    final List<AllowSaleRuleRelateList> data = this.distinctData(list);
    Set<String> itemKeys =
        data.stream()
            .filter(a -> StringUtils.isNotBlank(a.getItemKey()))
            .map(AllowSaleRuleRelateList::getItemKey)
            .collect(Collectors.toSet());
    Set<String> subItemKeys =
        data.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSubItemKey()))
            .map(AllowSaleRuleRelateList::getSubItemKey)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(subItemKeys) || CollectionUtils.isEmpty(itemKeys)) {
      return;
    }
    this.allowSaleRuleRelateListRepository.deleteBySubItemKeys(subItemKeys);
    // 需要先保存关系数据再执行刷新清单操作
    this.allowSaleRuleRelateListRepository.saveBatch(data);
    this.allowSaleListService.createByItemKeys(itemKeys);
  }

  @Override
  @Transactional
  public void deleteByList(List<AllowSaleRuleRelateList> list) {
    // 根据subItemKeys删除关联数据
    // 根据itemKeys刷新清单最终数据
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    for (AllowSaleRuleRelateList item : list) {
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    final List<AllowSaleRuleRelateList> data = this.distinctData(list);
    Set<String> subItemKeys =
        data.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSubItemKey()))
            .map(AllowSaleRuleRelateList::getSubItemKey)
            .collect(Collectors.toSet());
    if (!CollectionUtils.isEmpty(subItemKeys)) {
      this.allowSaleRuleRelateListRepository.deleteBySubItemKeys(subItemKeys);
    }
    Set<String> itemKeys =
        data.stream()
            .filter(a -> StringUtils.isNotBlank(a.getItemKey()))
            .map(AllowSaleRuleRelateList::getItemKey)
            .collect(Collectors.toSet());
    this.allowSaleListService.createByItemKeys(itemKeys);
  }

  /**
   * 去重数据
   *
   * @param list
   * @return
   */
  private List<AllowSaleRuleRelateList> distinctData(List<AllowSaleRuleRelateList> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return list.stream().distinct().collect(Collectors.toList());
  }
}
