package com.biz.crm.dms.business.allow.sale.local.list.service.notifier;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.product.entity.AllowSaleRuleProduct;
import com.biz.crm.dms.business.allow.sale.local.rule.entity.AllowSaleRule;
import com.biz.crm.dms.business.allow.sale.local.rule.service.AllowSaleRuleService;
import com.biz.crm.dms.business.allow.sale.sdk.constant.AllowSaleRuleConstant;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.customer.CustomerDimensionItemVo;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.customer.CustomerDimensionVo;
import com.biz.crm.dms.business.allow.sale.sdk.element.vo.AllowSaleElementDataVo;
import com.biz.crm.dms.business.allow.sale.sdk.element.vo.AllowSaleElementVo;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleDimensionTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleElementEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleCreateTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleProductRelateTypeEnums;
import com.biz.crm.dms.business.contract.sdk.enums.ContractCancellationState;
import com.biz.crm.dms.business.contract.sdk.enums.EffectiveStatusEnum;
import com.biz.crm.dms.business.contract.sdk.event.contract.ContractEventListener;
import com.biz.crm.dms.business.contract.sdk.event.contractsign.ContractSignEventListener;
import com.biz.crm.dms.business.contract.sdk.vo.contract.ContractVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/**
 * 合同关联允销监听实现
 *
 * @author sunx
 * @date 2022/5/19
 */
@Slf4j
@Component
public class AllowContractSignEventListenerImpl
    implements ContractSignEventListener, ContractEventListener {

  @Autowired(required = false)
  private AllowSaleRuleService allowSaleRuleService;

  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  /** 合同要素编码-可购 */
  private static final String CONTRACT_ELEMENT_CODE = "allowSaleElementRegister";

  @Override
  @Transactional
  public void onSignContract(ContractVo vo) {
    if (Objects.isNull(vo)
        || StringUtils.isBlank(vo.getContractCode())
        || Objects.isNull(vo.getElementDataMap())) {
      return;
    }
    final String contractCode = vo.getContractCode();
    final String contractName = vo.getContractName();
    final String cusCode = vo.getCusCode();
    final String effectiveStatus = vo.getEffectiveStatus();
    final String enableStatus =
        EffectiveStatusEnum.EFFECTIVE_ACTIVE.getDictCode().equals(effectiveStatus)
            ? EnableStatusEnum.ENABLE.getCode()
            : EnableStatusEnum.DISABLE.getCode();

    final Map<String, List<AllowSaleElementVo>> map = this.findAllowSaleElementMap(vo);
    if (map == null || map.isEmpty()) {
      return;
    }
    List<AllowSaleElementVo> productList = map.get(AllowSaleElementEnums.PRODUCT.getCode());
    List<AllowSaleElementVo> productLevelList =
        map.get(AllowSaleElementEnums.PRODUCT_LEVEL.getCode());
    Set<String> productCodeSet = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(productList)) {
      productCodeSet =
          productList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
              .map(AllowSaleElementVo::getProductCode)
              .collect(Collectors.toSet());
    }
    Set<String> productLevelCodeSet = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(productLevelList)) {
      productLevelCodeSet =
          productLevelList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getProductLevelCode()))
              .map(AllowSaleElementVo::getProductLevelCode)
              .collect(Collectors.toSet());
    }
    this.createRule(
        contractCode, contractName, cusCode, enableStatus, productCodeSet, productLevelCodeSet);
  }

  @Override
  @Transactional
  public void onCancellation(ContractVo vo) {
    if (Objects.isNull(vo)) {
      return;
    }
    final String contractCode = vo.getContractCode();
    if (StringUtils.isBlank(contractCode)) {
      return;
    }
    String enableStatus = EnableStatusEnum.DISABLE.getCode();
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      this.allowSaleRuleService.updateByContractCodesAndEnableStatus(
          Lists.newArrayList(contractCode), enableStatus);
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  @Override
  @Transactional
  public void onEnable(List<ContractVo> list) {
    this.enableAndDisableRule(list, EnableStatusEnum.ENABLE.getCode());
  }

  @Override
  @Transactional
  public void onDisable(List<ContractVo> list) {
    this.enableAndDisableRule(list, EnableStatusEnum.DISABLE.getCode());
  }

  @Override
  @Transactional
  public void onMaturity(List<ContractVo> list) {
    this.enableAndDisableRule(list, EnableStatusEnum.DISABLE.getCode());
  }

  @Override
  @Transactional
  public void onEffective(List<ContractVo> list) {
    this.enableAndDisableRule(list, EnableStatusEnum.ENABLE.getCode());
  }

  /**
   * 根据合同创建允销规则
   *
   * @param contractCode
   * @param contractName
   * @param cusCode
   * @param enableStatus
   * @param productCodeSet
   * @param productLevelCodeSet
   */
  private void createRule(
      String contractCode,
      String contractName,
      String cusCode,
      String enableStatus,
      Set<String> productCodeSet,
      Set<String> productLevelCodeSet) {

    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      final AllowSaleRule allowSaleRule = new AllowSaleRule();
      allowSaleRule.setRuleName("合同-" + contractName);
      allowSaleRule.setDimensionType(AllowSaleDimensionTypeEnums.CUSTOMER.getCode());
      allowSaleRule.setCreateType(AllowSaleRuleCreateTypeEnums.CONTRACT.getCode());
      allowSaleRule.setRelateCode(contractCode);
      final CustomerDimensionVo customerDimensionVo = new CustomerDimensionVo();
      final CustomerDimensionItemVo customerDimensionItemVo = new CustomerDimensionItemVo();
      customerDimensionItemVo.setCustomerCode(cusCode);
      customerDimensionVo.setList(Lists.newArrayList(customerDimensionItemVo));
      allowSaleRule.setDimensionJson((JSONObject) JSON.toJSON(customerDimensionVo));
      enableStatus = Optional.ofNullable(enableStatus).orElse(EnableStatusEnum.DISABLE.getCode());
      allowSaleRule.setEnableStatus(enableStatus);
      List<AllowSaleRuleProduct> productList = Lists.newArrayList();
      if (!CollectionUtils.isEmpty(productCodeSet)) {
        for (String item : productCodeSet) {
          final AllowSaleRuleProduct cur = new AllowSaleRuleProduct();
          cur.setRelateCode(item);
          cur.setRelateType(AllowSaleRuleProductRelateTypeEnums.PRODUCT.getCode());
          productList.add(cur);
        }
      }
      if (!CollectionUtils.isEmpty(productLevelCodeSet)) {
        for (String item : productLevelCodeSet) {
          final AllowSaleRuleProduct cur = new AllowSaleRuleProduct();
          cur.setRelateCode(item);
          cur.setRelateType(AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL.getCode());
          productList.add(cur);
        }
      }
      allowSaleRule.setProductList(productList);
      if (CollectionUtils.isEmpty(allowSaleRule.getProductList())) {
        log.error("商品信息为空，不能创建规则");
        return;
      }
      this.allowSaleRuleService.create(allowSaleRule);
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  /**
   * 根据合同更新可购规则状态
   *
   * @param list
   * @param enableStatus
   */
  private void enableAndDisableRule(List<ContractVo> list, String enableStatus) {
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      String newEnableStatus =
          EnableStatusEnum.ENABLE.getCode().equals(enableStatus)
              ? EnableStatusEnum.DISABLE.getCode()
              : EnableStatusEnum.ENABLE.getCode();
      if (CollectionUtils.isEmpty(list)) {
        return;
      }
      Set<String> contractCodes =
          list.stream()
              .filter(
                  a ->
                      StringUtils.isNotBlank(a.getContractCode())
                          && EffectiveStatusEnum.EFFECTIVE_ACTIVE
                              .getKey()
                              .equals(a.getEffectiveStatus())
                          && ContractCancellationState.CANCELLATION
                              .getKey()
                              .equals(a.getCancellationState()))
              .map(ContractVo::getContractCode)
              .collect(Collectors.toSet());

      if (EnableStatusEnum.DISABLE.getCode().equals(enableStatus)) {
        contractCodes =
            list.stream()
                .filter(a -> StringUtils.isNotBlank(a.getContractCode()))
                .map(ContractVo::getContractCode)
                .collect(Collectors.toSet());
      }

      if (CollectionUtils.isEmpty(contractCodes)) {
        return;
      }
      final List<AllowSaleRule> ruleList =
          this.allowSaleRuleService.findByContractCodes(contractCodes);
      if (!CollectionUtils.isEmpty(ruleList)) {
        final Set<String> ids =
            ruleList.stream()
                .filter(a -> newEnableStatus.equals(a.getEnableStatus()))
                .map(AllowSaleRule::getId)
                .collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(ids)) {
          return;
        }
        if (EnableStatusEnum.ENABLE.getCode().equals(enableStatus)) {
          this.allowSaleRuleService.updateByContractCodesAndEnableStatus(Lists.newArrayList(ids),enableStatus);
        } else {
          this.allowSaleRuleService.updateByContractCodesAndEnableStatus(Lists.newArrayList(ids),enableStatus);
        }
      }
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  /**
   * 获取合同允销商品信息
   *
   * @param vo
   * @return
   */
  private Map<String, List<AllowSaleElementVo>> findAllowSaleElementMap(ContractVo vo) {
    if (Objects.isNull(vo)
        || StringUtils.isBlank(vo.getContractCode())
        || Objects.isNull(vo.getElementDataMap())) {
      return Maps.newHashMap();
    }
    final JSONObject jsonObject = vo.getElementDataMap().get(CONTRACT_ELEMENT_CODE);
    if (Objects.isNull(jsonObject)) {
      return Maps.newHashMap();
    }
    AllowSaleElementDataVo allowSaleElementDataVo =
        JSON.parseObject(jsonObject.toJSONString(), AllowSaleElementDataVo.class);
    if (Objects.isNull(allowSaleElementDataVo)
        || Objects.isNull(allowSaleElementDataVo.getAllowSaleMap())
        || allowSaleElementDataVo.getAllowSaleMap().isEmpty()) {
      return Maps.newHashMap();
    }
    return allowSaleElementDataVo.getAllowSaleMap();
  }
}
