package com.biz.crm.dms.business.allow.sale.local.list.service.notifier;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.list.model.AllowSaleRuleProductTrigger;
import com.biz.crm.dms.business.allow.sale.local.rule.service.internal.AllowSaleRuleListAdapter;
import com.biz.crm.mdm.business.product.sdk.dto.ProductEventDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductSingleEventDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.event.ProductEventListener;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/**
 * 商品事件实现
 *
 * @author sunx
 * @date 2022/5/25
 */
@Component
public class AllowProductEventListenerImpl implements ProductEventListener {
  @Autowired(required = false)
  private AllowSaleRuleListAdapter allowSaleRuleListAdapter;

  @Override
  @Transactional
  public void onCreate(ProductEventDto productEventDto) {
    final ProductVo productVo = productEventDto.getNewest();
    Validate.notNull(productVo, "商品信息不能为空");
    if (EnableStatusEnum.DISABLE.getCode().equals(productVo.getEnableStatus())
        || IsShelfEnum.DOWN.getCode().equals(productVo.getIsShelf())) {
      return;
    }
    this.handlerAllowSale(
        productVo.getProductCode(), productVo.getProductLevelCode(), Boolean.TRUE);
  }

  @Override
  @Transactional
  public void onUpdate(ProductEventDto productEventDto) {
    final ProductVo newest = productEventDto.getNewest();
    final ProductVo original = productEventDto.getOriginal();
    Validate.notNull(newest, "商品信息不能为空");
    Validate.notNull(original, "商品信息不能为空");
    //如果上下架状态，起禁用状态，产品层级没有改变，则不刷新允销
    if (newest.getEnableStatus().equals(original.getEnableStatus())
        && newest.getIsShelf().equals(original.getIsShelf())
        && newest.getProductLevelCode().equals(original.getProductCode())
    ){
      return;
    }
    String productCode = original.getProductCode();
    this.handlerAllowSale(productCode, original.getProductLevelCode(), Boolean.FALSE);
    this.handlerAllowSale(productCode, newest.getProductLevelCode(), Boolean.TRUE);
  }

  @Override
  @Transactional
  public void onDelete(ProductEventDto productEventDto) {
    final ProductVo original = productEventDto.getOriginal();
    Validate.notNull(original, "商品信息不能为空");
    if (EnableStatusEnum.DISABLE.getCode().equals(original.getEnableStatus())
        || IsShelfEnum.DOWN.getCode().equals(original.getIsShelf())) {
      return;
    }
    this.handlerAllowSale(original.getProductCode(), original.getProductLevelCode(), Boolean.FALSE);
  }

  @Override
  @Transactional
  public void onEnable(ProductSingleEventDto productSingleEventDto) {
    final JSONObject o = (JSONObject) JSON.toJSON(productSingleEventDto.getNewest());
    Validate.notNull(o, "商品信息不能为空");
    Object productCode = o.get("productCode");
    Validate.notNull(productCode, "商品编码不能为空");
    Object isShelf = o.get("isShelf");
    Validate.notNull(isShelf, "商品上下架状态不能为空");
    if (IsShelfEnum.DOWN.getCode().equals(isShelf.toString())) {
      return;
    }
    String productLevelCode =
        Optional.ofNullable(o.get("productLevelCode")).orElse(StringUtils.EMPTY).toString();
    this.handlerAllowSale(productCode.toString(), productLevelCode, Boolean.TRUE);
  }

  @Override
  @Transactional
  public void onDisable(ProductSingleEventDto productSingleEventDto) {
    final JSONObject o = (JSONObject) JSON.toJSON(productSingleEventDto.getNewest());
    Validate.notNull(o, "商品信息不能为空");
    Object productCode = o.get("productCode");
    Validate.notNull(productCode, "商品编码不能为空");
    Object isShelf = o.get("isShelf");
    Validate.notNull(isShelf, "商品上下架状态不能为空");
    String productLevelCode =
        Optional.ofNullable(o.get("productLevelCode")).orElse(StringUtils.EMPTY).toString();
    if (IsShelfEnum.DOWN.getCode().equals(isShelf.toString())) {
      return;
    }
    this.handlerAllowSale(productCode.toString(), productLevelCode, Boolean.FALSE);
  }

  @Override
  @Transactional
  public void onUpShelf(ProductSingleEventDto productSingleEventDto) {
    final JSONObject o = (JSONObject) JSON.toJSON(productSingleEventDto.getNewest());
    Validate.notNull(o, "商品信息不能为空");
    Object productCode = o.get("productCode");
    Validate.notNull(productCode, "商品编码不能为空");
    String productLevelCode =
        Optional.ofNullable(o.get("productLevelCode")).orElse(StringUtils.EMPTY).toString();
    Object enableStatus = o.get("enableStatus");
    Validate.notNull(enableStatus, "商品启禁用状态不能为空");
    if (EnableStatusEnum.DISABLE.getCode().equals(enableStatus.toString())) {
      return;
    }
    this.handlerAllowSale(productCode.toString(), productLevelCode, Boolean.TRUE);
  }

  @Override
  @Transactional
  public void onDownShelf(ProductSingleEventDto productSingleEventDto) {
    final JSONObject o = (JSONObject) JSON.toJSON(productSingleEventDto.getNewest());
    Validate.notNull(o, "商品信息不能为空");
    Object productCode = o.get("productCode");
    Validate.notNull(productCode, "商品编码不能为空");
    String productLevelCode =
        Optional.ofNullable(o.get("productLevelCode")).orElse(StringUtils.EMPTY).toString();
    Object enableStatus = o.get("enableStatus");
    Validate.notNull(enableStatus, "商品启禁用状态不能为空");
    if (EnableStatusEnum.DISABLE.getCode().equals(enableStatus.toString())) {
      return;
    }
    this.handlerAllowSale(productCode.toString(), productLevelCode, Boolean.FALSE);
  }

  /**
   * 根据商品信息操作可购信息
   *
   * @param productCode
   * @param productLevelCode
   * @param createFlag
   */
  private void handlerAllowSale(String productCode, String productLevelCode, Boolean createFlag) {
    if (StringUtils.isBlank(productCode)) {
      return;
    }
    final AllowSaleRuleProductTrigger product = new AllowSaleRuleProductTrigger();
    product.setProductCode(productCode);
    product.setProductLevelCode(productLevelCode);
    if (Boolean.TRUE.equals(createFlag)) {
      this.allowSaleRuleListAdapter.createByProduct(product);
    } else {
      this.allowSaleRuleListAdapter.deleteByProduct(product);
    }
  }
}
