package com.biz.crm.dms.business.allow.sale.local.list.service.notifier;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.dimension.service.DimensionService;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleRuleRelateList;
import com.biz.crm.dms.business.allow.sale.local.list.model.RelateRule;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleRuleRelateListService;
import com.biz.crm.dms.business.allow.sale.local.product.service.AllowSaleRuleProductService;
import com.biz.crm.dms.business.allow.sale.local.rule.entity.AllowSaleRule;
import com.biz.crm.dms.business.allow.sale.local.rule.service.AllowSaleRuleService;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelEventDto;
import com.biz.crm.mdm.business.product.level.sdk.event.ProductLevelEventListener;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 允销实现商品层级更新事件
 *
 * @author sunx
 * @date 2022/7/4
 */
@Component
public class AllowProductLevelEventListenerImpl implements ProductLevelEventListener {

  @Autowired(required = false)
  private DimensionService dimensionService;

  @Autowired(required = false)
  private AllowSaleRuleProductService allowSaleRuleProductService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private AllowSaleRuleService allowSaleRuleService;

  @Autowired(required = false)
  private AllowSaleRuleRelateListService allowSaleRuleRelateListService;

  @Override
  public void onBatchCreate(List<ProductLevelVo> productLevelVos) {}

  @Override
  public void onBatchUpdate(List<ProductLevelVo> productLevelVos) {}

  @Override
  public void onBatchDelete(List<ProductLevelVo> productLevelVos) {}

  @Override
  public void onBatchEnable(List<ProductLevelVo> productLevelVos) {}

  @Override
  public void onBatchDisable(List<ProductLevelVo> eventDtos) {}

  @Override
  @Transactional
  public void onUpdate(ProductLevelEventDto dto) {
    // 获取修改前的父级层级编码关联的规则编码集合
    // 获取修改后的父级层级编码关联的规则编码集合
    // 找出需要删除、或新增数据的规则集合
    // 获取影响到的规则基本信息
    // 获取当前层级及其下级所有的上架且启用是商品信息
    // 组合信息，更新清单数据
    if (Objects.isNull(dto)
        || dto.getParentCode().equals(dto.getOldParentCode())
        || StringUtils.isBlank(dto.getProductLevelCode())) {
      return;
    }
    // 产品层级历史关联的可购规则编码集合
    Set<String> oldAllowRuleCodes = Sets.newHashSet();
    // 改变之后关联的可购规则编码集合
    Set<String> allowRuleCodes = Sets.newHashSet();
    if (!StringUtils.isBlank(dto.getOldParentCode())) {
      final Map<String, Set<String>> map =
          this.allowSaleRuleProductService.findProductLevelRelateRuleCodesMapByProductLevelCodes(
              Lists.newArrayList(dto.getOldParentCode()));
      oldAllowRuleCodes = map.getOrDefault(dto.getOldParentCode(), Sets.newHashSet());
    }

    if (!StringUtils.isBlank(dto.getParentCode())) {
      final Map<String, Set<String>> map =
          this.allowSaleRuleProductService.findProductLevelRelateRuleCodesMapByProductLevelCodes(
              Lists.newArrayList(dto.getParentCode()));
      allowRuleCodes = map.getOrDefault(dto.getParentCode(), Sets.newHashSet());
    }
    if (CollectionUtils.isEmpty(oldAllowRuleCodes) && CollectionUtils.isEmpty(allowRuleCodes)) {
      return;
    }

    Set<String> delRuleCodeSet =
        Sets.newHashSet(Sets.difference(oldAllowRuleCodes, allowRuleCodes));
    Set<String> addRuleCodeSet =
        Sets.newHashSet(Sets.difference(allowRuleCodes, oldAllowRuleCodes));
    Set<String> allRuleCodeSet = Sets.newHashSet();
    allRuleCodeSet.addAll(addRuleCodeSet);
    allRuleCodeSet.addAll(delRuleCodeSet);
    if (CollectionUtils.isEmpty(allRuleCodeSet)) {
      return;
    }
    final List<AllowSaleRule> ruleList =
        this.allowSaleRuleService.findByCodesAndTenantCode(
            Lists.newArrayList(allRuleCodeSet), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(ruleList)) {
      return;
    }
    // 未删除且已启用的可购规则
    Map<String, AllowSaleRule> mapRule =
        ruleList.stream()
            .filter(
                a ->
                    DelFlagStatusEnum.NORMAL.getCode().equals(a.getDelFlag())
                        && EnableStatusEnum.ENABLE.getCode().equals(a.getEnableStatus()))
            .collect(
                Collectors.toMap(AllowSaleRule::getRuleCode, Function.identity(), (a, b) -> a));
    // 获取当前层级及下级的所有上架且启用的商品
    // k-产品编码，v-关联层级对应的降维码集合
    final Map<String, String> map =
        this.productVoService.findAllowSaleProductByProductLevelCodes(
            Sets.newHashSet(dto.getProductLevelCode()));
    if (map.isEmpty()) {
      return;
    }
    Set<String> productCodeSet = map.keySet();
    this.handlerAllowSaleList(delRuleCodeSet, addRuleCodeSet, mapRule, productCodeSet);
  }

  /**
   * 同步影响的可购清单数据
   *
   * @param delRuleCodeSet
   * @param addRuleCodeSet
   * @param mapRule
   * @param productCodeSet
   */
  private void handlerAllowSaleList(
      Set<String> delRuleCodeSet,
      Set<String> addRuleCodeSet,
      Map<String, AllowSaleRule> mapRule,
      Set<String> productCodeSet) {
    if (mapRule.isEmpty() || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    String tenantCode = TenantUtils.getTenantCode();
    for (Entry<String, AllowSaleRule> item : mapRule.entrySet()) {
      String ruleCode = item.getKey();
      AllowSaleRule rule = item.getValue();
      List<AllowSaleRuleRelateList> list =
          this.findItemRelateList(rule, productCodeSet, tenantCode);
      if (CollectionUtils.isEmpty(list)) {
        continue;
      }
      if (delRuleCodeSet.contains(ruleCode)) {
        this.allowSaleRuleRelateListService.deleteByList(list);
      } else if (addRuleCodeSet.contains(ruleCode)) {
        this.allowSaleRuleRelateListService.createByList(list);
      }
    }
  }

  /**
   * 获取可购清单组合数据
   *
   * @param rule
   * @param productCodeSet
   * @param tenantCode
   * @return
   */
  private List<AllowSaleRuleRelateList> findItemRelateList(
      AllowSaleRule rule, Set<String> productCodeSet, String tenantCode) {
    Map<String, Set<String>> businessCodeMap =
        this.dimensionService.findRuleRelateBusinessCodesMapByRuleCodes(
            Lists.newArrayList(rule.getRuleCode()), rule.getDimensionType());
    if (businessCodeMap.isEmpty()) {
      return Lists.newLinkedList();
    }
    Set<String> businessCodeSet =
        businessCodeMap.getOrDefault(rule.getRuleCode(), Sets.newHashSet());
    final String listType = this.dimensionService.getListType(rule.getDimensionType());
    List<AllowSaleRuleRelateList> list = Lists.newArrayList();
    for (String businessCode : businessCodeSet) {
      for (String productCode : productCodeSet) {
        final AllowSaleRuleRelateList cur = new AllowSaleRuleRelateList();
        cur.setTenantCode(tenantCode);
        cur.setRuleType(rule.getRuleType());
        cur.setRuleCode(rule.getRuleCode());
        cur.setBusinessCode(businessCode);
        cur.setProductCode(productCode);
        cur.setListType(listType);
        cur.setItemKey(
            StringUtils.joinWith(
                ":", cur.getListType(), cur.getBusinessCode(), cur.getProductCode()));
        cur.setSubItemKey(
            StringUtils.joinWith(
                ":",
                cur.getRuleCode(),
                cur.getListType(),
                cur.getBusinessCode(),
                cur.getProductCode()));
        final RelateRule relateRule = new RelateRule();
        relateRule.setRuleType(cur.getRuleType());
        relateRule.setRuleCode(cur.getRuleCode());
        relateRule.setListType(cur.getListType());
        cur.setRule(relateRule);
        list.add(cur);
      }
    }
    return list;
  }
}
