package com.biz.crm.dms.business.allow.sale.local.rule.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.allow.sale.local.rule.entity.AllowSaleRule;
import com.biz.crm.dms.business.allow.sale.local.rule.service.AllowSaleRuleService;
import com.biz.crm.dms.business.allow.sale.sdk.constant.AllowSaleRuleConstant;
import com.biz.crm.dms.business.allow.sale.sdk.rule.dto.AllowSaleRulePaginationDto;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 允销规则主细信息(AllowSaleRule)表控制层
 *
 * @author sunx
 * @date 2021-11-19 15:16:26
 */
@Slf4j
@Api(tags = "允销规则主细信息: AllowSaleRule: 允销规则信息")
@RestController
@RequestMapping(value = {"/v1/allowSaleRule/allowSaleRule"})
public class AllowSaleRuleController {

  @Autowired(required = false)
  private AllowSaleRuleService allowSaleRuleService;

  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  /**
   * 查询分页列表
   *
   * @param pageable 分页
   * @param paginationDto 查询参数
   * @return
   */
  @ApiOperation(value = "查询分页列表")
  @GetMapping(value = {"/findByConditions"})
  public Result<Page<AllowSaleRule>> findByConditions(
      @PageableDefault(50) Pageable pageable, AllowSaleRulePaginationDto paginationDto) {
    try {
      Page<AllowSaleRule> result =
          this.allowSaleRuleService.findByConditions(pageable, paginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 详情
   *
   * @param id 主键
   * @return
   */
  @ApiOperation(value = "详情")
  @GetMapping(value = {"/findDetailById"})
  public Result<AllowSaleRule> findDetailById(@RequestParam("id") String id) {
    try {
      AllowSaleRule allowSaleRule = this.allowSaleRuleService.findDetailById(id);
      return Result.ok(allowSaleRule);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建
   *
   * @param allowSaleRule 允销规则
   * @return
   */
  @ApiOperation(value = "创建")
  @PostMapping(value = "")
  public Result<AllowSaleRule> create(@RequestBody AllowSaleRule allowSaleRule) {
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      AllowSaleRule current = this.allowSaleRuleService.create(allowSaleRule);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  /**
   * 创建
   *
   * @param allowSaleRule 允销规则
   * @return
   */
  @ApiOperation(value = "更新")
  @PatchMapping(value = "")
  public Result<AllowSaleRule> update(@RequestBody AllowSaleRule allowSaleRule) {
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      AllowSaleRule current = this.allowSaleRuleService.update(allowSaleRule);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  /**
   * 逻辑删除
   *
   * @param ids 主键
   * @return
   */
  @ApiOperation(value = "逻辑删除")
  @DeleteMapping("/delete")
  public Result<?> delete(@RequestParam("ids") List<String> ids) {
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      this.allowSaleRuleService.updateDelFlagByIds(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  /**
   * 启用
   *
   * @param ids 主键
   * @return
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      this.allowSaleRuleService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }

  /**
   * 禁用
   *
   * @param ids 主键
   * @return
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    Validate.isTrue(
        this.redisMutexService.tryLock(AllowSaleRuleConstant.RULE_LOCK_KEY, TimeUnit.SECONDS, 5),
        "存在进行中的规则处理，请稍后重试");
    try {
      this.allowSaleRuleService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(AllowSaleRuleConstant.RULE_LOCK_KEY);
    }
  }
}
