package com.biz.crm.dms.business.allow.sale.local.rule.service.internal;

import com.biz.crm.dms.business.allow.sale.local.dimension.service.DimensionService;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleRuleRelateList;
import com.biz.crm.dms.business.allow.sale.local.list.model.AllowSaleRuleBusinessTrigger;
import com.biz.crm.dms.business.allow.sale.local.list.model.AllowSaleRuleProductTrigger;
import com.biz.crm.dms.business.allow.sale.local.list.model.RelateRule;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleRuleRelateListService;
import com.biz.crm.dms.business.allow.sale.local.product.service.AllowSaleRuleProductService;
import com.biz.crm.dms.business.allow.sale.local.rule.entity.AllowSaleRule;
import com.biz.crm.dms.business.allow.sale.local.rule.service.AllowSaleRuleService;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleDimensionTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleTypeEnums;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/**
 * 可购规则生成可购清单适配器bean
 *
 * @author sunx
 * @date 2022/6/25
 */
@Slf4j
@Component
public class AllowSaleRuleListAdapter {

  @Autowired(required = false)
  private DimensionService dimensionService;

  @Autowired(required = false)
  private AllowSaleRuleProductService allowSaleRuleProductService;

  @Autowired(required = false)
  private AllowSaleRuleRelateListService allowSaleRuleRelateListService;

  @Autowired(required = false)
  private AllowSaleRuleService allowSaleRuleService;

  /**
   * 根据可购规则生成可购清单数据
   *
   * @param allowSaleRule
   */
  @Transactional
  public void createByRule(AllowSaleRule allowSaleRule) {
    this.handlerCreateByRule(allowSaleRule);
  }

  /**
   * 根据可购规则删除可购清单数据
   *
   * @param allowSaleRule
   */
  @Transactional
  public void deleteByRule(AllowSaleRule allowSaleRule) {
    Validate.notNull(allowSaleRule, "规则数据不能为空");
    Validate.notBlank(allowSaleRule.getRuleCode(), "规则编码不能为空");
    Validate.notBlank(allowSaleRule.getDimensionType(), "规则维度不能为空");
    String listType = this.dimensionService.getListType(allowSaleRule.getDimensionType());
    String ruleType = allowSaleRule.getRuleType();
    if (StringUtils.isBlank(ruleType)) {
      ruleType = AllowSaleRuleTypeEnums.ALLOW.getDictCode();
    }
    final RelateRule relateRule = new RelateRule();
    relateRule.setListType(listType);
    relateRule.setRuleCode(allowSaleRule.getRuleCode());
    relateRule.setRuleType(ruleType);
    this.allowSaleRuleRelateListService.deleteByRule(relateRule);
  }

  /**
   * 商品变更致使清单执行新增操作
   *
   * @param product
   */
  @Transactional
  public void createByProduct(AllowSaleRuleProductTrigger product) {
    this.validateProduct(product);
    final List<AllowSaleRuleRelateList> list = this.findByProduct(product);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    this.allowSaleRuleRelateListService.createByList(list);
  }

  /**
   * 商品变更致使清单执行清理操作
   *
   * @param product
   */
  @Transactional
  public void deleteByProduct(AllowSaleRuleProductTrigger product) {
    this.validateProduct(product);
    final List<AllowSaleRuleRelateList> list = this.findByProduct(product);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    this.allowSaleRuleRelateListService.deleteByList(list);
  }

  /**
   * 业务维度变更致使清单执行新增操作
   *
   * @param business
   */
  @Transactional
  public void createByBusiness(AllowSaleRuleBusinessTrigger business) {
    this.validateBusiness(business);
    final List<AllowSaleRuleRelateList> list = this.findByBusiness(business);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    this.allowSaleRuleRelateListService.createByList(list);
  }

  /**
   * 业务维度变更致使清单执行清理操作
   *
   * @param business
   */
  @Transactional
  public void deleteByBusiness(AllowSaleRuleBusinessTrigger business) {
    this.validateBusiness(business);
    final List<AllowSaleRuleRelateList> list = this.findByBusiness(business);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    this.allowSaleRuleRelateListService.deleteByList(list);
  }

  /** 根据规则新增可购清单数据 */
  private void handlerCreateByRule(AllowSaleRule allowSaleRule) {
    // 1. 根据规则编码删除对应的可购清单历史数据
    // 2. 根据规则维度+维度配置信息获取详细信息（终端或客户信息）
    // 3. 根据spu、产品层级、商品编码获取对应的商品的详细信息
    // 4. 构建可购清单明细新，保存可购清单信息
    this.validateUpdateData(allowSaleRule);
    AllowSaleDimensionTypeEnums ruleTypeEnums =
        AllowSaleDimensionTypeEnums.findByCode(allowSaleRule.getDimensionType());
    Validate.notNull(ruleTypeEnums, "非法的规则维度");
    AllowSaleListTypeEnums listTypeEnums =
        AllowSaleListTypeEnums.findByRuleType(allowSaleRule.getDimensionType());
    Validate.notNull(listTypeEnums, "非法的规则维度");

    Map<String, Set<String>> businessCodeMap =
        this.dimensionService.findRuleRelateBusinessCodesMapByRuleCodes(
            Lists.newArrayList(allowSaleRule.getRuleCode()), allowSaleRule.getDimensionType());
    Set<String> businessCodeSet = businessCodeMap.get(allowSaleRule.getRuleCode());
    if (CollectionUtils.isEmpty(businessCodeSet)) {
      log.info("未获取到对应的客户或终端信息，无需更新可购清单");
      // 需要清除历史数据
      this.deleteByRule(allowSaleRule);
      return;
    }

    Map<String, Set<String>> productCodeMap =
        this.allowSaleRuleProductService.findRuleRelateProductCodesMapByRuleCodes(
            Lists.newArrayList(allowSaleRule.getRuleCode()));
    Set<String> productCodeSet = productCodeMap.get(allowSaleRule.getRuleCode());
    if (CollectionUtils.isEmpty(productCodeSet)) {
      log.info("未获取到对应的商品信息，无需更新可购清单");
      // 需要清除历史数据
      this.deleteByRule(allowSaleRule);
      return;
    }
    String listType = this.dimensionService.getListType(allowSaleRule.getDimensionType());
    String ruleCode = allowSaleRule.getRuleCode();
    String ruleType = allowSaleRule.getRuleType();
    if (StringUtils.isBlank(ruleType)) {
      ruleType = AllowSaleRuleTypeEnums.ALLOW.getDictCode();
    }

    final RelateRule relateRule = new RelateRule();
    relateRule.setListType(listType);
    relateRule.setRuleCode(ruleCode);
    relateRule.setRuleType(ruleType);

    List<AllowSaleRuleRelateList> list =
        Lists.newArrayListWithCapacity(businessCodeSet.size() * productCodeSet.size());
    for (String businessCode : businessCodeSet) {
      for (String productCode : productCodeSet) {
        AllowSaleRuleRelateList cur = new AllowSaleRuleRelateList();
        cur.setRuleCode(ruleCode);
        cur.setRuleType(ruleType);
        cur.setTenantCode(TenantUtils.getTenantCode());
        cur.setListType(listType);
        cur.setBusinessCode(businessCode);
        cur.setProductCode(productCode);
        cur.setItemKey(
            StringUtils.joinWith(
                ":", cur.getListType(), cur.getBusinessCode(), cur.getProductCode()));
        cur.setSubItemKey(
            StringUtils.joinWith(
                ":",
                cur.getRuleCode(),
                cur.getListType(),
                cur.getBusinessCode(),
                cur.getProductCode()));
        list.add(cur);
      }
    }
    this.allowSaleRuleRelateListService.createByRule(relateRule, list);
  }

  /**
   * 根据商品信息（编码、层级编码）获取关联明细
   *
   * @param product
   * @return
   */
  private List<AllowSaleRuleRelateList> findByProduct(AllowSaleRuleProductTrigger product) {
    // 根据商品编码或层级编码获取关联的规则
    // 获取规则的基本信息
    // 根据规则获取维度关联的业务编码集合信息（终端、经销商编码集合【businessCode集合】）
    // 构建规则关联清单数据
    String productCode = product.getProductCode();
    String productLevelCode = product.getProductLevelCode();
    Map<String, Set<String>> productCodeRuleMap =
        this.allowSaleRuleProductService.findSkuRelateRuleCodesMapBySkuCodes(
            Lists.newArrayList(productCode));
    Set<String> set1 = productCodeRuleMap.getOrDefault(productCode, Sets.newHashSet());
    Set<String> set2 = Sets.newHashSet();
    if (StringUtils.isNotBlank(productLevelCode)) {
      Map<String, Set<String>> productLevelCodeRuleMap =
          this.allowSaleRuleProductService.findProductLevelRelateRuleCodesMapByProductLevelCodes(
              Lists.newArrayList(productLevelCode));
      set2 = productLevelCodeRuleMap.getOrDefault(productLevelCode, Sets.newHashSet());
    }
    final SetView<String> unionSet = Sets.union(set1, set2);
    if (unionSet.isEmpty()) {
      return Lists.newLinkedList();
    }
    List<AllowSaleRule> allowSaleRuleList =
        this.allowSaleRuleService.findByCodesAndTenantCode(
            Lists.newArrayList(unionSet), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(allowSaleRuleList)) {
      return Lists.newLinkedList();
    }
    Map<String, AllowSaleRule> mapRule =
        allowSaleRuleList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRuleCode()))
            .collect(
                Collectors.toMap(AllowSaleRule::getRuleCode, Function.identity(), (a, b) -> a));
    // k-ruleCode,v-businessCodeSet
    Map<String, Set<String>> mapRuleBusiness = this.findRuleBusiness(allowSaleRuleList);
    if (mapRuleBusiness.isEmpty()) {
      return Lists.newLinkedList();
    }

    String tenantCode = TenantUtils.getTenantCode();
    List<AllowSaleRuleRelateList> list = Lists.newLinkedList();
    for (Entry<String, Set<String>> item : mapRuleBusiness.entrySet()) {
      String ruleCode = item.getKey();
      Set<String> businessCodeSet = item.getValue();
      AllowSaleRule rule = mapRule.get(ruleCode);
      if (StringUtils.isBlank(ruleCode)
          || CollectionUtils.isEmpty(businessCodeSet)
          || Objects.isNull(rule)) {
        continue;
      }
      final String listType = this.dimensionService.getListType(rule.getDimensionType());
      for (String businessCode : businessCodeSet) {
        final AllowSaleRuleRelateList cur = new AllowSaleRuleRelateList();
        cur.setTenantCode(tenantCode);
        cur.setRuleType(rule.getRuleType());
        cur.setRuleCode(rule.getRuleCode());
        cur.setBusinessCode(businessCode);
        cur.setProductCode(productCode);
        cur.setListType(listType);
        cur.setItemKey(
            StringUtils.joinWith(
                ":", cur.getListType(), cur.getBusinessCode(), cur.getProductCode()));
        cur.setSubItemKey(
            StringUtils.joinWith(
                ":",
                cur.getRuleCode(),
                cur.getListType(),
                cur.getBusinessCode(),
                cur.getProductCode()));
        final RelateRule relateRule = new RelateRule();
        relateRule.setRuleType(cur.getRuleType());
        relateRule.setRuleCode(cur.getRuleCode());
        relateRule.setListType(cur.getListType());
        cur.setRule(relateRule);
        list.add(cur);
      }
    }

    return list;
  }

  /**
   * 根据维度信息（1、终端编码|终端对应的组织编码；2、经销商编码|经销商对应的组织编码）获取关联明细
   *
   * @param business
   * @return
   */
  private List<AllowSaleRuleRelateList> findByBusiness(AllowSaleRuleBusinessTrigger business) {
    // 根据维度获取关联的规则
    // 获取规则的基本信息
    // 根据规则获取关联商品编码集合信息
    // 构建规则关联清单数据
    String dimensionType = business.getDimensionType();
    String businessCode = business.getBusinessCode();
    String relateTypeCode = business.getRelateTypeCode();
    String relateDimensionType = business.getRelateDimensionType();

    // 根据经销商编码维度获取关联的规则信息
    final Map<String, Set<String>> map1 =
        this.dimensionService.findRelateRuleCodesMapByRelateCodes(
            Lists.newArrayList(businessCode), dimensionType);
    Set<String> ruleCodeSet = Sets.newHashSet();
    if (!map1.isEmpty()) {
      ruleCodeSet.addAll(map1.getOrDefault(businessCode, Sets.newHashSet()));
    }

    // 根据附加维度获取关联的规则信息
    if (StringUtils.isNoneBlank(relateTypeCode, relateDimensionType)) {
      Map<String, Set<String>> map =
          this.dimensionService.findRelateRuleCodesMapByRelateCodes(
              Lists.newArrayList(relateTypeCode), relateDimensionType);
      if (!map.isEmpty()) {
        ruleCodeSet.addAll(map.getOrDefault(relateTypeCode, Sets.newHashSet()));
      }
    }

    if (CollectionUtils.isEmpty(ruleCodeSet)) {
      return Lists.newLinkedList();
    }
    List<AllowSaleRule> allowSaleRuleList =
        this.allowSaleRuleService.findByCodesAndTenantCode(
            Lists.newArrayList(ruleCodeSet), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(allowSaleRuleList)) {
      return Lists.newLinkedList();
    }
    Map<String, AllowSaleRule> mapRule =
        allowSaleRuleList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRuleCode()))
            .collect(
                Collectors.toMap(AllowSaleRule::getRuleCode, Function.identity(), (a, b) -> a));

    // k-规则编码，v-商品编码集合
    final Map<String, Set<String>> mapRuleProduct =
        this.allowSaleRuleProductService.findRuleRelateProductCodesMapByRuleCodes(
            Lists.newArrayList(ruleCodeSet));
    if (mapRuleProduct.isEmpty()) {
      return Lists.newLinkedList();
    }

    String tenantCode = TenantUtils.getTenantCode();
    List<AllowSaleRuleRelateList> list = Lists.newLinkedList();
    for (Entry<String, Set<String>> item : mapRuleProduct.entrySet()) {
      String ruleCode = item.getKey();
      Set<String> productCodeSet = item.getValue();
      AllowSaleRule rule = mapRule.get(ruleCode);
      if (StringUtils.isBlank(ruleCode)
          || CollectionUtils.isEmpty(productCodeSet)
          || Objects.isNull(rule)) {
        continue;
      }
      final String listType = this.dimensionService.getListType(rule.getDimensionType());
      for (String productCode : productCodeSet) {
        final AllowSaleRuleRelateList cur = new AllowSaleRuleRelateList();
        cur.setTenantCode(tenantCode);
        cur.setRuleType(rule.getRuleType());
        cur.setRuleCode(rule.getRuleCode());
        cur.setBusinessCode(businessCode);
        cur.setProductCode(productCode);
        cur.setListType(listType);
        cur.setItemKey(
            StringUtils.joinWith(
                ":", cur.getListType(), cur.getBusinessCode(), cur.getProductCode()));
        cur.setSubItemKey(
            StringUtils.joinWith(
                ":",
                cur.getRuleCode(),
                cur.getListType(),
                cur.getBusinessCode(),
                cur.getProductCode()));
        final RelateRule relateRule = new RelateRule();
        relateRule.setRuleType(cur.getRuleType());
        relateRule.setRuleCode(cur.getRuleCode());
        relateRule.setListType(cur.getListType());
        cur.setRule(relateRule);
        list.add(cur);
      }
    }

    return list;
  }

  /**
   * 获取规则对应的业务编码集合
   *
   * @param list
   * @return
   */
  private Map<String, Set<String>> findRuleBusiness(List<AllowSaleRule> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    // k-dimensionType,v-ruleCode 集合
    Map<String, Set<String>> dimensionTypeMap =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getDimensionType(), a.getRuleCode()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleRule::getDimensionType,
                    Collectors.mapping(AllowSaleRule::getRuleCode, Collectors.toSet())));

    if (dimensionTypeMap.isEmpty()) {
      return Maps.newHashMap();
    }
    // k-ruleCode,v-businessCodeSet
    Map<String, Set<String>> map = Maps.newHashMap();
    for (Entry<String, Set<String>> item : dimensionTypeMap.entrySet()) {
      final Map<String, Set<String>> curMap =
          this.dimensionService.findRuleRelateBusinessCodesMapByRuleCodes(
              Lists.newArrayList(item.getValue()), item.getKey());
      if (curMap.isEmpty()) {
        continue;
      }
      for (Entry<String, Set<String>> sub : curMap.entrySet()) {
        map.put(sub.getKey(), sub.getValue());
      }
    }
    return map;
  }

  /**
   * 允销规则信息变更时入参数据验证
   *
   * @param allowSaleRule
   */
  private void validateUpdateData(AllowSaleRule allowSaleRule) {
    Validate.notNull(allowSaleRule, "规则数据不能为空");
    Validate.notBlank(allowSaleRule.getRuleCode(), "规则编码不能为空");
    Validate.notBlank(allowSaleRule.getDimensionType(), "规则维度不能为空");
    Validate.notBlank(allowSaleRule.getCreateType(), "创建方式不能为空");
    Validate.notNull(allowSaleRule.getDimensionJson(), "维度配置信息不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(allowSaleRule.getProductList()), "可购商品规则不能为空");
  }

  /**
   * 验证商品参数信息
   *
   * @param product
   */
  private void validateProduct(AllowSaleRuleProductTrigger product) {
    Validate.notNull(product, "商品信息不能为空");
    Validate.notEmpty(product.getProductCode(), "商品编码不能为空");
    if (StringUtils.isBlank(product.getProductLevelCode())) {
      product.setProductLevelCode(StringUtils.EMPTY);
    }
  }

  /**
   * 验证维度参数信息
   *
   * @param business
   */
  private void validateBusiness(AllowSaleRuleBusinessTrigger business) {
    Validate.notNull(business, "维度信息不能为空");
    Validate.notEmpty(business.getDimensionType(), "业务维度类型编码不能为空");
    Validate.notEmpty(business.getBusinessCode(), "维度业务编码不能为空");
    if (StringUtils.isBlank(business.getRelateTypeCode())) {
      business.setRelateTypeCode(StringUtils.EMPTY);
      business.setRelateDimensionType(StringUtils.EMPTY);
    } else {
      Validate.notEmpty(business.getRelateDimensionType(), "关联维度编码不能为空");
    }
  }
}
