package com.biz.crm.dms.business.allow.sale.local.dimension.service.register;

import com.biz.crm.dms.business.allow.sale.local.dimension.detail.entity.AllowSaleRuleDetail;
import com.biz.crm.dms.business.allow.sale.local.dimension.detail.model.DetailRelateVo;
import com.biz.crm.dms.business.allow.sale.local.dimension.detail.service.AllowSaleRuleDetailService;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.Dimension;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.register.AllowSaleRuleDimensionRegister;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.terminal.TerminalDimensionItemVo;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.terminal.TerminalDimensionVo;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleDimensionTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalQueryDto;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 终端维度配置注册器实现
 *
 * @author sunx
 * @date 2021/11/23
 */
@Component
@Slf4j
public class TerminalDimensionVoRegisterImpl
    implements AllowSaleRuleDimensionRegister<TerminalDimensionVo> {

  @Autowired(required = false)
  private AllowSaleRuleDetailService allowSaleRuleDetailService;

  @Autowired(required = false)
  private TerminalVoService terminalVoService;

  @Override
  public String getType() {
    return AllowSaleDimensionTypeEnums.TERMINAL.getCode();
  }

  @Override
  public String getTypeDesc() {
    return AllowSaleDimensionTypeEnums.TERMINAL.getDesc();
  }

  @Override
  public int sort() {
    return 3;
  }

  @Override
  public Class<TerminalDimensionVo> getDimensionClass() {
    return TerminalDimensionVo.class;
  }

  @Override
  public String getListType() {
    return AllowSaleListTypeEnums.TERMINAL.getCode();
  }

  @Override
  public void saveDimensionDetail(Dimension dimension) {
    Validate.notNull(dimension, "客户维度规则信息不能为空");
    TerminalDimensionVo vo = (TerminalDimensionVo) dimension;
    Validate.notBlank(vo.getRuleCode(), "允销规则编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(vo.getList()), "终端维度规则明细不能为空");
    List<AllowSaleRuleDetail> list = Lists.newArrayList();
    vo.getList()
        .forEach(
            a -> {
              AllowSaleRuleDetail cur = new AllowSaleRuleDetail();
              cur.setRelateCode(a.getTerminalCode());
              cur.setRuleCode(vo.getRuleCode());
              list.add(cur);
            });
    allowSaleRuleDetailService.batchSave(list);
  }

  @Override
  public TerminalDimensionVo findByRuleCode(String ruleCode) {
    if (StringUtils.isBlank(ruleCode)) {
      return null;
    }
    List<AllowSaleRuleDetail> list =
        allowSaleRuleDetailService.findByRuleCodes(Lists.newArrayList(ruleCode));
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }

    List<String> terminalCodeList =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRelateCode()))
            .map(AllowSaleRuleDetail::getRelateCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return null;
    }
    List<TerminalVo> terminalVoList =
        terminalVoService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
    if (CollectionUtils.isEmpty(terminalVoList)) {
      return null;
    }
    Map<String, String> terminalNameMap =
        terminalVoList.stream()
            .collect(
                Collectors.toMap(
                    TerminalVo::getTerminalCode, TerminalVo::getTerminalName, (a, b) -> a));

    TerminalDimensionVo vo = new TerminalDimensionVo();
    List<TerminalDimensionItemVo> subList = Lists.newArrayList();
    list.forEach(
        a -> {
          TerminalDimensionItemVo cur = new TerminalDimensionItemVo();
          cur.setTerminalCode(a.getRelateCode());
          cur.setTerminalName(terminalNameMap.getOrDefault(a.getRelateCode(), StringUtils.EMPTY));
          subList.add(cur);
        });
    vo.setList(subList);
    return vo;
  }

  @Override
  public Map<String, Set<String>> findRelateRuleCodesMapByRelateCodes(List<String> relateCodeList) {
    if (CollectionUtils.isEmpty(relateCodeList)) {
      return Maps.newHashMap();
    }
    List<DetailRelateVo> list =
        this.allowSaleRuleDetailService.findByDimensionTypeAndRelateCodes(
            this.getType(), relateCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }

    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
        .collect(
            Collectors.groupingBy(
                DetailRelateVo::getRelateCode,
                Collectors.mapping(DetailRelateVo::getRuleCode, Collectors.toSet())));
  }

  @Override
  public Map<String, Set<String>> findRuleRelateBusinessCodesMapByRuleCodes(
      List<String> ruleCodeList) {
    if (CollectionUtils.isEmpty(ruleCodeList)) {
      return Maps.newHashMap();
    }
    List<AllowSaleRuleDetail> list = this.allowSaleRuleDetailService.findByRuleCodes(ruleCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }

    Set<String> relateCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRelateCode()))
            .map(AllowSaleRuleDetail::getRelateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(relateCodeSet)) {
      return Maps.newHashMap();
    }

    // 获取未删除且已通过的终端信息
    final TerminalQueryDto queryDto = new TerminalQueryDto();
    queryDto.setTerminalCodeSet(relateCodeSet);
    final Set<String> customerCodeSet = this.terminalVoService.findByTerminalQueryDto(queryDto);
    if (CollectionUtils.isEmpty(customerCodeSet)) {
      return Maps.newHashMap();
    }

    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
        .collect(
            Collectors.groupingBy(
                AllowSaleRuleDetail::getRuleCode,
                Collectors.mapping(AllowSaleRuleDetail::getRelateCode, Collectors.toSet())));
  }
}
