package com.biz.crm.dms.business.allow.sale.local.list.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleList;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleRuleRelateList;
import com.biz.crm.dms.business.allow.sale.local.list.repository.AllowSaleRuleRelateListRepository;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleListService;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListPaginationDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListProductPaginationDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.service.AllowSaleListProductVoService;
import com.biz.crm.dms.business.allow.sale.sdk.list.vo.AllowSaleListProductVo;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerQueryDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.price.sdk.dto.FindBusinessPriceDto;
import com.biz.crm.mdm.business.price.sdk.dto.FindBusinessPriceItemDimensionDto;
import com.biz.crm.mdm.business.price.sdk.dto.FindBusinessPriceItemDto;
import com.biz.crm.mdm.business.price.sdk.dto.SearchPriceDimensionItemDto;
import com.biz.crm.mdm.business.price.sdk.enums.PriceDimensionEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.biz.crm.mdm.business.product.sdk.dto.ProductDetailQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.enums.ProductExtInfoTypeEnum;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.product.spu.sdk.service.ProductSpuVoService;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuVo;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalQueryDto;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalRelaOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.StopWatch;

/**
 * 带商品明细的可购清单接口实现
 *
 * @author sunx
 * @date 2022/1/6
 */
@Slf4j
@Service
public class AllowSaleListProductVoServiceImpl implements AllowSaleListProductVoService {

  @Autowired(required = false)
  private AllowSaleListService allowSaleListService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private TerminalVoService terminalVoService;

  @Autowired(required = false)
  private PriceModelVoService priceModelVoService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private ProductSpuVoService productSpuVoService;

  @Autowired(required = false)
  private AllowSaleRuleRelateListRepository allowSaleRuleRelateListRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<AllowSaleListProductVo> findByConditions(
      Pageable pageable, AllowSaleListProductPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    final AllowSaleListPaginationDto dto1 = this.findByAllowSaleListProductPaginationDto(dto);
    if (Objects.isNull(dto1)) {
      return new Page<>();
    }
    Page<AllowSaleList> page = this.allowSaleListService.findByConditions(pageable, dto1);
    if (Objects.isNull(page)) {
      return new Page<>();
    }
    List<AllowSaleList> records = page.getRecords();
    List<AllowSaleListProductVo> list =
        (List<AllowSaleListProductVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                records,
                AllowSaleList.class,
                AllowSaleListProductVo.class,
                HashSet.class,
                ArrayList.class);
    this.handleRecords(list, dto);
    Page<AllowSaleListProductVo> pageResult = new Page<>();
    pageResult.setRecords(list);
    pageResult.setCurrent(pageable.getPageNumber());
    pageResult.setTotal(page.getTotal());
    return pageResult;
  }

  /**
   * 获取允销清单查询参数
   *
   * <p>分别调用商品sdk，终端sdk、经销商sdk获取模糊匹配的业务编码集合
   *
   * <p>当获取到的数量大于500时，取获取到数据的前500个业务编码集合执行后续的范围查询
   *
   * @param dto
   * @return
   */
  private AllowSaleListPaginationDto findByAllowSaleListProductPaginationDto(
      AllowSaleListProductPaginationDto dto) {
    dto = Optional.ofNullable(dto).orElse(new AllowSaleListProductPaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setListType(
        Optional.ofNullable(dto.getListType()).orElse(AllowSaleListTypeEnums.CUSTOMER.getCode()));
    dto.setRuleType(
        Optional.ofNullable(dto.getRuleType()).orElse(AllowSaleRuleTypeEnums.ALLOW.getDictCode()));
    dto.setPriceFlag(Optional.ofNullable(dto.getPriceFlag()).orElse(Boolean.FALSE));
    AllowSaleListPaginationDto dto1 =
        this.nebulaToolkitService.copyObjectByBlankList(
            dto, AllowSaleListPaginationDto.class, HashSet.class, ArrayList.class);

    // 商品查询参数不为空的时候需要通过sdk获取商品编码集合关联查询允销（为保证sql正常，尽量精确查询，如果筛选商品的结果集大于500，取500）
    Boolean f =
        StringUtils.isAllBlank(
            dto.getProductCode(), dto.getProductName(), dto.getProductLevelCode());
    if (Boolean.FALSE.equals(f)) {
      ProductQueryDto queryDto = new ProductQueryDto();
      queryDto.setProductCode(dto.getProductCode());
      queryDto.setProductName(dto.getProductName());
      if (StringUtils.isNotBlank(dto.getProductLevelCode())) {
        queryDto.setProductLevelCodeList(Lists.newArrayList(dto.getProductLevelCode()));
      }
      queryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      queryDto.setIsShelf(IsShelfEnum.UP.getCode());
      queryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      List<ProductVo> productVos = this.productVoService.findByProductQueryDto(queryDto);
      if (CollectionUtils.isEmpty(productVos)) {
        return null;
      }
      List<String> productCodes =
          productVos.stream()
              .map(ProductVo::getProductCode)
              .limit(500)
              .collect(Collectors.toList());
      dto1.setProductCodes(productCodes);
    }

    // 查询业务参数不为空时，通过sdk获取业务编码数据关联查询（经销商、终端）,（为保证sql正常，尽量精确查询，如果筛选商品的结果集大于500，取500）
    if (StringUtils.isNotBlank(dto.getBusinessName())
        || StringUtils.isNotBlank(dto.getBusinessCodeLike())
        || StringUtils.isNotBlank(dto.getChannel())
        || StringUtils.isNotBlank(dto.getOrgName())) {
      if (dto.getListType().equals(AllowSaleListTypeEnums.CUSTOMER.getCode())) {
        final CustomerQueryDto queryDto = new CustomerQueryDto();
        queryDto.setCustomerName(dto.getBusinessName());
        queryDto.setChannel(dto.getChannel());
        queryDto.setOrgName(dto.getOrgName());
        queryDto.setCustomerCode(dto.getBusinessCodeLike());
        final Set<String> set = this.customerVoService.findByCustomerQueryDto(queryDto);
        if (CollectionUtils.isEmpty(set)) {
          return null;
        }
        dto1.setBusinessCodeSet(set.stream().limit(500).collect(Collectors.toSet()));
      } else if (dto.getListType().equals(AllowSaleListTypeEnums.TERMINAL.getCode())) {
        final TerminalQueryDto queryDto = new TerminalQueryDto();
        queryDto.setTerminalName(dto.getBusinessName());
        queryDto.setChannel(dto.getChannel());
        queryDto.setOrgName(dto.getOrgName());
        queryDto.setTerminalCode(dto.getBusinessCodeLike());
        final Set<String> set = this.terminalVoService.findByTerminalQueryDto(queryDto);
        if (CollectionUtils.isEmpty(set)) {
          return null;
        }
        dto1.setBusinessCodeSet(set.stream().limit(500).collect(Collectors.toSet()));
      }
    }
    return dto1;
  }

  /**
   * 处理可购清单明细信息
   *
   * @param list
   * @param dto
   */
  private void handleRecords(
      List<AllowSaleListProductVo> list, AllowSaleListProductPaginationDto dto) {
    String listType = dto.getListType();
    Boolean priceFlag = dto.getPriceFlag();
    Boolean productImageFlag = dto.getProductImageFlag();
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    // k-businessCode,v-Set<productCode>
    StopWatch sw = new StopWatch();
    Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getBusinessCode(), a.getProductCode()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleListProductVo::getBusinessCode,
                    Collectors.mapping(
                        AllowSaleListProductVo::getProductCode, Collectors.toSet())));
    sw.start("1");
    Map<String, CustomerVo> mapCustomer =
        listType.equals(AllowSaleListTypeEnums.CUSTOMER.getCode())
            ? this.findCustomerMap(map.keySet())
            : null;
    Map<String, TerminalVo> mapTerminal =
        listType.equals(AllowSaleListTypeEnums.TERMINAL.getCode())
            ? this.findTerminalMap(map.keySet())
            : null;
    sw.stop();

    sw.start("2");
    // 询价+商品信息+商品spuCode信息（后期优化可以多线程处理）
    Map<String, PriceModelVo> priceModelVoMap =
        Boolean.TRUE.equals(priceFlag)
            ? this.findPriceMap(list, mapCustomer, mapTerminal, listType)
            : null;
    sw.stop();

    sw.start("3");
    Set<String> productCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
            .map(AllowSaleListProductVo::getProductCode)
            .collect(Collectors.toSet());
    Map<String, ProductVo> productMap = this.findProductMap(productCodeSet, productImageFlag);
    Map<String, String> productSpuCodeMap = this.findProductSpuCodeMap(productCodeSet);
    sw.stop();

    sw.start("5");
    Set<String> itemKeys =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getItemKey()))
            .map(AllowSaleListProductVo::getItemKey)
            .collect(Collectors.toSet());
    Map<String, String> relateRuleMap = this.findRelateRuleMap(itemKeys);
    sw.stop();

    sw.start("4");
    this.initAllowSaleListProduct(
        list,
        listType,
        priceFlag,
        mapCustomer,
        mapTerminal,
        priceModelVoMap,
        productMap,
        productSpuCodeMap,
        relateRuleMap);
    sw.stop();
    log.info("tj[{}]", sw.prettyPrint());
  }



  /**
   * 根据客户编码集合获取对应的客户详情
   *
   * @param customerCodeSet
   * @return k-customerCode,v-客户信息
   */
  private Map<String, CustomerVo> findCustomerMap(Set<String> customerCodeSet) {
    if (CollectionUtils.isEmpty(customerCodeSet)) {
      return Maps.newHashMap();
    }
    List<CustomerVo> list = this.customerVoService.findForPriceByCustomerCodes(customerCodeSet);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNotBlank(a.getCustomerCode()))
        .collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity(), (a, b) -> a));
  }

  /**
   * 根据终端编码集合获取对应的终端详情
   *
   * @param terminalCodeSet
   * @return k-terminalCode,v-终端信息
   */
  private Map<String, TerminalVo> findTerminalMap(Set<String> terminalCodeSet) {
    if (CollectionUtils.isEmpty(terminalCodeSet)) {
      return Maps.newHashMap();
    }
    List<TerminalVo> list =
        this.terminalVoService.findMainDetailsByTerminalCodes(Lists.newArrayList(terminalCodeSet));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
        .collect(Collectors.toMap(TerminalVo::getTerminalCode, Function.identity(), (a, b) -> a));
  }

  /**
   * 根据终端编码集合获取对应的终端详情
   *
   * @param list
   * @param mapCustomer
   * @param mapTerminal
   * @param listType
   * @return k-terminalCode,v-终端信息
   */
  private Map<String, PriceModelVo> findPriceMap(
      List<AllowSaleListProductVo> list,
      Map<String, CustomerVo> mapCustomer,
      Map<String, TerminalVo> mapTerminal,
      String listType) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getProductCode(), a.getBusinessCode()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleListProductVo::getBusinessCode,
                    Collectors.mapping(
                        AllowSaleListProductVo::getProductCode, Collectors.toSet())));
    if (!listType.equals(AllowSaleListTypeEnums.CUSTOMER.getCode())
        && !listType.equals(AllowSaleListTypeEnums.TERMINAL.getCode())) {
      return Maps.newHashMap();
    }
    final FindBusinessPriceDto dto = new FindBusinessPriceDto();
    dto.setUserType(listType);
    List<FindBusinessPriceItemDto> items = Lists.newLinkedList();
    for (Entry<String, Set<String>> item : map.entrySet()) {
      final FindBusinessPriceItemDto cur = new FindBusinessPriceItemDto();
      cur.setBusinessCode(item.getKey());
      cur.setProductCodeSet(item.getValue());
      List<SearchPriceDimensionItemDto> dimensionItems = null;
      if (listType.equals(AllowSaleListTypeEnums.CUSTOMER.getCode())) {
        CustomerVo customerVo = mapCustomer.get(item.getKey());
        dimensionItems = this.findSearchPriceDimensionItemByCustomerVo(customerVo);
      } else if (listType.equals(AllowSaleListTypeEnums.TERMINAL.getCode())) {
        TerminalVo terminalVo = mapTerminal.get(item.getKey());
        dimensionItems = this.findSearchPriceDimensionItemByTerminalVo(terminalVo);
      }
      if (CollectionUtils.isNotEmpty(dimensionItems)) {
        List<FindBusinessPriceItemDimensionDto> dimensions = Lists.newLinkedList();
        for (SearchPriceDimensionItemDto searchPriceDimensionItemDto : dimensionItems) {
          final FindBusinessPriceItemDimensionDto dto1 = new FindBusinessPriceItemDimensionDto();
          dto1.setCode(searchPriceDimensionItemDto.getDimensionCode());
          dto1.setValues(searchPriceDimensionItemDto.getRelateCodeSet());
          dimensions.add(dto1);
        }
        cur.setDimensions(dimensions);
      }
      items.add(cur);
    }
    dto.setList(items);
    return this.priceModelVoService.findBusinessPrice(dto);
  }

  /**
   * 根据终端信息获取询价维度item信息
   *
   * @param terminalVo
   * @return
   */
  private List<SearchPriceDimensionItemDto> findSearchPriceDimensionItemByTerminalVo(
      TerminalVo terminalVo) {
    if (Objects.isNull(terminalVo)) {
      return Lists.newLinkedList();
    }
    List<SearchPriceDimensionItemDto> dimensionItems = Lists.newLinkedList();
    if (StringUtils.isNotBlank(terminalVo.getTerminalCode())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.TERMINAL.getDictCode());
      cur.setRelateCodeSet(Sets.newHashSet(terminalVo.getTerminalCode()));
      dimensionItems.add(cur);
    }
    if (CollectionUtils.isNotEmpty(terminalVo.getOrgList())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.ORG.getDictCode());
      cur.setRelateCodeSet(
          terminalVo.getOrgList().stream()
              .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
              .map(TerminalRelaOrgVo::getOrgCode)
              .collect(Collectors.toSet()));
      dimensionItems.add(cur);
    }
    if (StringUtils.isNotBlank(terminalVo.getChannel())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.CHANNEL.getDictCode());
      cur.setRelateCodeSet(Sets.newHashSet(terminalVo.getChannel()));
      dimensionItems.add(cur);
    }
    return dimensionItems;
  }

  /**
   * 根据经销商信息获取询价维度item信息
   *
   * @param customerVo
   * @return
   */
  private List<SearchPriceDimensionItemDto> findSearchPriceDimensionItemByCustomerVo(
      CustomerVo customerVo) {
    if (Objects.isNull(customerVo)) {
      return Lists.newLinkedList();
    }
    List<SearchPriceDimensionItemDto> dimensionItems = Lists.newLinkedList();
    if (StringUtils.isNotBlank(customerVo.getCustomerCode())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.CUSTOMER.getDictCode());
      cur.setRelateCodeSet(Sets.newHashSet(customerVo.getCustomerCode()));
      dimensionItems.add(cur);
    }
    if (CollectionUtils.isNotEmpty(customerVo.getOrgList())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.ORG.getDictCode());
      cur.setRelateCodeSet(
          customerVo.getOrgList().stream()
              .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
              .map(CustomerRelateOrgVo::getOrgCode)
              .collect(Collectors.toSet()));
      dimensionItems.add(cur);
    }
    if (StringUtils.isNotBlank(customerVo.getChannel())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.CHANNEL.getDictCode());
      cur.setRelateCodeSet(Sets.newHashSet(customerVo.getChannel()));
      dimensionItems.add(cur);
    }
    if (StringUtils.isNotBlank(customerVo.getPriceGroup())) {
      SearchPriceDimensionItemDto cur = new SearchPriceDimensionItemDto();
      cur.setDimensionCode(PriceDimensionEnum.PRICE_GROUP.getDictCode());
      cur.setRelateCodeSet(Sets.newHashSet(customerVo.getPriceGroup()));
      dimensionItems.add(cur);
    }
    return dimensionItems;
  }

  /**
   * 根据商品编码集合获取对应的商品详情
   *
   * @param productCodeSet
   * @param productImageFlag 是否需要获取图片信息
   * @return k-productCode,v-商品信息
   */
  private Map<String, ProductVo> findProductMap(
      Set<String> productCodeSet, Boolean productImageFlag) {
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return Maps.newHashMap();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setCodeQueryFlag(true);
    dto.setCodes(Lists.newArrayList(productCodeSet));
    final Set<String> set = Sets.newHashSet(ProductExtInfoTypeEnum.PRODUCT_LEVEL.getDictCode());
    if (Boolean.TRUE.equals(productImageFlag)) {
      set.add(ProductExtInfoTypeEnum.PICTURE.getDictCode());
    }
    dto.setExInfoCodes(set);
    List<ProductVo> list = this.productVoService.findDetailsByProductDetailQueryDto(dto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
        .collect(Collectors.toMap(ProductVo::getProductCode, Function.identity(), (a, b) -> a));
  }

  /**
   * 根据商品编码集合获取对应的spuCode集合详情
   *
   * @param productCodeSet
   * @return k-productCode,v-spuCode1,spuCode2...
   */
  private Map<String, String> findProductSpuCodeMap(Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return Maps.newHashMap();
    }
    Map<String, List<ProductSpuVo>> map =
        this.productSpuVoService.findRelateSpuMapByProductCodes(Lists.newArrayList(productCodeSet));
    Map<String, String> re = Maps.newHashMap();
    if (!map.isEmpty()) {
      for (Entry<String, List<ProductSpuVo>> item : map.entrySet()) {
        re.put(
            item.getKey(),
            item.getValue().stream()
                .map(ProductSpuVo::getSpuCode)
                .collect(Collectors.joining(",")));
      }
    }
    return re;
  }

  /**
   * 获取清单对应的关联的规则编码信息
   *
   * @param itemKeys
   * @return
   */
  private Map<String, String> findRelateRuleMap(Set<String> itemKeys) {
    if(CollectionUtils.isEmpty(itemKeys)){
      return Maps.newHashMap();
    }
    final List<AllowSaleRuleRelateList> list =
        this.allowSaleRuleRelateListRepository.findByItemKeys(itemKeys);
    if(CollectionUtils.isEmpty(list)){
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getItemKey(), a.getRuleCode()))
        .sorted(Comparator.comparing(AllowSaleRuleRelateList::getRuleType))
        .collect(
            Collectors.groupingBy(
                AllowSaleRuleRelateList::getItemKey,
                Collectors.mapping(AllowSaleRuleRelateList::getRuleCode, Collectors.joining(","))));
  }

  /**
   * 绑定扩展信息(终端名称|经销商名称+组织名称+渠道名称+价格组名称+商品信息+价格信息)
   *
   * @param list 可购商品清单数据
   * @param listType 清单类型
   * @param priceFlag 是否询价
   * @param mapCustomer 经销商map
   * @param mapTerminal 终端map
   * @param priceModelVoMap 价格map
   * @param productMap 商品map
   * @param productSpuCodeMap spu-map
   * @param relateRuleMap 关联规则信息
   */
  private void initAllowSaleListProduct(
      List<AllowSaleListProductVo> list,
      String listType,
      Boolean priceFlag,
      Map<String, CustomerVo> mapCustomer,
      Map<String, TerminalVo> mapTerminal,
      Map<String, PriceModelVo> priceModelVoMap,
      Map<String, ProductVo> productMap,
      Map<String, String> productSpuCodeMap,
      Map<String, String> relateRuleMap) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    for (AllowSaleListProductVo item : list) {
      item.setListType(listType);
      item.setRelateRuleCode(relateRuleMap.getOrDefault(item.getItemKey(), StringUtils.EMPTY));
      if (listType.equals(AllowSaleListTypeEnums.CUSTOMER.getCode())) {
        CustomerVo customerVo = mapCustomer.get(item.getBusinessCode());
        if (Objects.nonNull(customerVo)) {
          item.setBusinessName(customerVo.getCustomerName());
          if (CollectionUtils.isNotEmpty(customerVo.getOrgList())) {
            item.setOrgCode(
                customerVo.getOrgList().stream()
                    .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
                    .map(CustomerRelateOrgVo::getOrgCode)
                    .collect(Collectors.joining(",")));
            item.setOrgName(
                customerVo.getOrgList().stream()
                    .filter(a -> StringUtils.isNotBlank(a.getOrgName()))
                    .map(CustomerRelateOrgVo::getOrgName)
                    .collect(Collectors.joining(",")));
          }
          item.setChannel(customerVo.getChannel());
          item.setPriceGroup(customerVo.getPriceGroup());
        }
      }
      if (listType.equals(AllowSaleListTypeEnums.TERMINAL.getCode())) {
        TerminalVo terminalVo = mapTerminal.get(item.getBusinessCode());
        if (Objects.nonNull(terminalVo)) {
          item.setBusinessName(terminalVo.getTerminalName());
          if (CollectionUtils.isNotEmpty(terminalVo.getOrgList())) {
            item.setOrgCode(
                terminalVo.getOrgList().stream()
                    .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
                    .map(TerminalRelaOrgVo::getOrgCode)
                    .collect(Collectors.joining(",")));
            item.setOrgName(
                terminalVo.getOrgList().stream()
                    .filter(a -> StringUtils.isNotBlank(a.getOrgName()))
                    .map(TerminalRelaOrgVo::getOrgName)
                    .collect(Collectors.joining(",")));
          }
          item.setChannel(terminalVo.getChannel());
        }
      }
      ProductVo productVo = productMap.get(item.getProductCode());
      if (Objects.nonNull(productVo)) {
        item.setProductLevelCode(productVo.getProductLevelCode());
        item.setProductLevelName(productVo.getProductLevelName());
        item.setProductName(productVo.getProductName());
        item.setSpec(productVo.getSpec());
        item.setSaleUnit(productVo.getSaleUnit());
      }
      item.setProduct(productVo);
      item.setSpuCode(productSpuCodeMap.get(item.getProductCode()));
      if (Boolean.TRUE.equals(priceFlag)) {
        PriceModelVo priceModelVo = priceModelVoMap.get(item.getPriceKey());
        if (Objects.nonNull(priceModelVo)) {
          item.setPrice(priceModelVo.getPrice());
          item.setPriceCode(priceModelVo.getPriceCode());
        }
      }
    }
  }
}
