package com.biz.crm.dms.business.allow.sale.local.list.service.notifier;

import com.biz.crm.dms.business.allow.sale.local.list.model.AllowSaleRuleBusinessTrigger;
import com.biz.crm.dms.business.allow.sale.local.rule.service.internal.AllowSaleRuleListAdapter;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleDimensionTypeEnums;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerEventDto;
import com.biz.crm.mdm.business.customer.sdk.event.CustomerEventListener;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 经销商信息变更事件
 *
 * @author sunx
 * @date 2022/5/25
 */
@Component
public class AllowCustomerEventListenerImpl implements CustomerEventListener {

  @Autowired(required = false)
  private AllowSaleRuleListAdapter allowSaleRuleListAdapter;

  @Override
  @Transactional
  public void onApproved(CustomerEventDto dto) {
    Validate.notNull(dto, "经销商信息不能为空");
    final CustomerVo vo = dto.getNewest();
    Validate.notNull(vo, "经销商信息不能为空");
    vo.setActApproveStatus(ProcessStatusEnum.PASS.getDictCode());
    final String customerCode = vo.getCustomerCode();
    if (StringUtils.isBlank(customerCode)) {
      return;
    }
    this.handlerAllowSale(vo, Boolean.TRUE);
  }

  @Override
  public void onCreateAndSubmit(CustomerEventDto dto) {
    // 保存并提交的需要删除历史组织对应的清单数据
    Validate.notNull(dto, "经销商信息不能为空");
    final CustomerVo vo = dto.getOriginal();
    if (Objects.isNull(vo)) {
      return;
    }
    this.onDelete(dto);
  }

  @Override
  @Transactional
  public void onUpdate(CustomerEventDto dto) {
    Validate.notNull(dto, "经销商参数不能为空");
    final CustomerVo oldVo = dto.getOriginal();
    final CustomerVo vo = dto.getNewest();
    Validate.notNull(vo, "经销商信息不能为空");

    final List<CustomerRelateOrgVo> oldOrgList = oldVo.getOrgList();
    final List<CustomerRelateOrgVo> orgList = vo.getOrgList();
    Set<String> oldOrgCodeSet = Sets.newHashSet();
    Set<String> orgCodeSet = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(oldOrgList)) {
      oldOrgCodeSet =
          oldOrgList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
              .map(CustomerRelateOrgVo::getOrgCode)
              .collect(Collectors.toSet());
    }
    if (!CollectionUtils.isEmpty(orgList)) {
      orgCodeSet =
          orgList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
              .map(CustomerRelateOrgVo::getOrgCode)
              .collect(Collectors.toSet());
    }
    // 需要删除的清单信息
    final SetView<String> d1 = Sets.difference(oldOrgCodeSet, orgCodeSet);
    // 需要新增的清单信息
    final SetView<String> d2 = Sets.difference(orgCodeSet, oldOrgCodeSet);
    if (!ProcessStatusEnum.PASS.getDictCode().equals(vo.getProcessStatus())
        || (d1.isEmpty() && d2.isEmpty())) {
      return;
    }
    if (!d1.isEmpty()) {
      List<CustomerRelateOrgVo> curOrgList = Lists.newLinkedList();
      for (String item : d1) {
        final CustomerRelateOrgVo cur = new CustomerRelateOrgVo();
        cur.setOrgCode(item);
        curOrgList.add(cur);
      }
      oldVo.setOrgList(curOrgList);
      this.handlerAllowSale(oldVo, Boolean.FALSE);
    }

    if (!d2.isEmpty()) {
      List<CustomerRelateOrgVo> curOrgList = Lists.newLinkedList();
      for (String item : d2) {
        final CustomerRelateOrgVo cur = new CustomerRelateOrgVo();
        cur.setOrgCode(item);
        curOrgList.add(cur);
      }
      vo.setOrgList(curOrgList);
      this.handlerAllowSale(vo, Boolean.TRUE);
    }
  }

  @Override
  @Transactional
  public void onDelete(CustomerEventDto dto) {
    Validate.notNull(dto, "经销商参数不能为空");
    final CustomerVo original = dto.getOriginal();
    Validate.notNull(original, "终端审核参数不能为空");
    if (!ProcessStatusEnum.PASS.getDictCode().equals(original.getProcessStatus())) {
      return;
    }
    this.handlerAllowSale(original, Boolean.FALSE);
  }

  /**
   * 根据经销商信息操作可购信息
   *
   * @param vo
   * @param createFlag
   */
  private void handlerAllowSale(CustomerVo vo, Boolean createFlag) {
    if (Objects.isNull(vo) || StringUtils.isBlank(vo.getCustomerCode())) {
      return;
    }
    final String customerCode = vo.getCustomerCode();
    final List<CustomerRelateOrgVo> orgList = vo.getOrgList();
    Set<String> orgCodeSet = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(orgList)) {
      orgCodeSet =
          orgList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
              .map(CustomerRelateOrgVo::getOrgCode)
              .collect(Collectors.toSet());
    }
    if (CollectionUtils.isEmpty(orgCodeSet)) {
      final AllowSaleRuleBusinessTrigger business = new AllowSaleRuleBusinessTrigger();
      business.setBusinessCode(customerCode);
      business.setDimensionType(AllowSaleDimensionTypeEnums.CUSTOMER.getCode());
      if (Boolean.TRUE.equals(createFlag)) {
        this.allowSaleRuleListAdapter.createByBusiness(business);
      } else {
        this.allowSaleRuleListAdapter.deleteByBusiness(business);
      }
    } else {
      for (String orgCode : orgCodeSet) {
        final AllowSaleRuleBusinessTrigger business = new AllowSaleRuleBusinessTrigger();
        business.setBusinessCode(customerCode);
        business.setDimensionType(AllowSaleDimensionTypeEnums.CUSTOMER.getCode());
        business.setRelateTypeCode(orgCode);
        business.setRelateDimensionType(AllowSaleDimensionTypeEnums.CUSTOMER_ORG.getCode());
        if (Boolean.TRUE.equals(createFlag)) {
          this.allowSaleRuleListAdapter.createByBusiness(business);
        } else {
          this.allowSaleRuleListAdapter.deleteByBusiness(business);
        }
      }
    }
  }
}
