package com.biz.crm.dms.business.allow.sale.local.list.service.notifier;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.dimension.service.DimensionService;
import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleRuleRelateList;
import com.biz.crm.dms.business.allow.sale.local.list.model.RelateRule;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleRuleRelateListService;
import com.biz.crm.dms.business.allow.sale.local.product.service.AllowSaleRuleProductService;
import com.biz.crm.dms.business.allow.sale.local.rule.entity.AllowSaleRule;
import com.biz.crm.dms.business.allow.sale.local.rule.service.AllowSaleRuleService;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleDimensionTypeEnums;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.org.sdk.dto.OrgEventBatchDto;
import com.biz.crm.mdm.business.org.sdk.dto.OrgEventDto;
import com.biz.crm.mdm.business.org.sdk.event.OrgEventListener;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 允销实现组织层级更新事件
 *
 * @author sunx
 * @date 2022/7/4
 */
@Component
public class AllowOrgEventListenerImpl implements OrgEventListener {

  @Autowired(required = false)
  private DimensionService dimensionService;

  @Autowired(required = false)
  private AllowSaleRuleProductService allowSaleRuleProductService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private TerminalVoService terminalVoService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private AllowSaleRuleService allowSaleRuleService;

  @Autowired(required = false)
  private AllowSaleRuleRelateListService allowSaleRuleRelateListService;

  @Override
  public void onDelete(List<String> orgCodes) {}

  @Override
  public void onDeleteBatch(OrgEventBatchDto orgEventBatchDto) {}

  @Override
  public void onEnableBatch(OrgEventBatchDto orgEventBatchDto) {}

  @Override
  public void onDisableBatch(OrgEventBatchDto orgEventBatchDto) {}

  @Override
  @Transactional
  public void onUpdate(OrgEventDto dto) {
    if (Objects.isNull(dto)
        || dto.getParentCode().equals(dto.getOldParentCode())
        || StringUtils.isBlank(dto.getOrgCode())) {
      return;
    }
    Set<String> set =
        Sets.newHashSet(
            AllowSaleDimensionTypeEnums.CUSTOMER_ORG.getCode(),
            AllowSaleDimensionTypeEnums.TERMINAL_ORG.getCode());
    for (String item : set) {
      this.onItemUpdate(dto, item);
    }
  }

  /**
   * item update
   *
   * @param dto
   * @param dimensionType
   */
  private void onItemUpdate(OrgEventDto dto, String dimensionType) {
    // 获取修改前的父级组织编码关联的规则编码集合
    // 获取修改后的父级组织编码关联的规则编码集合
    // 找出需要删除、或新增数据的规则集合

    // 产品层级历史关联的可购规则编码集合
    Set<String> oldAllowRuleCodes = Sets.newHashSet();
    // 改变之后关联的可购规则编码集合
    Set<String> allowRuleCodes = Sets.newHashSet();
    if (StringUtils.isNotBlank(dto.getOldParentCode())) {
      final Map<String, Set<String>> map =
          this.dimensionService.findRelateRuleCodesMapByRelateCodes(
              Lists.newArrayList(dto.getOldParentCode()), dimensionType);
      oldAllowRuleCodes = map.getOrDefault(dto.getOldParentCode(), Sets.newHashSet());
    }
    if (StringUtils.isNotBlank(dto.getParentCode())) {
      final Map<String, Set<String>> map =
          this.dimensionService.findRelateRuleCodesMapByRelateCodes(
              Lists.newArrayList(dto.getParentCode()), dimensionType);
      allowRuleCodes = map.getOrDefault(dto.getParentCode(), Sets.newHashSet());
    }

    Set<String> delRuleCodeSet =
        Sets.newHashSet(Sets.difference(oldAllowRuleCodes, allowRuleCodes));
    Set<String> addRuleCodeSet =
        Sets.newHashSet(Sets.difference(allowRuleCodes, oldAllowRuleCodes));
    Set<String> allRuleCodeSet = Sets.newHashSet();
    allRuleCodeSet.addAll(addRuleCodeSet);
    allRuleCodeSet.addAll(delRuleCodeSet);
    if (CollectionUtils.isEmpty(allRuleCodeSet)) {
      return;
    }
    final List<AllowSaleRule> ruleList =
        this.allowSaleRuleService.findByCodesAndTenantCode(
            Lists.newArrayList(allRuleCodeSet), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(ruleList)) {
      return;
    }
    // 未删除且已启用的可购规则
    Map<String, AllowSaleRule> mapRule =
        ruleList.stream()
            .filter(
                a ->
                    DelFlagStatusEnum.NORMAL.getCode().equals(a.getDelFlag())
                        && EnableStatusEnum.ENABLE.getCode().equals(a.getEnableStatus()))
            .collect(
                Collectors.toMap(AllowSaleRule::getRuleCode, Function.identity(), (a, b) -> a));

    // 获取当前组织及下级的所有审核通过且未删除的经销商|终端
    // k-经销商|终端编码，v-关联层级对应的降维码集合
    Set<String> businessCodeSet = Sets.newHashSet();
    String orgCode = dto.getOrgCode();
    if (dimensionType.equals(AllowSaleDimensionTypeEnums.CUSTOMER_ORG.getCode())) {
      final Map<String, Set<String>> map1 =
          this.customerVoService.findAllowSaleCustomerByOrgCodes(Sets.newHashSet(orgCode));
      if (!map1.isEmpty()) {
        businessCodeSet.addAll(map1.keySet());
      }
    } else if (dimensionType.equals(AllowSaleDimensionTypeEnums.TERMINAL_ORG.getCode())) {
      final Map<String, Set<String>> map1 =
          this.terminalVoService.findAllowSaleTerminalByOrgCodes(Sets.newHashSet(orgCode));
      if (!map1.isEmpty()) {
        businessCodeSet.addAll(map1.keySet());
      }
    }
    if (CollectionUtils.isEmpty(businessCodeSet)) {
      return;
    }
    this.handlerAllowSaleList(delRuleCodeSet, addRuleCodeSet, mapRule, businessCodeSet);
  }

  /**
   * 同步影响的可购清单数据
   *
   * @param delRuleCodeSet
   * @param addRuleCodeSet
   * @param mapRule
   * @param businessCodeSet
   */
  private void handlerAllowSaleList(
      Set<String> delRuleCodeSet,
      Set<String> addRuleCodeSet,
      Map<String, AllowSaleRule> mapRule,
      Set<String> businessCodeSet) {
    if (mapRule.isEmpty() || CollectionUtils.isEmpty(businessCodeSet)) {
      return;
    }
    String tenantCode = TenantUtils.getTenantCode();
    for (Entry<String, AllowSaleRule> item : mapRule.entrySet()) {
      String ruleCode = item.getKey();
      AllowSaleRule rule = item.getValue();
      List<AllowSaleRuleRelateList> list =
          this.findItemRelateList(rule, businessCodeSet, tenantCode);
      if (CollectionUtils.isEmpty(list)) {
        continue;
      }
      if (delRuleCodeSet.contains(ruleCode)) {
        this.allowSaleRuleRelateListService.deleteByList(list);
      } else if (addRuleCodeSet.contains(ruleCode)) {
        this.allowSaleRuleRelateListService.createByList(list);
      }
    }
  }

  /**
   * 获取可购清单组合数据
   *
   * @param rule
   * @param businessCodeSet
   * @param tenantCode
   * @return
   */
  private List<AllowSaleRuleRelateList> findItemRelateList(
      AllowSaleRule rule, Set<String> businessCodeSet, String tenantCode) {
    Map<String, Set<String>> productCodeSetMap =
        this.allowSaleRuleProductService.findRuleRelateProductCodesMapByRuleCodes(
            Lists.newArrayList(rule.getRuleCode()));
    if (productCodeSetMap.isEmpty()) {
      return Lists.newLinkedList();
    }
    Set<String> productCodeSet =
        productCodeSetMap.getOrDefault(rule.getRuleCode(), Sets.newHashSet());
    final String listType = this.dimensionService.getListType(rule.getDimensionType());
    List<AllowSaleRuleRelateList> list = Lists.newArrayList();
    for (String businessCode : businessCodeSet) {
      for (String productCode : productCodeSet) {
        final AllowSaleRuleRelateList cur = new AllowSaleRuleRelateList();
        cur.setTenantCode(tenantCode);
        cur.setRuleType(rule.getRuleType());
        cur.setRuleCode(rule.getRuleCode());
        cur.setBusinessCode(businessCode);
        cur.setProductCode(productCode);
        cur.setListType(listType);
        cur.setItemKey(
            StringUtils.joinWith(
                ":", cur.getListType(), cur.getBusinessCode(), cur.getProductCode()));
        cur.setSubItemKey(
            StringUtils.joinWith(
                ":",
                cur.getRuleCode(),
                cur.getListType(),
                cur.getBusinessCode(),
                cur.getProductCode()));
        final RelateRule relateRule = new RelateRule();
        relateRule.setRuleType(cur.getRuleType());
        relateRule.setRuleCode(cur.getRuleCode());
        relateRule.setListType(cur.getListType());
        cur.setRule(relateRule);
        list.add(cur);
      }
    }
    return list;
  }
}
