package com.biz.crm.dms.business.allow.sale.local.product.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.local.product.entity.AllowSaleRuleProduct;
import com.biz.crm.dms.business.allow.sale.local.product.model.ProductRelateVo;
import com.biz.crm.dms.business.allow.sale.local.product.repository.AllowSaleRuleProductRepository;
import com.biz.crm.dms.business.allow.sale.local.product.service.AllowSaleRuleProductService;
import com.biz.crm.dms.business.allow.sale.local.util.AllowSaleUtil;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleProductRelateTypeEnums;
import com.biz.crm.mdm.business.product.level.sdk.dto.RelateProductLevelCodeQueryDto;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.product.spu.sdk.service.ProductSpuRelateSkuVoService;
import com.biz.crm.mdm.business.product.spu.sdk.service.ProductSpuVoService;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuRelateSkuVo;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 允销规则关联的可购商品信息(AllowSaleRuleProduct)表服务实现类
 *
 * @author sunx
 * @date 2021-11-19 15:17:33
 */
@Slf4j
@Service
public class AllowSaleRuleProductServiceImpl implements AllowSaleRuleProductService {

  @Autowired(required = false)
  private AllowSaleRuleProductRepository allowSaleRuleProductRepository;

  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private ProductSpuVoService productSpuVoService;

  @Autowired(required = false)
  private ProductSpuRelateSkuVoService productSpuRelateSkuVoService;

  @Override
  @Transactional
  public List<AllowSaleRuleProduct> findByRuleCode(String ruleCode) {
    if (StringUtils.isBlank(ruleCode)) {
      return Lists.newLinkedList();
    }
    List<AllowSaleRuleProduct> list =
        this.allowSaleRuleProductRepository.findByRuleCodes(Lists.newArrayList(ruleCode));
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }

    Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRelateType()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleRuleProduct::getRelateType,
                    Collectors.mapping(AllowSaleRuleProduct::getRelateCode, Collectors.toSet())));
    if (Objects.isNull(map) || map.isEmpty()) {
      return list;
    }
    Map<String, String> mapProduct = this.findProductNameMap(map);
    Map<String, String> mapSpu = this.findSpuNameMap(map);
    Map<String, String> mapProductLevel = this.findProductLevelNameMap(map);

    for (AllowSaleRuleProduct item : list) {
      if (AllowSaleRuleProductRelateTypeEnums.PRODUCT.getCode().equals(item.getRelateType())) {
        item.setRelateName(mapProduct.get(item.getRelateCode()));
      }
      if (AllowSaleRuleProductRelateTypeEnums.SPU.getCode().equals(item.getRelateType())) {
        item.setRelateName(mapSpu.get(item.getRelateCode()));
      }
      if (AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL
          .getCode()
          .equals(item.getRelateType())) {
        item.setRelateName(mapProductLevel.get(item.getRelateCode()));
      }
    }
    return list;
  }

  @Override
  @Transactional
  public void batchSave(List<AllowSaleRuleProduct> list) {
    Validate.isTrue(CollectionUtils.isNotEmpty(list), "参数不能为空");
    String ruleCode = list.get(0).getRuleCode();
    Validate.notBlank(ruleCode, "规则编码不能为空");
    Optional<AllowSaleRuleProduct> first =
        list.stream().filter(a -> !ruleCode.equals(a.getRuleCode())).findFirst();
    Validate.isTrue(!first.isPresent(), "当前处理的集合必须属于同一个规则");
    Set<String> set = Sets.newHashSet();
    list.forEach(a -> Validate.isTrue(set.add(a.getItemCode()), "存在相同可购商品信息"));
    this.allowSaleRuleProductRepository.deleteByRuleCode(ruleCode);
    this.allowSaleRuleProductRepository.saveBatch(list);
  }

  @Override
  public Map<String, Set<String>> findRuleRelateProductCodesMapByRuleCodes(
      List<String> ruleCodeList) {
    if (CollectionUtils.isEmpty(ruleCodeList)) {
      return Maps.newHashMap();
    }
    List<AllowSaleRuleProduct> list =
        this.allowSaleRuleProductRepository.findByRuleCodes(ruleCodeList);
    Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRelateType()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleRuleProduct::getRelateType,
                    Collectors.mapping(AllowSaleRuleProduct::getRelateCode, Collectors.toSet())));
    if (Objects.isNull(map) || map.isEmpty()) {
      return Maps.newHashMap();
    }

    //    Map<String, Set<String>> spuRelateProductCodeMap = this.findSpuRelateProductCodeMap(map);
    Map<String, Set<String>> spuRelateProductCodeMap = Maps.newHashMap();
    Map<String, Set<String>> levelRelateProductCodeMap = this.findLevelRelateProductCodeMap(map);
    Set<String> productCodeSet = this.findProductRelateMap(map);

    Set<String> allProductCodeSet = Sets.newHashSet();
    if (!spuRelateProductCodeMap.isEmpty()) {
      spuRelateProductCodeMap.values().forEach(allProductCodeSet::addAll);
    }

    if (!levelRelateProductCodeMap.isEmpty()) {
      levelRelateProductCodeMap.values().forEach(allProductCodeSet::addAll);
    }
    if (!productCodeSet.isEmpty()) {
      allProductCodeSet.addAll(productCodeSet);
    }
    if (CollectionUtils.isEmpty(allProductCodeSet)) {
      return Maps.newHashMap();
    }

    final ProductQueryDto queryDto = new ProductQueryDto();
    queryDto.setIsShelf(IsShelfEnum.UP.getCode());
    queryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    queryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    queryDto.setProductCodeList(Lists.newArrayList(allProductCodeSet));
    final List<ProductVo> normalProductVoList =
        this.productVoService.findByProductQueryDto(queryDto);
    if (CollectionUtils.isEmpty(normalProductVoList)) {
      return Maps.newHashMap();
    }

    Set<String> normalProductCodeSet =
        normalProductVoList.stream().map(ProductVo::getProductCode).collect(Collectors.toSet());

    Map<String, List<AllowSaleRuleProduct>> ruleProductMap =
        list.stream()
            .filter(
                a -> StringUtils.isNoneBlank(a.getRuleCode(), a.getRelateType(), a.getRelateCode()))
            .collect(Collectors.groupingBy(AllowSaleRuleProduct::getRuleCode));

    if (Objects.isNull(ruleProductMap) || ruleProductMap.isEmpty()) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> reMap = Maps.newHashMap();
    for (Entry<String, List<AllowSaleRuleProduct>> item : ruleProductMap.entrySet()) {
      Set<String> curProductCodeSet = Sets.newHashSet();
      item.getValue()
          .forEach(
              a -> {
                if (AllowSaleRuleProductRelateTypeEnums.PRODUCT
                    .getCode()
                    .equals(a.getRelateType())) {
                  if (normalProductCodeSet.contains(a.getRelateCode())) {
                    curProductCodeSet.add(a.getRelateCode());
                  }
                } else if (AllowSaleRuleProductRelateTypeEnums.SPU
                    .getCode()
                    .equals(a.getRelateType())) {
                  Set<String> cur = spuRelateProductCodeMap.get(a.getRelateCode());
                  final SetView<String> view = Sets.intersection(cur, normalProductCodeSet);
                  if (CollectionUtils.isNotEmpty(view)) {
                    curProductCodeSet.addAll(view);
                  }
                } else if (AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL
                    .getCode()
                    .equals(a.getRelateType())) {
                  Set<String> cur =
                      levelRelateProductCodeMap.getOrDefault(a.getRelateCode(), Sets.newHashSet());
                  final SetView<String> view = Sets.intersection(cur, normalProductCodeSet);
                  if (CollectionUtils.isNotEmpty(view)) {
                    curProductCodeSet.addAll(cur);
                  }
                }
              });
      if (CollectionUtils.isNotEmpty(curProductCodeSet)) {
        reMap.put(item.getKey(), curProductCodeSet);
      }
    }
    return reMap;
  }

  @Override
  public Map<String, Set<String>> findProductLevelRelateRuleCodesMapByProductLevelCodes(
      List<String> productLevelCodeList) {
    if (CollectionUtils.isEmpty(productLevelCodeList)) {
      return Maps.newHashMap();
    }

    final RelateProductLevelCodeQueryDto relateProductLevelCodeQueryDto =
        new RelateProductLevelCodeQueryDto();
    relateProductLevelCodeQueryDto.setSearchType(1);
    relateProductLevelCodeQueryDto.setProductLevelCodeSet(Sets.newHashSet(productLevelCodeList));
    // k-层级编码（根据productLevelCodeList获取到包含自己及上级层级编码），v-降维码
    final Map<String, String> productLevelRuleCodeMap =
        this.productLevelVoSdkService.findByRelateProductLevelCodeQueryDto(
            relateProductLevelCodeQueryDto);
    if (productLevelRuleCodeMap.isEmpty()) {
      return Maps.newHashMap();
    }
    List<ProductRelateVo> list =
        this.allowSaleRuleProductRepository.findProductRelateVoListByRelateTypeAndRelateCodes(
            AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL.getCode(),
            Lists.newArrayList(productLevelRuleCodeMap.keySet()));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    // k-商品层次编码,v-可购规则编码集合
    final Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
            .collect(
                Collectors.groupingBy(
                    ProductRelateVo::getRelateCode,
                    Collectors.mapping(ProductRelateVo::getRuleCode, Collectors.toSet())));
    return AllowSaleUtil.findRelateRule(productLevelCodeList, productLevelRuleCodeMap, map);
  }

  @Override
  public Map<String, Set<String>> findSpuRelateRuleCodesMapBySpuCodes(List<String> spuCodeList) {
    if (CollectionUtils.isEmpty(spuCodeList)) {
      return Maps.newHashMap();
    }
    List<ProductRelateVo> list =
        this.allowSaleRuleProductRepository.findProductRelateVoListByRelateTypeAndRelateCodes(
            AllowSaleRuleProductRelateTypeEnums.SPU.getCode(), spuCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }

    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
        .collect(
            Collectors.groupingBy(
                ProductRelateVo::getRelateCode,
                Collectors.mapping(ProductRelateVo::getRuleCode, Collectors.toSet())));
  }

  @Override
  public Map<String, Set<String>> findSkuRelateRuleCodesMapBySkuCodes(
      List<String> productCodeList) {
    if (CollectionUtils.isEmpty(productCodeList)) {
      return Maps.newHashMap();
    }
    List<ProductRelateVo> list =
        this.allowSaleRuleProductRepository.findProductRelateVoListByRelateTypeAndRelateCodes(
            AllowSaleRuleProductRelateTypeEnums.PRODUCT.getCode(), productCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }

    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getRelateCode(), a.getRuleCode()))
        .collect(
            Collectors.groupingBy(
                ProductRelateVo::getRelateCode,
                Collectors.mapping(ProductRelateVo::getRuleCode, Collectors.toSet())));
  }

  /**
   * 获取商品层级编码名称map
   *
   * @param map
   * @return k-productLevelCode,v-productLevelName
   */
  private Map<String, String> findProductLevelNameMap(Map<String, Set<String>> map) {
    Set<String> productLevelCodeSet =
        map.get(AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL.getCode());
    if (CollectionUtils.isEmpty(productLevelCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProductLevelVo> list =
        this.productLevelVoSdkService.findListByCodes(Lists.newArrayList(productLevelCodeSet));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getProductLevelCode(), a.getProductLevelName()))
        .collect(
            Collectors.toMap(
                ProductLevelVo::getProductLevelCode,
                ProductLevelVo::getProductLevelName,
                (a, b) -> a));
  }

  /**
   * 获取spu编码名称map
   *
   * @param map
   * @return k-spuCode,v-spuName
   */
  private Map<String, String> findSpuNameMap(Map<String, Set<String>> map) {
    Set<String> spuCodeSet = map.get(AllowSaleRuleProductRelateTypeEnums.SPU.getCode());
    if (CollectionUtils.isEmpty(spuCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProductSpuVo> list =
        this.productSpuVoService.findBySpuCodes(Lists.newArrayList(spuCodeSet));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSpuCode(), a.getSpuName()))
        .collect(Collectors.toMap(ProductSpuVo::getSpuCode, ProductSpuVo::getSpuName, (a, b) -> a));
  }

  /**
   * 获取商品编码名称map
   *
   * @param map
   * @return k-productCode,v-productName
   */
  private Map<String, String> findProductNameMap(Map<String, Set<String>> map) {
    Set<String> productCodeSet = map.get(AllowSaleRuleProductRelateTypeEnums.PRODUCT.getCode());
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProductVo> list =
        this.productVoService.findDetailsByIdsOrProductCodes(
            Lists.newLinkedList(), Lists.newArrayList(productCodeSet));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getProductCode(), a.getProductName()))
        .collect(
            Collectors.toMap(ProductVo::getProductCode, ProductVo::getProductName, (a, b) -> a));
  }

  /**
   * 根据产品层级编码获取关联的商品编码集合map（当前层级及子级层级关联的上架并启用且未删除的商品）
   *
   * @param map
   * @return k-productLevelCode,v-set<productCode>
   */
  private Map<String, Set<String>> findLevelRelateProductCodeMap(Map<String, Set<String>> map) {
    Set<String> productLevelCodeSet =
        map.get(AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL.getCode());
    if (CollectionUtils.isEmpty(productLevelCodeSet)) {
      return Maps.newHashMap();
    }
    final RelateProductLevelCodeQueryDto relateProductLevelCodeQueryDto =
        new RelateProductLevelCodeQueryDto();
    relateProductLevelCodeQueryDto.setSearchType(-1);
    relateProductLevelCodeQueryDto.setProductLevelCodeSet(productLevelCodeSet);
    // map1 k-层级对应自己的编码,v-降维码
    final Map<String, String> map1 =
        this.productLevelVoSdkService.findByRelateProductLevelCodeQueryDto(
            relateProductLevelCodeQueryDto);
    if (map1.isEmpty()) {
      return Maps.newHashMap();
    }
    final ProductQueryDto productQueryDto = new ProductQueryDto();
    productQueryDto.setProductLevelCodeList(Lists.newArrayList(map1.keySet()));
    productQueryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productQueryDto.setIsShelf(IsShelfEnum.UP.getCode());
    productQueryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    List<ProductVo> list = this.productVoService.findByProductQueryDto(productQueryDto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    for (ProductVo item : list) {
      final String ruleCode = map1.get(item.getProductLevelCode());
      if (StringUtils.isBlank(ruleCode)) {
        continue;
      }
      item.setProductLevelCode(ruleCode);
    }
    // k-商品层级对应的降维码，v-商品层级编码集合
    final Map<String, Set<String>> mapLevelProduct =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getProductLevelCode(), a.getProductCode()))
            .collect(
                Collectors.groupingBy(
                    ProductVo::getProductLevelCode,
                    Collectors.mapping(ProductVo::getProductCode, Collectors.toSet())));

    Map<String, Set<String>> re = Maps.newHashMap();
    for (String item : productLevelCodeSet) {
      final String ruleCode = map1.get(item);
      if (StringUtils.isBlank(ruleCode)) {
        continue;
      }
      Set<String> set = Sets.newHashSet();
      for (Entry<String, Set<String>> sub : mapLevelProduct.entrySet()) {
        if (!sub.getKey().startsWith(ruleCode)) {
          continue;
        }
        set.addAll(sub.getValue());
      }
      if (CollectionUtils.isNotEmpty(set)) {
        re.put(item, set);
      }
    }
    return re;
  }

  /**
   * 获取启用并上架的商品编码集合
   *
   * @param map
   * @return
   */
  private Set<String> findProductRelateMap(Map<String, Set<String>> map) {
    Set<String> productCodeSet = map.get(AllowSaleRuleProductRelateTypeEnums.PRODUCT.getCode());
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return Sets.newHashSet();
    }
    final ProductQueryDto productQueryDto = new ProductQueryDto();
    productQueryDto.setProductCodeList(Lists.newArrayList(productCodeSet));
    productQueryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productQueryDto.setIsShelf(IsShelfEnum.UP.getCode());
    productQueryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    List<ProductVo> list = this.productVoService.findByProductQueryDto(productQueryDto);
    if (CollectionUtils.isEmpty(list)) {
      return Sets.newHashSet();
    }
    return list.stream()
        .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
        .map(ProductVo::getProductCode)
        .collect(Collectors.toSet());
  }

  /**
   * 根据spu编码获取对spu关联的商品编码集合map
   *
   * @param map
   * @return k-spuCode,v-set<productCode>
   */
  private Map<String, Set<String>> findSpuRelateProductCodeMap(Map<String, Set<String>> map) {
    Set<String> spuCodeSet = map.get(AllowSaleRuleProductRelateTypeEnums.SPU.getCode());
    if (CollectionUtils.isEmpty(spuCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProductSpuRelateSkuVo> list =
        this.productSpuRelateSkuVoService.findBySpuCodes(Lists.newArrayList(spuCodeSet));
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSpuCode(), a.getProductCode()))
        .collect(
            Collectors.groupingBy(
                ProductSpuRelateSkuVo::getSpuCode,
                Collectors.mapping(ProductSpuRelateSkuVo::getProductCode, Collectors.toSet())));
  }
}
