package com.biz.crm.dms.business.allow.sale.local.util;

import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleList;
import com.biz.crm.dms.business.allow.sale.sdk.constant.AllowSaleRuleConstant;
import com.biz.crm.dms.business.allow.sale.sdk.dimension.vo.DimensionDictVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * 可购模块工具类
 *
 * @author sunx
 * @date 2022/6/28
 */
public class AllowSaleUtil {

  private AllowSaleUtil() {}

  /**
   * 根据itemKey获取清单主要信息
   *
   * @param itemKey 行记录编码
   * @return
   */
  public static AllowSaleList findByItemKey(String itemKey) {
    if (StringUtils.isBlank(itemKey) || !itemKey.contains(AllowSaleRuleConstant.SEPARATOR)) {
      return null;
    }
    final String[] split = StringUtils.split(itemKey, AllowSaleRuleConstant.SEPARATOR);
    if (split.length == 3) {
      final AllowSaleList allowSaleList = new AllowSaleList();
      allowSaleList.setListType(split[0]);
      allowSaleList.setBusinessCode(split[1]);
      allowSaleList.setProductCode(split[2]);
      return allowSaleList;
    }
    return null;
  }

  /**
   * 获取（组织|层级）对应上级关联的规则关系信息
   *
   * @param queryCodeList 原始查询（组织|层级）编码集合
   * @param ruleCodeMap 根据queryCodeList获取到的本级及上级的（组织|层级）编码-对应降维码map
   * @param mapRule 根据ruleCodeMap的key集合获取到关联规则map，k-（组织|层级）code,v-可购规则编码集合
   * @return k-（组织|层级）编码，v-规则编码
   */
  public static Map<String, Set<String>> findRelateRule(
      List<String> queryCodeList,
      Map<String, String> ruleCodeMap,
      Map<String, Set<String>> mapRule) {
    if (CollectionUtils.isEmpty(queryCodeList) || ruleCodeMap.isEmpty() || mapRule.isEmpty()) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> map = Maps.newHashMap();
    for (String item : queryCodeList) {
      String orgRuleCode = ruleCodeMap.get(item);
      if (StringUtils.isBlank(orgRuleCode)) {
        continue;
      }
      Set<String> rule = Sets.newHashSet();
      for (Entry<String, Set<String>> sub : mapRule.entrySet()) {
        final String cur = ruleCodeMap.get(sub.getKey());
        if (StringUtils.isBlank(cur)
            || !orgRuleCode.startsWith(cur)
            || CollectionUtils.isEmpty(sub.getValue())) {
          continue;
        }
        rule.addAll(sub.getValue());
      }
      if (CollectionUtils.isNotEmpty(rule)) {
        map.put(item, rule);
      }
    }
    return map;
  }

  /**
   * 获取（组织|层级）对应的下级所有关联的（终端编码|经销商编码）
   *
   * @param curRuleMap k-（组织|层级）编码，v-降维码
   * @param mapBusinessRule （组织|层级）对应下级的获取到的（经销商编码|终端编码）业务编码对应组织的降维码，1个经销商-多组织
   * @return k-curRuleMap的key（组织|层级编码）,v-经销商|终端编码ø集合
   */
  public static Map<String, Set<String>> findRelateBusiness(
      Map<String, String> curRuleMap, Map<String, Set<String>> mapBusinessRule) {
    if (curRuleMap.isEmpty() || mapBusinessRule.isEmpty()) {
      return Maps.newHashMap();
    }

    List<DimensionDictVo> list = Lists.newArrayList();
    for (Entry<String, Set<String>> item : mapBusinessRule.entrySet()) {
      for (String sub : item.getValue()) {
        if (StringUtils.isBlank(sub)) {
          continue;
        }
        final DimensionDictVo dictVo = new DimensionDictVo();
        dictVo.setDimensionType(item.getKey());
        dictVo.setDimensionDesc(sub);
        list.add(dictVo);
      }
    }
    // k -降维码,v-经销商编码集合
    Map<String, Set<String>> mapTerminalRuleConvert =
        list.stream()
            .collect(
                Collectors.groupingBy(
                    DimensionDictVo::getDimensionDesc,
                    Collectors.mapping(DimensionDictVo::getDimensionType, Collectors.toSet())));
    if (mapTerminalRuleConvert.isEmpty()) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> map = Maps.newHashMap();
    for (Entry<String, String> item : curRuleMap.entrySet()) {
      // 降维码
      String s1 = item.getValue();
      Set<String> cur = Sets.newHashSet();
      for (Entry<String, Set<String>> sub : mapTerminalRuleConvert.entrySet()) {
        String s2 = sub.getKey();
        if (s2.startsWith(s1)) {
          cur.addAll(sub.getValue());
        }
      }
      if (CollectionUtils.isEmpty(cur)) {
        continue;
      }
      map.put(item.getKey(), cur);
    }
    return map;
  }
}
