package com.biz.crm.dms.business.contract.local.repository.contract;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.contract.local.entity.contract.Contract;
import com.biz.crm.dms.business.contract.local.mapper.contract.ContractMapper;
import com.biz.crm.dms.business.contract.sdk.dto.contract.ContractCustomerDto;
import com.biz.crm.dms.business.contract.sdk.dto.contract.ContractDto;
import com.biz.crm.dms.business.contract.sdk.enums.ActApproveStatusExtendEnum;
import com.biz.crm.dms.business.contract.sdk.enums.ContractCancellationState;
import com.biz.crm.dms.business.contract.sdk.enums.EffectiveStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * @description: 合同 的 数据库访问类 {@link Contract}
 * @author: rentao
 * @date: 2021/11/16 14:00
 */
@Component
public class ContractRepository extends ServiceImpl<ContractMapper, Contract> {

  /**
   * 分页条件查询客户物料
   *
   * @param pageable 分页信息
   * @param dto      查询筛选条件
   * @return 分页数据
   */
  public Page<Contract> findByConditions(Page<Contract> pageable, @Param("dto") ContractDto dto) {
    return this.baseMapper.findByConditions(pageable, dto);
  }

  /**
   * 编码查询查询合同
   *
   * @param code 合同模板编码
   * @return
   */
  public Contract findByContractCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return this.lambdaQuery()
        .eq(Contract::getContractCode, code)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .one();
  }

  /**
   * 编码查询查询合同
   *
   * @param contractCodes 合同编码
   * @return
   */
  public List<Contract> findByContractCodes(List<String> contractCodes) {
    if (CollectionUtils.isEmpty(contractCodes)) {
      return null;
    }
    return this.lambdaQuery()
        .in(Contract::getContractCode, contractCodes)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .eq(Contract::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .list();
  }

  /**
   * 编码查询查询合同
   *
   * @param templateCodes 合同模板编码
   * @return
   */
  public List<Contract> findByTemplateCode(List<String> templateCodes) {
    if (CollectionUtils.isEmpty(templateCodes)) {
      return null;
    }
    return this.lambdaQuery()
        .in(Contract::getTemplateCode, templateCodes)
        .eq(Contract::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .list();
  }

  /**
   * 通过日期查询正常状态的数据
   *
   * @param dateTime (YYYY-mm-dd)时间之前数据
   * @return
   */
  public List<Contract> findByStartTime(String dateTime) {
    if (StringUtils.isEmpty(dateTime)) {
      return null;
    }
    return this.lambdaQuery()
        .le(Contract::getStartTime, dateTime)
        .ge(Contract::getEndTime, dateTime)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .eq(Contract::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(Contract::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Contract::getEffectiveStatus, EffectiveStatusEnum.EFFECTIVE_STAY.getDictCode())
        .eq(Contract::getCancellationState, ContractCancellationState.CANCELLATION.getDictCode())
        .eq(Contract::getContractStatus,  ActApproveStatusExtendEnum.SIGN_CONTRACT.getDictCode())
        .list();
  }


  /**
   * 通过日期查询过期数据
   *
   * @param dateTime (YYYY-mm-dd)时间之前数据
   * @return
   */
  public List<Contract> findByEndTime(String dateTime) {
    if (StringUtils.isEmpty(dateTime)) { return null; }
    return this.lambdaQuery()
        .le(Contract::getEndTime, dateTime)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .list();
  }


  /**
   * 未作废，
   * 已签约
   * 需要提醒
   * 通知日期为当前时间数据
   *
   * @param expireEminderDate (YYYY-mm-dd) 当前时间数据
   * @return
   */
  public List<Contract> findByExpireEminderDate(String expireEminderDate) {
    if (StringUtils.isEmpty(expireEminderDate)) {
      return null;
    }
    return this.lambdaQuery()
        .eq(Contract::getExpirationReminderDate, expireEminderDate)
        .eq(Contract::getContractStatus, ActApproveStatusExtendEnum.SIGN_CONTRACT.getKey())
        .eq(Contract::getCancellationState, Integer.valueOf(BooleanEnum.FALSE.getNumStr()))
        .eq(Contract::getExpirationReminder, BooleanEnum.TRUE.getNumStr())
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .list();
  }

  /**
   * 根据主键集合，修改 enable_status
   *
   * @param enable
   * @param ids
   */
  public void updateEnableStatusByIdIn(EnableStatusEnum enable, List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.lambdaUpdate()
        .set(Contract::getEnableStatus, enable.getCode())
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .in(Contract::getId, ids)
        .update();
  }

  /**
   * 根据合同编码，修改 合同状态
   *
   * @param contractStatus
   * @param contractCode
   */
  public void updateSignStateById(String contractStatus,String effectiveStatus, String contractCode) {
    if (StringUtils.isBlank(contractCode)) {
      return;
    }
    this.lambdaUpdate()
        .set(Contract::getContractStatus, contractStatus)
        .set(Contract::getEffectiveStatus, effectiveStatus)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .eq(Contract::getContractCode, contractCode)
        .update();
  }

  /**
   * 根据合同编码，修改 cancellationState
   *
   * @param cancellationState
   * @param contractCode
   */
  public void updateCancellationStateByContractCode(
      Integer cancellationState, String contractCode) {
    if (StringUtils.isBlank(contractCode)) {
      return;
    }
    this.lambdaUpdate()
        .set(Contract::getCancellationState, cancellationState)
        .eq(Contract::getContractCode, contractCode)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .update();
  }

  /**
   * 客户-合同分页查询
   *
   * @param pageable
   * @param contractCustomerDto
   * @return
   */
  public Page<Contract> findByContractCustomerDto(Page<Contract> pageable,
      ContractCustomerDto contractCustomerDto) {
    return this.baseMapper.findByContractCustomerDto(pageable, contractCustomerDto);
  }

  /**
   * 根据审批单号获取合同实体
   *
   * @param processNo 审批单号
   * @return 合同实体
   */
  public Contract findByContractProcessNo(String processNo) {
    if (StringUtils.isEmpty(processNo)) {
      return null;
    }
    return this.lambdaQuery().eq(Contract::getProcessNumber, processNo)
        .one();
  }

  /**
   * 根据合同编码，修改推送状态
   *
   * @param contractStatus
   * @param contractCode
   */
  public void updateContractStatusByContractCode(
      String contractStatus, String contractCode) {
    if (StringUtils.isBlank(contractCode)) {
      return;
    }
    this.lambdaUpdate()
        .set(Contract::getContractStatus, contractStatus)
        .eq(Contract::getContractCode, contractCode)
        .eq(Contract::getTenantCode, TenantUtils.getTenantCode())
        .update();
  }

  /**
   * 客户-待签署合同数量
   *
   * @param contractCustomerDto
   * @return
   */
  public Integer findUnsignedCountByCurrentCustomer(ContractCustomerDto contractCustomerDto) {
    contractCustomerDto.setTenantCode(TenantUtils.getTenantCode());
    return this.baseMapper.findUnsignedCountByCurrentCustomer(contractCustomerDto);
  }

  /**
   * 根据经销商编码查询生效的合同集合
   *
   * @param customerCode    客户代码
   * @param effectiveStatus 有效状态
   * @return {@link List}<{@link Contract}>
   */
  public List<Contract> findByCustomerCodeAndEffectiveStatus(String customerCode, String effectiveStatus) {
    List<Contract> list = this.lambdaQuery()
        .eq(Contract::getCusCode, customerCode)
        .eq(Contract::getEffectiveStatus, effectiveStatus)
        .list();
    return list;
  }

}
