package com.biz.crm.dms.business.contract.local.service.contract.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.contract.local.constant.ContractConstant;
import com.biz.crm.dms.business.contract.local.entity.contract.Contract;
import com.biz.crm.dms.business.contract.local.entity.contracthtml.ContractHtmlEntity;
import com.biz.crm.dms.business.contract.local.entity.contracttemplate.ContractTemplate;
import com.biz.crm.dms.business.contract.local.repository.contract.ContractRepository;
import com.biz.crm.dms.business.contract.local.repository.contracthtml.ContractHtmlRepository;
import com.biz.crm.dms.business.contract.local.service.contract.ContractService;
import com.biz.crm.dms.business.contract.local.service.contracttemplate.ContractTemplateService;
import com.biz.crm.dms.business.contract.local.service.contracttemplate.ContractTemplateVoService;
import com.biz.crm.dms.business.contract.local.utils.contract.ContractUtil;
import com.biz.crm.dms.business.contract.sdk.dto.contract.ContractDto;
import com.biz.crm.dms.business.contract.sdk.dto.contract.ContractEventDto;
import com.biz.crm.dms.business.contract.sdk.enums.ActApproveStatusExtendEnum;
import com.biz.crm.dms.business.contract.sdk.enums.ContractCancellationState;
import com.biz.crm.dms.business.contract.sdk.enums.EffectiveStatusEnum;
import com.biz.crm.dms.business.contract.sdk.event.contract.ContractEventListener;
import com.biz.crm.dms.business.contract.sdk.event.contract.ContractManagementEventListener;
import com.biz.crm.dms.business.contract.sdk.event.contractsign.ContractSignEventListener;
import com.biz.crm.dms.business.contract.sdk.register.contractelement.ContractElementRegister;
import com.biz.crm.dms.business.contract.sdk.service.ContractVoService;
import com.biz.crm.dms.business.contract.sdk.strategy.ContractSignStrategy;
import com.biz.crm.dms.business.contract.sdk.vo.contract.ContractVo;
import com.biz.crm.dms.business.contract.sdk.vo.contractelementdata.ContractElementDataVo;
import com.biz.crm.dms.business.contract.sdk.vo.contracttemplate.ContractTemplateVo;
import com.biz.crm.dms.business.interaction.sdk.dto.base.ScopeDto;
import com.biz.crm.dms.business.interaction.sdk.dto.notice.NoticeDto;
import com.biz.crm.dms.business.interaction.sdk.enums.NoticeType;
import com.biz.crm.dms.business.interaction.sdk.enums.ScopeType;
import com.biz.crm.dms.business.interaction.sdk.service.notice.NoticeVoService;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description: 合同
 * @author: rentao
 * @date: 2021/11/16 14:29
 */
@Service
public class ContractServiceImpl<T extends ContractElementDataVo> implements ContractService {

  @Autowired(required = false)
  private ContractRepository contractRepository;
  @Autowired(required = false)
  private ContractHtmlRepository contractHtmlRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private ContractTemplateVoService contractTemplateVoService;
  @Autowired(required = false)
  private ContractTemplateService contractTemplateService;
  @Autowired(required = false)
  private ContractVoService contractVoService;
  @Autowired(required = false)
  private NoticeVoService noticeVoService;
  @Autowired(required = false)
  private List<ContractEventListener> eventListeners;
  @Autowired(required = false)
  private List<ContractElementRegister> elementRegisters;
  @Autowired(required = false)
  private List<ContractSignStrategy> contractSignStrategies;
  @Autowired(required = false)
  private List<ContractSignEventListener> contractSignEventListeners;
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;
  @Autowired
  private LoginUserService loginUserService;
  
  /**
   * 分页条件查询合同
   *
   * @param pageable 分页参数
   * @param dto 客户物料查询参数
   * @return 客户物料实体
   */
  @Override
  public Page<Contract> findByConditions(Pageable pageable, ContractDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<Contract> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Contract> pageResult = contractRepository.findByConditions(page, dto);
    return pageResult;
  }

  /**
   * 新增合同
   *
   * @param contractDto
   * @return
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public Contract create(ContractDto contractDto) {
    /** 1，基础验证 2，验证合同模板 3，组装数据 4，保存要素 5，保存合同 6，发通知 */
    ContractUtil.validateAdd(contractDto);
    if (StringUtils.isNotBlank(contractDto.getContractCode())) {
      ContractVo contractVo = this.contractVoService
          .findDetailsByCode(contractDto.getContractCode());
      Validate.isTrue(Objects.isNull(contractVo), "合同编码已存在");
    }
    // 2
    ContractTemplateVo contractTemplateVo = this.validateContractTemplate(contractDto);
    // 3
    Contract entity = this.transformCreate(contractDto);
    // 4
    this.saveContractElementData(contractDto, contractTemplateVo);
    // 5
    this.contractRepository.save(entity);
    // 6 发通知
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      ContractVo contractVo = this.nebulaToolkitService
          .copyObjectByBlankList(contractDto, ContractVo.class, HashSet.class, ArrayList.class);
      this.eventListeners.forEach(
          eventListener -> {
            eventListener.onCreate(contractVo);
          });
    }
    //7 保存html
    this.saveHtml(contractDto.getHtmlStr(), entity.getContractCode());
    contractDto.setId(entity.getId());
    //8 保存日志
    ContractEventDto contractEventDto = new ContractEventDto();
    contractEventDto.setOnlyKey(entity.getId());
    contractEventDto.setNewest(contractDto);
    SerializableBiConsumer<ContractManagementEventListener, ContractEventDto> onCreate = ContractManagementEventListener::onCreate;
    this.nebulaNetEventClient
        .publish(contractEventDto, ContractManagementEventListener.class, onCreate);
    //9 审批发起
    String commitProcess = this.commitProcess(contractDto);
    entity.setProcessNumber(commitProcess);
    this.contractRepository.updateById(entity);
    return entity;
  }

  /**
   * 编辑合同
   *
   * @param contractDto
   * @return
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public Contract update(ContractDto contractDto) {
    /** 1，验证基础数据 2，模板验证 3，保存要素数据 4，数据组装计算验证 5，更新合同 6，事件通知 todo 注：当前没有设置 什么样的数据状态可以修改 */
    // 1
    ContractUtil.validateEdit(contractDto);
    // 2
    ContractTemplateVo contractTemplateVo = this.validateContractTemplate(contractDto);
    // 3
    this.updateContractElementData(contractDto, contractTemplateVo);
    // 4
    ContractVo contractOld = this.contractVoService.findById(contractDto.getId());
    Contract entity = this.transformUpdate(contractDto, contractOld);
    // 5
    this.contractRepository.updateById(entity);
    // 6
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      ContractVo newVo =
          this.nebulaToolkitService
              .copyObjectByBlankList(entity, ContractVo.class, HashSet.class, ArrayList.class);
      this.eventListeners.forEach(
          eventListener -> {
            eventListener.onUpdate(contractOld, newVo);
          });
    }
    //7编辑html
    this.contractHtmlRepository
        .updateByContractCode(contractDto.getHtmlStr(), contractDto.getContractCode());
    //8日志
    ContractEventDto contractEventDto = new ContractEventDto();
    contractEventDto.setOnlyKey(entity.getId());
    contractEventDto.setNewest(contractDto);
    contractEventDto.setOriginal(contractOld);
    SerializableBiConsumer<ContractManagementEventListener, ContractEventDto> onUpdate = ContractManagementEventListener::onUpdate;
    this.nebulaNetEventClient
        .publish(contractEventDto, ContractManagementEventListener.class, onUpdate);
    //9发起审批流程
    String commitProcess = this.commitProcess(contractDto);
    ProcessBusinessDto processBusiness = contractDto.getProcessBusiness();
    entity.setProcessNumber(commitProcess);
    entity.setProcessRemark(processBusiness.getRemark());
    this.contractRepository.updateById(entity);
    return entity;
  }

  /**
   * 批量启用
   *
   * @param ids
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    List<Contract> entities = this.contractRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(),
        "数据启用用个数不匹配");
    this.contractRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    // 启用合同模板事件通知
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      List<ContractVo> voList = (List<ContractVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(entities, Contract.class, ContractVo.class, HashSet.class,
              ArrayList.class);
      eventListeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
    //日志
    for (Contract contract : entities) {
      ContractEventDto productSingleEventDto = new ContractEventDto();
      productSingleEventDto.setOnlyKey(contract.getId());
      JSONObject jsonObjectOld = new JSONObject();
      JSONObject jsonObjectNew = new JSONObject();
      jsonObjectOld.put("enableStatus", contract.getEnableStatus());
      jsonObjectNew.put("enableStatus", EnableStatusEnum.ENABLE.getCode());
      productSingleEventDto.setOriginal(jsonObjectOld);
      productSingleEventDto.setNewest(jsonObjectNew);
      SerializableBiConsumer<ContractManagementEventListener, ContractEventDto> onEnable = ContractManagementEventListener::onEnable;
      this.nebulaNetEventClient.publish(productSingleEventDto, ContractManagementEventListener.class, onEnable);
    }
  }

  /**
   * 批量禁用
   *
   * @param ids
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<Contract> entities = this.contractRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(),
        "数据禁用个数不匹配");
    this.contractRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    // 禁用合同模板事件通知
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      List<ContractVo> voList =
          (List<ContractVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  entities, Contract.class, ContractVo.class, HashSet.class, ArrayList.class);
      eventListeners.forEach(
          listener -> {
            listener.onDisable(voList);
          });
    }
    //日志
    for (Contract contract : entities) {
      ContractEventDto productSingleEventDto = new ContractEventDto();
      productSingleEventDto.setOnlyKey(contract.getId());
      JSONObject jsonObjectOld = new JSONObject();
      JSONObject jsonObjectNew = new JSONObject();
      jsonObjectOld.put("enableStatus", contract.getEnableStatus());
      jsonObjectNew.put("enableStatus", EnableStatusEnum.DISABLE.getCode());
      productSingleEventDto.setOriginal(jsonObjectOld);
      productSingleEventDto.setNewest(jsonObjectNew);
      SerializableBiConsumer<ContractManagementEventListener, ContractEventDto> onDisable = ContractManagementEventListener::onDisable;
      this.nebulaNetEventClient
          .publish(productSingleEventDto, ContractManagementEventListener.class, onDisable);
    }
  }

  /**
   * 合同只能签约 正常状态的合同 （未过期，审批通过，未作废，未禁用，未删除）的合同 已推送 未过期，审批通过，未作废，未禁用，未删除
   *
   * @param contractCode
   * @author rentao
   * @date a
   */
  @Override
  @Transactional
  public void signContract(String contractCode) {
    ContractVo contractVo = this.contractVoService.findDetailsByCode(contractCode);
    Validate.isTrue(CollectionUtils.isNotEmpty(contractSignStrategies), "系统中没有签约方式实现,请检查");
    contractSignStrategies.forEach(s->{
     if(s.getContractSignTypeCode().equals(contractVo.getContractSignType())) {
       s.onRequestContractSign(contractVo);
     }
    });
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    String nowDate = format.format(new Date());
    Validate.isTrue(nowDate.compareTo(contractVo.getEndTime()) <= 0, "合同有效期已过不能签约！");
    String effectiveStatus;
    if (nowDate.compareTo(contractVo.getStartTime()) >= 0) {
      effectiveStatus = EffectiveStatusEnum.EFFECTIVE_ACTIVE.getDictCode();
    } else {
      effectiveStatus = EffectiveStatusEnum.EFFECTIVE_STAY.getDictCode();
    }
    this.contractRepository.updateSignStateById(ActApproveStatusExtendEnum.SIGN_CONTRACT.getKey(), effectiveStatus, contractCode);
    // 合同签约事件通知（如果是签约后生成数据则需要通知）
    if (CollectionUtils.isNotEmpty(this.contractSignEventListeners)) {
      contractVo.setEffectiveStatus(effectiveStatus);
      this.contractSignEventListeners.forEach(
          signEventListener -> {
            signEventListener.onSignContract(contractVo);
          });
    }
  }

  /**
   * 合同作废
   *
   * @param contractCode
   */
  @Override
  @Transactional
  public void cancellation(String contractCode) {
    ContractVo contractVo = this.contractVoService.findDetailsByCode(contractCode);
    Validate.notNull(contractVo, "合同不存在");
    Integer cancellationState = contractVo.getCancellationState();
    this.contractRepository.updateCancellationStateByContractCode(
        ContractCancellationState.NOT_CANCELLATION.getKey(), contractCode);
    contractVo.setCancellationState(ContractCancellationState.NOT_CANCELLATION.getKey());
    // 合同作废事件通知
    if (CollectionUtils.isNotEmpty(this.contractSignEventListeners)) {
      this.contractSignEventListeners.forEach(
          signEventListener -> {
            signEventListener.onCancellation(contractVo);
          });
    }
    //日志
    ContractEventDto productSingleEventDto = new ContractEventDto();
    productSingleEventDto.setOnlyKey(contractVo.getId());
    JSONObject jsonObjectOld = new JSONObject();
    JSONObject jsonObjectNew = new JSONObject();
    jsonObjectOld.put("cancellationState", cancellationState);
    jsonObjectNew.put("cancellationState", ContractCancellationState.NOT_CANCELLATION.getKey());
    productSingleEventDto.setOriginal(jsonObjectOld);
    productSingleEventDto.setNewest(jsonObjectNew);
    SerializableBiConsumer<ContractManagementEventListener, ContractEventDto> onCancellation = ContractManagementEventListener::onCancellation;
    this.nebulaNetEventClient
        .publish(productSingleEventDto, ContractManagementEventListener.class, onCancellation);

  }

  @Override
  public void handlePush(String contractCode) {
    ContractVo contractVo = this.contractVoService.findDetailsByCode(contractCode);
    Validate.notNull(contractVo, "合同不存在");
    Validate.isTrue(
        ActApproveStatusExtendEnum.PUSH_NO_CONTRACT.getKey().equals(contractVo.getContractStatus())
            ||
            ActApproveStatusExtendEnum.PUSH_FAIL_CONTRACT.getKey()
                .equals(contractVo.getContractStatus()), "合同不是可推送状态！");
    Validate.isTrue(
        Integer.valueOf(BooleanEnum.FALSE.getNumStr()).equals(contractVo.getCancellationState()),
        "合同已作废");
    Validate
        .isTrue(EnableStatusEnum.ENABLE.getCode().equals(contractVo.getEnableStatus()), "合同已禁用");
    Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(contractVo.getDelFlag()), "合同已删除");
    Validate.isTrue(
        !EffectiveStatusEnum.EFFECTIVE_LOSE.getKey().equals(contractVo.getEffectiveStatus()),
        "合同已失效");
    this.contractRepository
        .updateContractStatusByContractCode(ActApproveStatusExtendEnum.SIGN_NO_LEAGUE.getKey(),
            contractCode);
  }

  /**
   * 合同过期（定时任务） 查询未删除未过期 数据中 endTime 小于当前年月日的数据过期
   */
  @Override
  @Transactional
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "合同过期")
  public void expire() {
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(new Date());
    calendar.set(Calendar.DAY_OF_YEAR, calendar.get(Calendar.DAY_OF_YEAR) - 1);
    List<Contract> contractList =
        this.contractRepository.findByEndTime(format.format(calendar.getTime()));
    // 合同过期
    if (CollectionUtils.isNotEmpty(contractList)) {
      contractList.forEach(contract -> {
        contract.setEffectiveStatus(EffectiveStatusEnum.EFFECTIVE_LOSE.getDictCode());
      });
      this.contractRepository.updateBatchById(contractList);
    }
    // 过期通知模块
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      List<ContractVo> voList =
          (List<ContractVo>) this.nebulaToolkitService
              .copyCollectionByWhiteList(contractList, Contract.class, ContractVo.class,
                  HashSet.class, ArrayList.class);
      eventListeners.forEach(
          listener -> {
            listener.onMaturity(voList);
          });
    }
  }


  /**
   *
   * 合同生效
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "合同生效")
  public void effective() {
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(new Date());
    calendar.set(Calendar.DAY_OF_YEAR, calendar.get(Calendar.DAY_OF_YEAR));
    List<Contract> contractList = this.contractRepository.findByStartTime(format.format(calendar.getTime()));
    // 合同生效
    if (CollectionUtils.isNotEmpty(contractList)) {
      contractList.forEach(contract -> {
        contract.setEffectiveStatus(EffectiveStatusEnum.EFFECTIVE_ACTIVE.getDictCode());
      });
      this.contractRepository.updateBatchById(contractList);
      // 生效通知模块
      if (CollectionUtils.isNotEmpty(eventListeners)) {
        List<ContractVo> voList =
            (List<ContractVo>) this.nebulaToolkitService
                .copyCollectionByWhiteList(contractList, Contract.class, ContractVo.class,
                    HashSet.class, ArrayList.class);
        eventListeners.forEach(
            listener -> {
              listener.onEffective(voList);
            });
      }
    }
  }

  /**
   * 合同到期提醒（定时任务） 查询已生效数据中 ExpireEminderDate 等于当前年月日的数据提醒
   */
  @Override
  @Transactional
  @DynamicTaskService(cornExpression = "0 0 1 * * ?", taskDesc = "合同到期提醒(定时任务)")
  public void expireEminder() {
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    String today = format.format(new Date());
    List<Contract> contracts = this.contractRepository.findByExpireEminderDate(today);
    this.transformContractToNoticeDto(contracts);
  }

  /**
   * 验证模板信息并返回模板
   *
   * @param contractDto
   * @return
   */
  private ContractTemplateVo validateContractTemplate(ContractDto contractDto) {
    Validate.isTrue(CollectionUtils.isNotEmpty(elementRegisters), "系统中没有注册要素,请检查");
    ContractTemplateVo contractTemplateVo =
        this.contractTemplateVoService.findDetailsByCode(contractDto.getTemplateCode());
    Validate.notNull(contractTemplateVo, "没有获取到模板");
    Validate.isTrue(
        CollectionUtils.isNotEmpty(contractTemplateVo.getTemplateElements()), "没有获取到模板要素列表");
    return contractTemplateVo;
  }

  /**
   * 保存要素内容
   *
   * @param contractDto
   * @param contractTemplateVo
   */
  private void updateContractElementData(
      ContractDto contractDto, ContractTemplateVo contractTemplateVo) {
    Map<String, JSONObject> data = contractDto.getElementDataMap();
    contractTemplateVo
        .getTemplateElements()
        .forEach(
            templateElement -> {
              List<ContractElementRegister> collect =
                  elementRegisters.stream()
                      .filter(
                          elementRegister -> elementRegister.getContractElementCode()
                                  .equals(templateElement.getElementCode()))
                      .collect(Collectors.toList());
              Validate.isTrue(CollectionUtils.isNotEmpty(collect), "要素不存在");
              ContractElementRegister contractElementRegister = collect.get(0);
              JSONObject jsonObject = data.get(contractElementRegister.getContractElementCode());
              Validate.notNull(jsonObject, "要素名称【%s】内容为空",
                  contractElementRegister.getContractElementName());
              Class<T> elementClass = contractElementRegister.getContractElementClass();
              contractElementRegister.onRequestContractUpdate(
                  contractDto.getContractCode(),
                  JSON.parseObject(jsonObject.toJSONString(), elementClass),
                  templateElement.getIndexCode());
            });
  }

  /**
   * 编辑要素内容
   *
   * @param contractDto
   * @param contractTemplateVo
   */
  private void saveContractElementData(
      ContractDto contractDto, ContractTemplateVo contractTemplateVo) {
    Map<String, JSONObject> data = contractDto.getElementDataMap();
    contractTemplateVo
        .getTemplateElements()
        .forEach(
            templateElement -> {
              List<ContractElementRegister> collect =
                  elementRegisters.stream()
                      .filter(
                          elementRegister ->
                              elementRegister
                                  .getContractElementCode()
                                  .equals(templateElement.getElementCode()))
                      .collect(Collectors.toList());
              Validate.isTrue(CollectionUtils.isNotEmpty(collect), "要素不存在");
              ContractElementRegister contractElementRegister = collect.get(0);
              JSONObject jsonObject = data.get(contractElementRegister.getContractElementCode());
              Validate.isTrue(Objects.nonNull(jsonObject),
                  "合同中:" + templateElement.getElementName() + "数据为空");
              Class<T> elementClass = contractElementRegister.getContractElementClass();
              contractElementRegister.onRequestContractCreate(
                  contractDto.getContractCode(),
                  JSON.parseObject(jsonObject.toJSONString(), elementClass),
                  templateElement.getIndexCode());
            });
  }

  /**
   * 合同集合转换成公告Dto 并保存
   *
   * @param contracts
   * @author rentao
   * @date a
   */
  private void transformContractToNoticeDto(List<Contract> contracts) {
    if (CollectionUtils.isNotEmpty(contracts)) {
      Map<String, List<Contract>> listMap =
          contracts.stream().collect(Collectors.groupingBy(Contract::getTemplateCode));
      Set<String> templateCodes = listMap.keySet();
      List<ContractTemplate> contractTemplates =
          contractTemplateService.findByTemplateCodes(templateCodes);
      if (CollectionUtils.isNotEmpty(contractTemplates)) {
        Map<String, ContractTemplate> contractTemplateMap =
            contractTemplates.stream()
                .collect(
                    Collectors.toMap(
                        ContractTemplate::getTemplateCode, t -> t, (key1, key2) -> key2));
        List<NoticeDto> noticeDtos = new ArrayList<>();
        Date date = new Date();
        listMap.forEach(
            (k, v) -> {
              ContractTemplate contractTemplate = contractTemplateMap.get(k);
              if (Objects.nonNull(contractTemplate)) {
                // 有效天数
                Integer effectiveDay = contractTemplate.getEffectiveDay();
                v.forEach(
                    contract -> {
                      NoticeDto noticeDto = new NoticeDto();
                      noticeDto.setTitle(ContractConstant.NOTICE_TITLE);
                      noticeDto.setType(NoticeType.INFORM.getKey());
                      noticeDto.setStartTime(this.getDate(date, 0));
                      noticeDto.setEndTime(this.getDate(date, -effectiveDay));
                      noticeDto.setIndexPopup(contractTemplate.getIndexPopup());
                      noticeDto.setPopupType(contractTemplate.getPopupType());
                      noticeDto.setContent(
                          contract.getContractCode() + contractTemplate.getContext());
                      ScopeDto scopeDto = new ScopeDto();
                      scopeDto.setScopeCode(contract.getCusCode());
                      scopeDto.setScopeType(ScopeType.CUSTOMER.name());
                      scopeDto.setContainFlag(true);
                      noticeDto.setScopeList(Collections.singletonList(scopeDto));
                      noticeDtos.add(noticeDto);
                    });
              }
            });
        noticeVoService.create(noticeDtos);
      }
    }
  }

  /**
   * 获取时间
   *
   * @param date
   * @param num  为负数拿到结束时间 为正数拿到结束时间
   * @author rentao
   * @date a
   */
  private Date getDate(Date date, int num) {
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(date);
    calendar.set(Calendar.DAY_OF_YEAR, calendar.get(Calendar.DAY_OF_YEAR) - num);
    if (num >= 0) {
      calendar.set(Calendar.HOUR_OF_DAY, 0);
      calendar.set(Calendar.MINUTE, 0);
      calendar.set(Calendar.SECOND, 0);
    } else {
      calendar.set(Calendar.HOUR_OF_DAY, 23);
      calendar.set(Calendar.MINUTE, 59);
      calendar.set(Calendar.SECOND, 59);
    }
    return calendar.getTime();
  }

  /**
   * 新增数据填充
   *
   * @param contractDto
   * @author rentao
   * @date
   */
  private Contract transformCreate(ContractDto contractDto) {
    Contract entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            contractDto, Contract.class, HashSet.class, ArrayList.class);
    if (StringUtils.isBlank(entity.getContractCode())) {
      entity.setContractCode(
          this.generateCodeService.generateCode(ContractConstant.CONTRACT_CODE, 1).get(0));
    } else {
      Validate.isTrue(entity.getContractCode().length() < 128, "合同编码长度不能超过128");
    }
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //是否作废
    entity.setCancellationState(Integer.valueOf(BooleanEnum.FALSE.getNumStr()));
    //合同状态
    entity.setContractStatus(ProcessStatusEnum.COMMIT.getDictCode());
    //生效状态
    entity.setEffectiveStatus(EffectiveStatusEnum.EFFECTIVE_STAY.getDictCode());
    contractDto.setContractCode(entity.getContractCode());
    ContractUtil.getExpireEminderDate(entity);
    return entity;
  }


  /**
   * 修改数据填充
   *
   * @param contractDto
   * @author rentao
   * @date
   */
  private Contract transformUpdate(ContractDto contractDto, ContractVo contractOld) {
    Contract entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            contractDto, Contract.class, HashSet.class, ArrayList.class);
    Validate.isTrue(Objects.nonNull(contractOld), "数据不存在");
    entity.setTenantCode(contractOld.getTenantCode());
    entity.setDelFlag(contractOld.getDelFlag());
    entity.setEnableStatus(contractOld.getEnableStatus());
    entity.setCancellationState(contractOld.getCancellationState());
    entity.setContractStatus(ProcessStatusEnum.COMMIT.getDictCode());
    ContractUtil.getExpireEminderDate(entity);
    return entity;
  }

  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  /**
   * 合同审批
   *
   * @param contractDto 合同审批
   */
  private String commitProcess(ContractDto contractDto) {
    ProcessBusinessDto businessProcessDto = contractDto.getProcessBusiness();
    businessProcessDto.setBusinessNo(contractDto.getContractCode());
    businessProcessDto.setBusinessFormJson(JsonUtils.obj2JsonString(contractDto));
    businessProcessDto.setBusinessCode(ContractConstant.CONTRACT_PROCESS_NAME);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(businessProcessDto);
    return processBusinessVo.getProcessNo();
  }

  private void saveHtml(String htmlStr, String contractCode) {
    if (StringUtils.isNotBlank(htmlStr) && StringUtils.isNotBlank(contractCode)) {
      ContractHtmlEntity contractHtmlEntity = new ContractHtmlEntity();
      contractHtmlEntity.setHtmlStr(htmlStr);
      contractHtmlEntity.setContractCode(contractCode);
      this.contractHtmlRepository.save(contractHtmlEntity);
    }
  }

}
