package com.biz.crm.dms.business.contract.local.service.contractarea.internal;

import com.biz.crm.dms.business.contract.local.entity.contractarea.ContractArea;
import com.biz.crm.dms.business.contract.local.repository.contractarea.ContractAreaRepository;
import com.biz.crm.dms.business.contract.local.service.contractarea.ContractAreaDataVoService;
import com.biz.crm.dms.business.contract.sdk.vo.contractarea.ContractAreaVo;
import com.biz.crm.dms.business.contract.sdk.vo.contractelementdata.ContractAreaDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * @description: 合同销售区域 service
 * @author: rentao
 * @date: 2021/12/17 16:11
 */
@Slf4j
@Service
public class ContractAreaDataVoServiceImpl implements ContractAreaDataVoService {

  @Autowired(required = false) private ContractAreaRepository contractAreaRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  /**
   * 合同编码查询
   *
   * @param contractcode
   * @author rentao
   * @date a
   */
  @Override
  public ContractAreaDataVo findByContractCode(String contractcode) {
    if (StringUtils.isBlank(contractcode)) {
      return null;
    }
    List<ContractArea> contractAreas = this.contractAreaRepository.findByContractCode(contractcode);
    if (CollectionUtils.isEmpty(contractAreas)) {
      return null;
    }
    List<ContractAreaVo> voList =
        (List<ContractAreaVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                contractAreas,
                ContractArea.class,
                ContractAreaVo.class,
                HashSet.class,
                ArrayList.class);
    ContractAreaDataVo vo = new ContractAreaDataVo();
    vo.setContractAreaVos(voList);
    return vo;
  }

  /**
   * 要素内容保存
   *
   * @param contractCode
   * @param contractAreaElementDataVo
   * @param indexCode
   * @author rentao
   */
  @Override
  @Transactional
  public ContractAreaDataVo createContractDetailsElement(
      String contractCode, ContractAreaDataVo contractAreaDataVo, Integer indexCode) {
    // 校验入参
    this.ValidateSaveOrUpdate(contractCode, contractAreaDataVo);
    List<ContractAreaVo> contractAreaVos = contractAreaDataVo.getContractAreaVos();
    List<ContractArea> voList =
        (List<ContractArea>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                contractAreaVos,
                ContractAreaVo.class,
                ContractArea.class,
                HashSet.class,
                ArrayList.class);
    this.contractAreaRepository.saveBatch(voList);
    return contractAreaDataVo;
  }

  /**
   * 要素内容修改
   *
   * @param contractCode
   * @param contractAreaDataVo
   * @param indexCode
   * @author rentao
   */
  @Override
  @Transactional
  public ContractAreaDataVo updateContractDetailsElement(
      String contractCode, ContractAreaDataVo contractAreaDataVo, Integer indexCode) {
    // 校验入参
    this.ValidateSaveOrUpdate(contractCode, contractAreaDataVo);
    this.contractAreaRepository.deleteByContractCode(contractCode);
    List<ContractAreaVo> contractAreaVos = contractAreaDataVo.getContractAreaVos();
    List<ContractArea> contractAreas =
        (List<ContractArea>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                contractAreaVos,
                ContractAreaVo.class,
                ContractArea.class,
                HashSet.class,
                ArrayList.class);
    contractAreas.forEach(contractArea -> {
      contractArea.setId(null);
    });
    this.contractAreaRepository.saveBatch(contractAreas);
    return contractAreaDataVo;
  }

  /**
   * 入参校验
   *
   * @param contractCode
   * @param contractAreaDataVo
   * @author rentao
   * @date
   */
  private void ValidateSaveOrUpdate(String contractCode, ContractAreaDataVo contractAreaDataVo) {
    // 校验入参
    Validate.notBlank(contractCode, "合同编码不能为空");
    Validate.notNull(contractAreaDataVo, "合同销售区域为空");
    Validate.isTrue(
        CollectionUtils.isNotEmpty(contractAreaDataVo.getContractAreaVos()), "合同销售区域不能为空");
    List<ContractAreaVo> contractAreaVos = contractAreaDataVo.getContractAreaVos();
    contractAreaVos.forEach(
        contractAreaVo -> {
          Validate.notBlank(contractAreaVo.getProvinceCode(), "省编码不能为空");
          Validate.notBlank(contractAreaVo.getCityCode(), "市编码不能为空");
          Validate.notBlank(contractAreaVo.getDistrictCode(), "区县编码不能为空");
          contractAreaVo.setContractCode(contractCode);
        });
  }
}
