package com.biz.crm.dms.business.contract.local.service.contractfiles.internal;

import com.biz.crm.dms.business.contract.local.entity.contractfiles.ContractFiles;
import com.biz.crm.dms.business.contract.local.repository.contractfiles.ContractFilesRepository;
import com.biz.crm.dms.business.contract.local.service.contractfiles.ContractFilesDataVoService;
import com.biz.crm.dms.business.contract.sdk.vo.contractelementdata.ContractFilesDataVo;
import com.biz.crm.dms.business.contract.sdk.vo.contractfiles.ContractFilesVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * @description: 合同附件
 * @author: rentao
 * @date: 2021/11/22 13:22
 */
@Slf4j
@Service
public class ContractFilesDataVoServiceImpl implements ContractFilesDataVoService {

  @Autowired(required = false) private ContractFilesRepository contractFilesRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  /**
   * 查询要素内容
   *
   * @param contractCode 合同编码
   * @author rentao
   * @date
   */
  @Override
  public ContractFilesDataVo findByContractCode(String contractCode) {
    List<ContractFiles> contractFiles = this.contractFilesRepository.findByContractCode(contractCode);
    if (CollectionUtils.isEmpty(contractFiles)) {
      return null;
    }
    List<ContractFilesVo> voList =
        (List<ContractFilesVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                contractFiles,
                ContractFiles.class,
                ContractFilesVo.class,
                HashSet.class,
                ArrayList.class);
    ContractFilesDataVo vo = new ContractFilesDataVo();
    vo.setContractFilesVos(voList);
    return vo;
  }

  /**
   * 新增合同要素
   *
   * @param contractCode 合同编码
   * @param contractFilesDateVo 文件要素
   * @param indexCode 文件要素在合同中的顺序
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public ContractFilesDataVo createContractFilesElement(
      String contractCode, ContractFilesDataVo contractFilesDateVo, Integer indexCode) {
    // 校验入参
    this.ValidateSaveOrUpdate(contractCode, contractFilesDateVo);
    List<ContractFilesVo> contractFilesVos = contractFilesDateVo.getContractFilesVos();
    List<ContractFiles> voList =
        (List<ContractFiles>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                contractFilesVos,
                ContractFilesVo.class,
                ContractFiles.class,
                HashSet.class,
                ArrayList.class);
    this.contractFilesRepository.saveBatch(voList);
    return contractFilesDateVo;
  }

  /**
   * 编辑合同要素
   *
   * @param contractCode 合同编码
   * @param contractFilesDateVo 文件要素
   * @param indexCode 文件要素在合同中的顺序
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public ContractFilesDataVo updateContractFilesElement(
      String contractCode, ContractFilesDataVo contractFilesDateVo, Integer indexCode) {
    // 校验入参
    this.ValidateSaveOrUpdate(contractCode, contractFilesDateVo);
    this.contractFilesRepository.deleteByContractCode(contractCode);
    List<ContractFilesVo> contractFilesVos = contractFilesDateVo.getContractFilesVos();
    List<ContractFiles> voList =
       (List<ContractFiles>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                contractFilesVos,
                ContractFilesVo.class,
                ContractFiles.class,
                HashSet.class,
                ArrayList.class);
    voList.forEach(contractFiles -> {
      contractFiles.setId(null);
    });
    this.contractFilesRepository.saveBatch(voList);
    return contractFilesDateVo;
  }

  private void ValidateSaveOrUpdate(String contractCode, ContractFilesDataVo contractFilesDateVo) {
    // 校验入参
    Validate.notBlank(contractCode, "合同编码不能为空");
    Validate.notNull(contractFilesDateVo, "合同附件为空");
    Validate.isTrue(
        CollectionUtils.isNotEmpty(contractFilesDateVo.getContractFilesVos()), "合同附件不能为空");
    List<ContractFilesVo> contractFilesVos = contractFilesDateVo.getContractFilesVos();
    contractFilesVos.forEach(
        contractFilesVo -> {
          Validate.notBlank(contractFilesVo.getOriginalFileName(), "文件名不能为空");
          Validate.notBlank(contractFilesVo.getFileCode(), "文件唯一标识不能为空");
          contractFilesVo.setContractCode(contractCode);
        });
  }
}
