package com.biz.crm.dms.business.contract.local.service.contractliability.internal;

import com.biz.crm.dms.business.contract.local.entity.contractliability.ContractLiability;
import com.biz.crm.dms.business.contract.local.repository.contractliability.ContractLiabilityRepository;
import com.biz.crm.dms.business.contract.local.service.contractliability.ContractLiabilityDataVoService;
import com.biz.crm.dms.business.contract.sdk.vo.contractelementdata.ContractLiabilityDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Objects;

/**
 * @description: 合同要素 违约责任
 * @author: rentao
 * @date: 2021/11/22 16:22
 */
@Slf4j
@Service
public class ContractLiabilityDataVoServiceImpl implements ContractLiabilityDataVoService {

  @Autowired(required = false) private ContractLiabilityRepository contractLiabilityRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  /**
   * 查询内容
   *
   * @param contractcode 合同编码
   * @author rentao
   * @date
   */
  @Override
  public ContractLiabilityDataVo findByContractCode(String contractcode) {
    ContractLiability contractLiability =
        contractLiabilityRepository.findByContractCode(contractcode);
    if (Objects.isNull(contractLiability)) {
      return null;
    }
    ContractLiabilityDataVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            contractLiability, ContractLiabilityDataVo.class, HashSet.class, ArrayList.class);
    return vo;
  }

  /**
   * 新增合同要素
   *
   * @param contractCode            合同编码
   * @param contractLiabilityDataVo 违约责任要素
   * @param indexCode               违约责任要素在合同中的顺序
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public ContractLiabilityDataVo createContractLiability(
      String contractCode, ContractLiabilityDataVo contractLiabilityDataVo, Integer indexCode) {
    // 校验入参
    this.validateSaveOrUpdate(contractCode, contractLiabilityDataVo);
    ContractLiability contractLiability =
        this.nebulaToolkitService.copyObjectByWhiteList(
            contractLiabilityDataVo, ContractLiability.class, HashSet.class, ArrayList.class);
    contractLiability.setContractCode(contractCode);
    this.contractLiabilityRepository.save(contractLiability);
    return contractLiabilityDataVo;
  }

  /**
   * 编辑合同要素
   *
   * @param contractCode            合同编码
   * @param contractLiabilityDataVo 违约责任要素
   * @param indexCode               违约责任要素在合同中的顺序
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public ContractLiabilityDataVo updateContractLiability(
      String contractCode, ContractLiabilityDataVo contractLiabilityDataVo, Integer indexCode) {
    // 校验入参
    this.validateSaveOrUpdate(contractCode, contractLiabilityDataVo);
    ContractLiability liability =
        this.contractLiabilityRepository.findByContractCode(contractCode);
    Validate.notNull(liability, "合同违约责任不存在");
    ContractLiability contractLiability =
        this.nebulaToolkitService.copyObjectByWhiteList(
            contractLiabilityDataVo, ContractLiability.class, HashSet.class, ArrayList.class);
    contractLiability.setContractCode(contractCode);
    contractLiability.setId(liability.getId());
    this.contractLiabilityRepository.updateById(contractLiability);
    return contractLiabilityDataVo;
  }

  private void validateSaveOrUpdate(
      String contractCode, ContractLiabilityDataVo contractLiabilityDataVo) {
    // 校验入参
    Validate.notBlank(contractCode, "合同编码不能为空");
    Validate.notNull(contractLiabilityDataVo, "合同违约责任为空");
    Validate.notBlank(contractLiabilityDataVo.getContractLiability(), "合同违约责任为空");
    Validate.isTrue(
        contractLiabilityDataVo.getContractLiability().length() <= 20000, "合同违约责任长度不能超过20000");
  }
}
