package com.biz.crm.dms.business.contract.local.service.contracttemplate.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.contract.local.constant.ContractConstant;
import com.biz.crm.dms.business.contract.local.entity.contracttemplate.ContractTemplate;
import com.biz.crm.dms.business.contract.local.repository.contracttemplate.ContractTemplateRepository;
import com.biz.crm.dms.business.contract.sdk.service.ContractVoService;
import com.biz.crm.dms.business.contract.local.service.contracttemplateelement.ContractTemplateElementVoService;
import com.biz.crm.dms.business.contract.local.service.contracttemplate.ContractTemplateService;
import com.biz.crm.dms.business.contract.local.utils.contracttemplate.ContractTemplateUtil;
import com.biz.crm.dms.business.contract.sdk.dto.contracttemplate.ContractTemplateDto;
import com.biz.crm.dms.business.contract.sdk.dto.contracttemplateelement.ContractTemplateElementDto;
import com.biz.crm.dms.business.contract.sdk.event.contracttemplate.ContractTemplateEventListener;
import com.biz.crm.dms.business.contract.sdk.register.contractelement.ContractElementRegister;
import com.biz.crm.dms.business.contract.sdk.vo.contract.ContractVo;
import com.biz.crm.dms.business.contract.sdk.vo.contracttemplate.ContractTemplateVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 合同模板
 * @author: rentao
 * @date: 2021/11/16 11:29
 */
@Slf4j
@Service
public class ContractTemplateServiceImpl implements ContractTemplateService {

  @Autowired(required = false)
  private ContractTemplateRepository contractTemplateRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private ContractTemplateElementVoService contractTemplateElementVoService;

  @Autowired(required = false)
  private ContractVoService contractVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<ContractTemplateEventListener> eventListeners;

  @Autowired(required = false)
  private List<ContractElementRegister> elementRegisters;

  /**
   * 分页条件查询合同模板
   *
   * @param pageable 分页参数
   * @param dto      客户物料查询参数
   * @return 客户物料实体
   */
  @Override
  public Page<ContractTemplate> findByConditions(Pageable pageable, ContractTemplateDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<ContractTemplate> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<ContractTemplate> pageResult = this.contractTemplateRepository.findByConditions(page, dto);
    return pageResult;
  }

  /**
   * 新增合同模板
   *
   * @param dto
   * @return
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public ContractTemplate create(ContractTemplateDto dto) {
    List<ContractTemplateElementDto> modelDtoList = dto.getTemplateElements();
    // 验证
    ContractTemplateUtil.validateAdd(dto);
    this.validateName(dto.getTemplateName(), null);
    dto.getTemplateElements()
        .forEach(
            s -> {
              this.validateContractElement(s.getElementCode());
            });
    // 合同模板编码
    String templateCode = generateCodeService.generateCode(ContractConstant.CONTRACT_TEMPLATE_CODE, 1).get(0);

    ContractTemplate contractTemplate = this.contractTemplateRepository
        .findByContractTemplateCode(templateCode);
    Validate.isTrue(Objects.isNull(contractTemplate), "模板编码已存在！");
    dto.setTemplateCode(templateCode);
    // 合同dto转换Entity
    ContractTemplate entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, ContractTemplate.class, HashSet.class, ArrayList.class);
    entity.setId(null);
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.contractTemplateRepository.save(entity);
    // 保存映射关系
    this.contractTemplateElementVoService.saveBatch(modelDtoList, dto.getTemplateCode());
    // 保存合同模板事件通知
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      ContractTemplateVo vo =
          this.nebulaToolkitService.copyObjectByBlankList(
              dto, ContractTemplateVo.class, HashSet.class, ArrayList.class);
      this.eventListeners.forEach(
          eventListener -> {
            eventListener.onCreate(vo);
          });
    }
    return entity;
  }

  /**
   * 编辑合同模板
   *
   * @param dto
   * @return
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public ContractTemplate update(ContractTemplateDto dto) {
    List<ContractTemplateElementDto> modelDtoList = dto.getTemplateElements();
    // 验证
    ContractTemplateUtil.validateEdit(dto);
    this.validateName(dto.getTemplateName(), dto.getId());
    dto.getTemplateElements()
        .forEach(
            s -> {
              this.validateContractElement(s.getElementCode());
            });
    Validate.isTrue(
        CollectionUtils.isEmpty(
            this.contractVoService.findByTemplateCode(
                Collections.singletonList(dto.getTemplateCode()))),
        "合同模板已存在合同,不能修改合同模板");
    // 保存
    ContractTemplate entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, ContractTemplate.class, HashSet.class, ArrayList.class);
    this.contractTemplateRepository.updateById(entity);
    // 保存映射关系
    this.contractTemplateElementVoService.updateBatch(modelDtoList, dto.getTemplateCode());
    // 编辑合同模板事件通知
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      ContractTemplateVo oldVo =
          this.nebulaToolkitService.copyObjectByWhiteList(
              entity, ContractTemplateVo.class, HashSet.class, ArrayList.class);
      ContractTemplateVo newVo =
          this.nebulaToolkitService.copyObjectByBlankList(
              dto, ContractTemplateVo.class, HashSet.class, ArrayList.class);
      this.eventListeners.forEach(
          eventListener -> {
            eventListener.onUpdate(oldVo, newVo);
          });
    }
    return entity;
  }

  /**
   * 批量启用
   *
   * @param ids
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    List<ContractTemplate> entities = this.contractTemplateRepository.listByIds(ids);
    Validate.isTrue(
        CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.contractTemplateRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    // 启用合同模板事件通知
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      List<ContractTemplateVo> voList =
          (List<ContractTemplateVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  entities,
                  ContractTemplate.class,
                  ContractTemplateVo.class,
                  HashSet.class,
                  ArrayList.class);
      eventListeners.forEach(
          listener -> {
            listener.onEnable(voList);
          });
    }
  }

  /**
   * 批量禁用
   *
   * @param ids
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<ContractTemplate> entities = this.contractTemplateRepository.listByIds(ids);
    Validate.isTrue(
        CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.contractTemplateRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    // 禁用合同模板事件通知
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      List<ContractTemplateVo> voList =
          (List<ContractTemplateVo>)
              nebulaToolkitService.copyCollectionByWhiteList(
                  entities,
                  ContractTemplate.class,
                  ContractTemplateVo.class,
                  HashSet.class,
                  ArrayList.class);
      eventListeners.forEach(
          listener -> {
            listener.onDisable(voList);
          });
    }
  }

  /**
   * 合同模板编码批量查询合同模板
   *
   * @param templateCodes
   * @author rentao
   * @date a
   */
  @Override
  public List<ContractTemplate> findByTemplateCodes(Set<String> templateCodes) {
    return this.contractTemplateRepository.findByTemplateCodes(templateCodes);
  }

  /**
   * 删除模板
   *
   * @param ids
   */
  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待删除的数据主键不能为空");
    List<ContractTemplate> entities = this.contractTemplateRepository.listByIds(ids);
    Validate.isTrue(
        CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    List<String> templateCodes = entities.stream().map(ContractTemplate::getTemplateCode)
        .collect(Collectors.toList());
    List<ContractVo> contractVos = this.contractVoService.findByTemplateCode(templateCodes);
    Validate.isTrue(CollectionUtils.isEmpty(contractVos), "模板已生成合同不能删除");
    this.contractTemplateRepository.updateDelFlagStatusByIdIn(DelFlagStatusEnum.DELETE, ids);
    // 启用合同模板事件通知
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      List<ContractTemplateVo> voList =
          (List<ContractTemplateVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  entities,
                  ContractTemplate.class,
                  ContractTemplateVo.class,
                  HashSet.class,
                  ArrayList.class);
      eventListeners.forEach(
          listener -> {
            listener.onDelete(voList);
          });
    }
  }

  /**
   * 校验模板名称是否重复
   *
   * @param name
   * @param id
   */
  private void validateName(String name, String id) {
    if (StringUtils.isBlank(name)) {
      return;
    }
    List<ContractTemplate> entities =
        this.contractTemplateRepository.findByContractTemplateInIds(name, id);
    Validate.isTrue(CollectionUtils.isEmpty(entities), "模板名称【%s】重复", name);
  }

  /**
   * 校验合同要素模板
   *
   * @param contractElementCode
   */
  private void validateContractElement(String contractElementCode) {
    Validate.notBlank(contractElementCode, "合同要素模板编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(elementRegisters), "系统没有模板注册");
    List<ContractElementRegister> contractElements =
        elementRegisters.stream()
            .filter(
                elementRegister ->
                    contractElementCode.equals(elementRegister.getContractElementCode()))
            .collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isNotEmpty(contractElements), "合同要素模板不存在");
  }
}
