package com.biz.crm.dms.business.costpool.capital.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.costpool.capital.local.entity.Capital;
import com.biz.crm.dms.business.costpool.capital.local.repository.CapitalRepository;
import com.biz.crm.dms.business.costpool.capital.local.service.CapitalFlowService;
import com.biz.crm.dms.business.costpool.capital.local.service.CapitalService;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CapitalFlowDto;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CapitalPageDto;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.capital.sdk.strategy.CapitalOperationTypeStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Optional;
import org.springframework.util.CollectionUtils;

/**
 * @description: 资金service
 * @author: rentao
 * @date: 2021/12/13 20:04
 */
@Slf4j
@Service
public class CapitalServiceImpl implements CapitalService {

  @Autowired(required = false)
  private CapitalRepository capitalRepository;
  @Autowired(required = false)
  private CapitalFlowService capitalFlowService;
  @Autowired(required = false)
  private List<CapitalOperationTypeStrategy> capitalOperationTypeStrategys;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 新增客户资金
   *
   * @param capitalDto
   * @return
   */
  @Override
  @Transactional
  public Capital create(CapitalDto capitalDto) {
    this.validateAdd(capitalDto);
    Capital entity = this.nebulaToolkitService.copyObjectByWhiteList(
        capitalDto, Capital.class, HashSet.class, ArrayList.class);
    this.capitalRepository.saveOrUpdate(entity);
    CapitalFlowDto capitalFlowDto = this.nebulaToolkitService
        .copyObjectByBlankList(capitalDto, CapitalFlowDto.class, HashSet.class, ArrayList.class);
    this.capitalFlowService.create(capitalFlowDto);
    return entity;
  }


  @Override
  @Transactional
  public void handleAdjust(CostPoolCapitalDto costPoolCapitalDto) {
    this.handleAdjustValidate(costPoolCapitalDto);
    //调用策略类，根据操作类型确认执行哪个策略实现
    CapitalAdjustTypeEnum operationTypeEnum = CapitalAdjustTypeEnum
        .getByKey(costPoolCapitalDto.getAdjustType());
    Validate.notNull(operationTypeEnum, "不支持此操作类型进行调整费用池信息");
    Validate.isTrue(!CollectionUtils.isEmpty(capitalOperationTypeStrategys), "未查询到操作类型对应的策略实现");
    for (CapitalOperationTypeStrategy operationTypeStrategy : capitalOperationTypeStrategys) {
      String group = operationTypeStrategy.getOperationTypeGroup();
      if (StringUtils.equals(group, costPoolCapitalDto.getAdjustType())) {
        operationTypeStrategy.onSaveCapitalInfos(costPoolCapitalDto);
        break;
      }
    }
  }

  @Override
  public Page<Capital> findByConditions(Pageable pageable, CapitalPageDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new CapitalPageDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<Capital> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.capitalRepository.findByConditions(page, dto);
  }

  @Override
  public Capital findById(String id) {
    Validate.notNull(id, "传入id不能为空");
    return this.capitalRepository.getById(id);
  }

  @Override
  public Capital findByCustomerCode(String customerCode) {
    Validate.notNull(customerCode, "传入客户编码不能为空");
    return this.capitalRepository.findByCusCode(customerCode);
  }

  /**
   * 调整验证（基础验证）
   *
   * @param
   */
  private void handleAdjustValidate(CostPoolCapitalDto costPoolCapitalDto) {
    Validate.notNull(costPoolCapitalDto, "调整时，对象信息不能为空！");
    Validate.notBlank(costPoolCapitalDto.getCustomerCode(), "调整时，客户编码不能为空！");
    Validate.notBlank(costPoolCapitalDto.getAdjustType(), "调整时，调整类型不能为空！");
    Validate.notNull(costPoolCapitalDto.getAdjustMoney(), "调整时，调整金额不能为空！");
  }

  /**
   * 校验初始化资金池 验证
   *
   * @param capitalDto
   */
  private void validateAdd(CapitalDto capitalDto) {
    Validate.notNull(capitalDto, "资金信息不能为空");
    Validate.notBlank(capitalDto.getCustomerCode(), "客户编码不能为空");
    Validate.notBlank(capitalDto.getCustomerName(), "客户名称不能为空");
    Validate.notNull(capitalDto.getCapitalTotal(), "资金账款总金额不能为空");
    Validate.notNull(capitalDto.getFrozeMoney(), "资金冻结总额不能为空");
    Validate.notNull(capitalDto.getCapitalAbleTotal(), "资金可用余额不能为空");
    Validate
        .isTrue(capitalDto.getCapitalAbleTotal().compareTo(BigDecimal.ZERO) >= 0, "资金可用余额不能小于零");
    Validate.isTrue(capitalDto.getCapitalTotal().compareTo(BigDecimal.ZERO) >= 0, "资金账款总金额不能小于零");
    Validate.isTrue(capitalDto.getFrozeMoney().compareTo(BigDecimal.ZERO) >= 0, "资金冻结总额不能小于零");
  }
}
