package com.biz.crm.dms.business.costpool.capital.local.service.strategy;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.dms.business.costpool.capital.local.entity.Capital;
import com.biz.crm.dms.business.costpool.capital.local.service.CapitalService;
import com.biz.crm.dms.business.costpool.capital.local.utils.CapitalOperationTypeUtil;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.capital.sdk.strategy.CapitalOperationTypeStrategy;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

/**
 * @description: 冻结解冻
 * @author: rentao
 * @date: 2022/3/14 12:46
 */
@Service
public class CapitalOperationTypeFreezeStrategy implements CapitalOperationTypeStrategy {

  @Autowired(required = false)
  private CapitalService capitalService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public String getOperationTypeGroup() {
    return CapitalAdjustTypeEnum.FREEZE_UNFREEZE.getKey();
  }

  @Override
  public String getOperationTypeGroupName() {
    return CapitalAdjustTypeEnum.FREEZE_UNFREEZE.getValue();
  }

  /**
   * 冻结解冻
   *
   * @param costPoolCapitalDto
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public void onSaveCapitalInfos(CostPoolCapitalDto costPoolCapitalDto) {
    //验证
    CapitalOperationTypeUtil.validateCostPoolCapitalDto(costPoolCapitalDto);
    Capital capital = this.capitalService.findByCustomerCode(costPoolCapitalDto.getCustomerCode());
    Validate.notNull(capital, "对应客户资金不存在不能编辑");
    BigDecimal adjustMoney = costPoolCapitalDto.getAdjustMoney();
    Validate.isTrue(adjustMoney.compareTo(BigDecimal.ZERO) != 0, "冻结/解冻额度不能0");
    CapitalDto capitalDto = this.transformDataUpdate(capital, costPoolCapitalDto);
    this.capitalService.create(capitalDto);
  }

  /**
   * 是否主动（页面展示）
   *
   * @author rentao
   * @date
   */
  @Override
  public BooleanEnum getActiveType() {
    return BooleanEnum.TRUE;
  }

  /**
   * 是否管理端（是，后台，否经销商页面调用）
   *
   * @author rentao
   * @date
   */
  @Override
  public BooleanEnum getAmanagementType() {
    return BooleanEnum.TRUE;
  }

  /**
   * 是否初始化客户资金
   *
   * @author rentao
   * @date
   */
  @Override
  public BooleanEnum getInitType() {
    return BooleanEnum.FALSE;
  }

  /**
   * 冻结解冻数据组装
   *
   * @param capital
   * @param costPoolCapitalDto
   * @author rentao
   * @date
   */
  private CapitalDto transformDataUpdate(Capital capital,
      CostPoolCapitalDto costPoolCapitalDto) {
    CapitalDto capitalDto =
        this.nebulaToolkitService
            .copyObjectByWhiteList(capital, CapitalDto.class, HashSet.class, ArrayList.class);
    // 资金可用余额
    BigDecimal capitalAbleTotalOld = capital.getCapitalAbleTotal();
    // 之前冻结金额
    BigDecimal frozeMoneyOld = capital.getFrozeMoney();
    //本次操作金额
    BigDecimal adjustMoney = costPoolCapitalDto.getAdjustMoney();
    // 现在冻结金额
    BigDecimal frozeMoneyNew = frozeMoneyOld.add(adjustMoney);
    Validate.notNull(frozeMoneyNew, "当前冻结金额不能为空。");
    Validate.isTrue(frozeMoneyNew.compareTo(frozeMoneyOld) != 0, "调整后不能等于调整前额度。");
    Validate.isTrue(frozeMoneyNew.compareTo(BigDecimal.ZERO) >= 0, "调整后冻结金额不能小于零。");
    // 新的可用余额
    BigDecimal capitalAbleTotalNew = capitalAbleTotalOld.subtract(adjustMoney);
    Validate.isTrue(capitalAbleTotalNew.compareTo(BigDecimal.ZERO) >= 0, "可用余额不能小于零。");
    capitalDto.setCapitalAbleTotal(capitalAbleTotalNew);
    capitalDto.setFrozeMoney(frozeMoneyNew);
    capitalDto.setAdjustMoney(adjustMoney);
    capitalDto.setAdjustType(costPoolCapitalDto.getAdjustType());
    capitalDto.setAdjustTypeName(CapitalAdjustTypeEnum.FREEZE_UNFREEZE.getValue());
    capitalDto.setCapitalFlowFileDtos(costPoolCapitalDto.getCapitalFlowFileDtos());
    if(StringUtils.isNotBlank(costPoolCapitalDto.getBillNo())){
      capitalDto.setBillNo(costPoolCapitalDto.getBillNo());
      capitalDto.setBillType(costPoolCapitalDto.getBillType());
    }else {
      capitalDto.setBillNo(costPoolCapitalDto.getFromCode());
      capitalDto.setBillType(costPoolCapitalDto.getFromDesc());
    }
    return capitalDto;
  }


}
