package com.biz.crm.dms.business.costpool.credit.local.controller;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditService;
import com.biz.crm.dms.business.costpool.credit.sdk.constant.CreditConstant;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditRepayNoticeDto;
import com.biz.crm.dms.business.costpool.credit.sdk.service.CreditVoService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 授信管理: Credit: 授信信息
 *
 * @author ning.zhang
 * @date 2021/12/20
 */
@Slf4j
@RestController
@RequestMapping("/v1/credit/credit")
@Api(tags = "授信管理: Credit: 授信信息")
public class CreditController {

  @Autowired(required = false)
  private CreditService creditService;
  @Autowired(required = false)
  private RedisMutexService redisMutexService;
  @Autowired(required = false)
  private CreditVoService creditVoService;

  /**
   * 新增授信信息
   *
   * @param dto 参数dto
   * @return 操作结果
   */
  @ApiOperation(value = "新增授信")
  @PostMapping("")
  public Result<?> create(@RequestBody CreditDto dto) {
    String lockKey = String.format(CreditConstant.CREDIT_CREATE_LOCK_KEY, TenantUtils.getTenantCode());
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, CreditConstant.CREDIT_TRY_LOCK_OUT_TIME)
        , CreditConstant.CREDIT_LOCK_FAIL_MESSAGE);
    try {
      this.creditService.create(dto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(lockKey);
    }
  }

  /**
   * 商城客户申请临时授信
   *
   * @param dto 参数dto
   * @return 操作结果
   */
  @ApiOperation(value = "商城客户申请临时授信")
  @PostMapping("/handleTemporaryCreditApply")
  public Result<?> handleTemporaryCreditApply(@RequestBody CreditDto dto) {
    String lockKey = String.format(CreditConstant.CREDIT_CREATE_LOCK_KEY, TenantUtils.getTenantCode());
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, CreditConstant.CREDIT_TRY_LOCK_OUT_TIME)
        , CreditConstant.CREDIT_LOCK_FAIL_MESSAGE);
    try {
      this.creditService.handleTemporaryCreditApply(dto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(lockKey);
    }
  }

  /**
   * 更新授信信息
   *
   * @param dto 参数dto
   * @return 操作结果
   */
  @ApiOperation(value = "更新授信信息")
  @PatchMapping("")
  public Result<?> update(@RequestBody CreditDto dto) {
    String lockKey = this.creditVoService.buildCashLockKeyByCreditId(dto.getId());
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, CreditConstant.CREDIT_TRY_LOCK_OUT_TIME)
        , CreditConstant.CREDIT_LOCK_FAIL_MESSAGE);
    try {
      this.creditService.update(dto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    } finally {
      this.redisMutexService.unlock(lockKey);
    }
  }

  /**
   * 启用授信
   *
   * @param customerCodes 授信客户编码集合
   * @return 操作结果
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enableBatch")
  public Result<?> enableBatch(@RequestBody List<String> customerCodes) {
    try {
      this.creditService.enableBatch(customerCodes);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用授信
   *
   * @param customerCodes 授信客户编码集合
   * @return 操作结果
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disableBatch")
  public Result<?> disableBatch(@RequestBody List<String> customerCodes) {
    try {
      this.creditService.disableBatch(customerCodes);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 删除授信
   *
   * @param ids 授信id集合
   * @return 操作结果
   */
  @ApiOperation(value = "删除")
  @DeleteMapping("/deleteBatch")
  public Result deleteBatch(@RequestParam("ids") List<String> ids) {
    try {
      this.creditService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 发送还款通知
   *
   * @param dto 参数dto
   */
  @ApiOperation(value = "发送还款通知")
  @PostMapping("/sendNotice")
  public Result<?> sendNotice(@RequestBody CreditRepayNoticeDto dto) {
    try {
      this.creditService.sendNotice(dto);
      return Result.ok("发送成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
