package com.biz.crm.dms.business.costpool.credit.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.mapper.CreditMapper;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 授信表的数据库访问类 {@link CreditEntity}
 *
 * @author ning.zhang
 * @date 2021-12-14 20:08:46
 */
@Component
public class CreditRepository extends ServiceImpl<CreditMapper, CreditEntity> {

  /**
   * 通过客户编码集合查询客户授信信息
   *
   * @param customerCodes 客户编码集合
   * @param creditType    授信类型
   * @param enableStatus  启用状态
   * @param tenantCode    租户编码
   * @return 客户授信信息集合
   */
  public List<CreditEntity> findByCustomerCodes(List<String> customerCodes, String creditType
          , String enableStatus, String tenantCode) {
    return this.lambdaQuery()
            .eq(CreditEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(CreditEntity::getTenantCode, tenantCode)
            .eq(StringUtils.isNotBlank(enableStatus), CreditEntity::getEnableStatus, enableStatus)
            .eq(StringUtils.isNotBlank(creditType), CreditEntity::getCreditType, creditType)
            .in(CreditEntity::getCustomerCode, customerCodes)
            .list();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
            .in(CreditEntity::getId, ids)
            .set(CreditEntity::getEnableStatus, enableStatus.getCode())
            .update();
  }

  /**
   * 根据审批单号获取授信实体
   *
   * @param processNo 审批单号
   * @return 授信实体
   */
  public CreditEntity findByCreditProcessNo(String processNo) {
    if (StringUtils.isEmpty(processNo)) {
      return null;
    }
    return this.lambdaQuery().eq(CreditEntity::getProcessNumber, processNo)
            .one();
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(CreditEntity::getId, ids)
        .set(CreditEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }
}
