package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditDownUpAccountEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditDownUpAccountRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditDownUpAccountService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditFileService;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.model.CreditFileModelDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditDownUpAccountCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditDownUpAccountDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditFileType;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

/**
 * 授信回款上账表服务实现类
 *
 * @author ning.zhang
 * @date 2021-12-14 20:09:26
 */
@Slf4j
@Service("creditDownUpAccountService")
public class CreditDownUpAccountServiceImpl implements CreditDownUpAccountService {

  @Autowired(required = false)
  private CreditDownUpAccountRepository creditDownUpAccountRepository;
  @Autowired(required = false)
  private CreditRepository creditRepository;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditFileService creditFileService;

  /**
   * 创建回款上账信息
   * 1.只有普通授信才能回款上账
   * 2.传入金额为正数表示释放授信额度,负数表示使用授信额度
   * 3.释放授信额度时:上账金额+可用余额(因为超可用余额冻结可能为负)+冻结金额不能大于授信总额度
   * 4.使用额度时: 上账金额(绝对值比较)不能大于可用金额
   * 5.并记录保存且生成资金变动明细
   *
   * @param dto 参数dto
   */
  @Override
  @Transactional
  public void create(CreditDownUpAccountDto dto) {
    this.createValidation(dto);
    CashAdjustOperateEnum adjustOperate= CashAdjustOperateEnum.DOWN_UP_ACCOUNT;
    CashAdjustTypeEnum adjustType= CashAdjustTypeEnum.getByDictCode(dto.getOperateType());
    CreditDownUpAccountCashDto cashFlowDto = new CreditDownUpAccountCashDto();
    cashFlowDto.setCreditId(dto.getCreditId());
    cashFlowDto.setOperateAmount(dto.getDownUpAmount());
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    //创建资金流水
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    CreditDownUpAccountEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CreditDownUpAccountEntity.class, HashSet.class, ArrayList.class);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    //持久化上账记录
    this.creditDownUpAccountRepository.save(entity);
    //更新保存文件信息
    CreditFileModelDto fileModelDto = new CreditFileModelDto();
    fileModelDto.setBusinessId(entity.getId());
    fileModelDto.setFileType(CreditFileType.DOWN_UP_ACCOUNT.getDictCode());
    fileModelDto.setFileList(dto.getFileList());
    this.creditFileService.update(Lists.newArrayList(fileModelDto));
  }

  /**
   * 在创建CreditFreeze模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditDownUpAccountDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setId(null);
    Validate.notBlank(dto.getCreditId(), "缺失授信ID！");
    Validate.notBlank(dto.getOperateType(), "缺失操作类型！");
    Validate.notBlank(dto.getOrderCode(), "缺失单据编码！");
    Validate.notBlank(dto.getOrderType(), "缺失单据类型！");
    Validate.notNull(dto.getDownUpAmount(), "缺失回款/上账金额!");
    Validate.isTrue(dto.getDownUpAmount().compareTo(BigDecimal.ZERO) != 0, "回款/上账金额不能等于0");
    List<String> supportOperateTypes = Lists.newArrayList(CashAdjustTypeEnum.OTHER.getDictCode()
        , CashAdjustTypeEnum.CUSTOMER_UP_ACCOUNT.getDictCode(), CashAdjustTypeEnum.WRITE_OFF_UP_ACCOUNT.getDictCode());
    Validate.isTrue(supportOperateTypes.contains(dto.getOperateType()), "不支持的操作类型！");
    CreditEntity creditEntity = this.creditRepository.getById(dto.getCreditId());
    Validate.notNull(creditEntity, "授信信息不存在!");
    Validate.isTrue(creditEntity.getCreditType().equals(CreditType.NORMAL_CREDIT.getDictCode()), "回款上账只支持普通授信");
    dto.setCustomerCode(creditEntity.getCustomerCode());
    Date nowDate = new Date();
    Validate.isTrue(creditEntity.getCreditEndTime().getTime() >= nowDate.getTime(), "普通授信已过期,无法回款上账");
  }
}
