package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditRefundEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRefundRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditRefundService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditRefundCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditRefundDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditOrderTypeEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;

/**
 * 授信退款实体类服务实现类
 *
 * @author ning.zhang
 * @date 2022-01-20 16:19:15
 */
@Slf4j
@Service("creditRefundService")
public class CreditRefundServiceImpl implements CreditRefundService {

  @Autowired(required = false)
  private CreditRefundRepository creditRefundRepository;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;

  @Override
  @Transactional
  public void create(CreditRefundDto dto) {
    this.createValidation(dto);
    CashAdjustOperateEnum adjustOperate = CashAdjustOperateEnum.getByDictCode(dto.getAdjustOperateCode());
    CashAdjustTypeEnum adjustType = CashAdjustTypeEnum.getByDictCode(dto.getAdjustTypeCode());
    CreditRefundEntity entity = new CreditRefundEntity();
    entity.setBusinessOrderCode(dto.getFromCode());
    entity.setRefundAmount(dto.getAmount());
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setOrderType(CreditOrderTypeEnum.REFUND_ORDER.getDictCode());
    CreditRefundCashDto cashFlowDto = new CreditRefundCashDto();
    cashFlowDto.setBusinessOrderCode(dto.getFromCode());
    cashFlowDto.setOperateAmount(dto.getAmount());
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    //创建资金流水
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    //持久化退款记录
    this.creditRefundRepository.save(entity);
  }

  /**
   * 在创建CreditRefund模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditRefundDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.notBlank(dto.getAdjustOperateCode(), "缺失调整操作编码！");
    Validate.notBlank(dto.getAdjustTypeCode(), "缺失调整类型编码！");
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码！");
    Validate.notBlank(dto.getFromCode(), "缺失来源编码!");
    Validate.notNull(dto.getAmount(), "缺失金额!");
    Validate.isTrue(dto.getAmount().compareTo(BigDecimal.ZERO) > 0, "金额必须大于0");
    Validate.isTrue(CashAdjustTypeEnum.CREDIT_RELEASE.getDictCode().equals(dto.getAdjustTypeCode()), "不支持的授信调整类型！");
    Validate.isTrue(CashAdjustOperateEnum.CLOSE_ORDER.getDictCode().equals(dto.getAdjustOperateCode())
        || CashAdjustOperateEnum.RETURN_ORDER.getDictCode().equals(dto.getAdjustOperateCode()), "不支持的授信调整操作！");
  }
}
