package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForEMS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditFileEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditFileRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditVoRepository;
import com.biz.crm.dms.business.costpool.credit.sdk.constant.CreditConstant;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPageDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditTemporaryPageDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditFileType;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.service.CreditVoService;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditFileVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerContactVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.workflow.sdk.constant.enums.ActApproveStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 授信Vo服务接口实现类
 *
 * @author ning.zhang
 * @date 2021/12/16
 */
@Service
public class CreditVoServiceImpl implements CreditVoService {

  @Autowired(required = false)
  private CreditVoRepository creditVoRepository;
  @Autowired(required = false)
  private CreditRepository creditRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private CreditFileRepository creditFileRepository;

  @Override
  public Page<CreditVo> findByCreditCommonPageDto(Pageable pageable, CreditPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CreditPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.creditVoRepository.findByCreditPageDto(pageable, dto);
  }

  @Override
  public Page<CreditVo> findByCreditTemporaryPageDto(Pageable pageable, CreditTemporaryPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CreditTemporaryPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.creditVoRepository.findByCreditTemporaryPageDto(pageable, dto);
  }

  @Override
  public CreditVo findByCommonCreditId(String commonCreditId) {
    if (StringUtils.isBlank(commonCreditId)) {
      return null;
    }
    return this.creditVoRepository.findByCommonCreditId(commonCreditId);
  }

  @Override
  public String buildCashLockKeyByCreditId(String creditId) {
    Validate.notBlank(creditId, "授信id不能为空！");
    CreditEntity creditEntity = this.creditRepository.getById(creditId);
    Validate.notNull(creditEntity, "授信信息不存在！");
    return String.format(CreditConstant.CREDIT_CASH_CHANGE_LOCK_KEY, creditEntity.getCustomerCode());
  }

  @Override
  public String buildCashLockKeyByCustomerCode(String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    return String.format(CreditConstant.CREDIT_CASH_CHANGE_LOCK_KEY, customerCode);
  }

  /**
   * 客户-商城授信头展示查询
   *
   * @param dto
   * @return
   */
  @Override
  public CreditVo findByTotalCreditDto(CreditDto dto) {
    CreditVo creditVo = new CreditVo();
    creditVo.setLongAmount(BigDecimal.ZERO);
    creditVo.setLongCanUseAmount(BigDecimal.ZERO);
    creditVo.setLongFreezeAmount(BigDecimal.ZERO);
    creditVo.setLongHaveUseAmount(BigDecimal.ZERO);
    creditVo.setTemporaryAmount(BigDecimal.ZERO);
    creditVo.setTemporaryCanUseAmount(BigDecimal.ZERO);
    creditVo.setTemporaryFreezeAmount(BigDecimal.ZERO);
    creditVo.setTemporaryHaveUseAmount(BigDecimal.ZERO);
    creditVo.setTotalAmount(BigDecimal.ZERO);
    LoginUserDetailsForEMS loginUserDetails = loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    if (StringUtils.isBlank(loginUserDetails.getCustomerCode())) {
      return creditVo;
    }
    dto.setCustomerCode(loginUserDetails.getCustomerCode());
    dto.setTenantCode(TenantUtils.getTenantCode());
    creditVo = this.creditVoRepository.findByTotalCreditDto(dto);
    if (ObjectUtils.isEmpty(creditVo)) {
      return null;
    }
    //查询客户信息
    CustomerVo customerVo = customerVoService.findDetailsByIdOrCode(null, loginUserDetails.getCustomerCode());
    if (Objects.nonNull(customerVo)) {
      creditVo.setRegisteredAddress(customerVo.getRegisteredAddress());
      List<CustomerContactVo> contactList = customerVo.getContactList();
      if (!CollectionUtils.isEmpty(contactList)) {
        CustomerContactVo customerContactVo = contactList.stream().filter(CustomerContactVo::getContactMain)
                .findFirst().orElse(new CustomerContactVo());
        creditVo.setContactName(customerContactVo.getContactName());
        creditVo.setContactPhone(customerContactVo.getContactPhone());
      }
    }
    return creditVo;
  }

  /**
   * 商城-临时授信申请记录信息分页列表
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<CreditVo> findApplyRecordByCreditTemporaryPageDto(Pageable pageable, CreditTemporaryPageDto dto) {
    LoginUserDetailsForEMS loginUserDetails = loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    if (StringUtils.isBlank(loginUserDetails.getCustomerCode())) {
      Page<CreditVo> pageResult = new Page<>(0L, 0L, 0L);
      return pageResult;
    }
    dto.setCreditType(CreditType.TEMPORARY_CREDIT.getDictCode());
    dto.setCustomerCode(loginUserDetails.getCustomerCode());
    //查询
    pageable = org.apache.commons.lang3.ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Page<CreditVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<CreditVo> entityResult = this.creditVoRepository.findApplyRecordByCreditTemporaryPageDto(page, dto);
    return entityResult;
  }

  /**
   * 商城-临时授信记录信息分页列表（审批通过）
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<CreditVo> findRecordByCreditTemporaryPageDto(Pageable pageable, CreditTemporaryPageDto dto) {
    LoginUserDetailsForEMS loginUserDetails = loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    if (StringUtils.isBlank(loginUserDetails.getCustomerCode())) {
      Page<CreditVo> pageResult = new Page<>(0L, 0L, 0L);
      return pageResult;
    }
    dto.setCreditType(CreditType.TEMPORARY_CREDIT.getDictCode());
    dto.setProcessStatus(ActApproveStatusEnum.APPROVED.getCode());
    //查询
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setCustomerCode(loginUserDetails.getCustomerCode());
    return this.creditVoRepository.findByCreditTemporaryPageDto(pageable, dto);
  }

  /**
   * 临时授信ID获取授信信息
   *
   * @param creditTemporaryId
   * @return
   */
  @Override
  public CreditVo findByCreditTemporaryId(String creditTemporaryId) {
    Validate.notBlank(creditTemporaryId, "临时授信ID不能为空");
    CreditEntity creditEntity = this.creditRepository.getById(creditTemporaryId);
    Validate.isTrue(!org.springframework.util.ObjectUtils.isEmpty(creditEntity), "未查询到临时授信信息");
    CreditVo creditVo = this.nebulaToolkitService.copyObjectByWhiteList(creditEntity, CreditVo.class, HashSet.class, ArrayList.class);
    creditVo.setTemporaryAmount(creditEntity.getCreditAmount());
    //根据授信ID查询附件信息
    List<CreditFileEntity> creditFileEntities = this.creditFileRepository.findByBusinessIdAndFileType(creditVo.getId(), CreditFileType.CREDIT.getDictCode());
    if (!CollectionUtils.isEmpty(creditFileEntities)) {
      List<CreditFileVo> creditFileVos = (List<CreditFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(creditFileEntities
              , CreditFileEntity.class, CreditFileVo.class, HashSet.class, ArrayList.class);
      creditVo.setFileList(creditFileVos);
    }
    //查询客户信息
    CustomerVo customerVo = customerVoService.findDetailsByIdOrCode(null, creditVo.getCustomerCode());
    if (!ObjectUtils.isEmpty(customerVo)) {
      creditVo.setRegisteredAddress(customerVo.getRegisteredAddress());
      creditVo.setCustomerName(customerVo.getCustomerName());
      List<CustomerContactVo> contactList = customerVo.getContactList();
      if (!CollectionUtils.isEmpty(contactList)) {
        CustomerContactVo customerContactVo = contactList.stream().filter(CustomerContactVo::getContactMain)
                .findFirst().orElse(new CustomerContactVo());
        creditVo.setContactName(customerContactVo.getContactName());
        creditVo.setContactPhone(customerContactVo.getContactPhone());
      }
    }
    return creditVo;
  }
}
