package com.biz.crm.dms.business.costpool.credit.local.service.register;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowItemEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditOccupyEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditPayEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowItemRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditOccupyRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditPayRepository;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditCashFlowExtendDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.OccupyTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.register.CreditCashFlowRegister;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashFlowExtendVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 授信资金支付使用流水拆分注册器实现(授信资金支付使用时导致授信资金流水变动,将资金拆分绑定到具体授信信息)
 * <p>
 * 适用场景描述:
 * 主要用于客户购买,兑换等支付使用授信资金的业务场景
 *
 * @author ning.zhang pengxi
 * @date 2021/12/16 2022/6/28
 */
@Component
public class CreditNewPayCashFlowRegisterImpl implements CreditCashFlowRegister<CreditPayCashDto> {

  @Autowired(required = false)
  private CreditOccupyRepository creditOccupyRepository;
  @Autowired(required = false)
  private CreditCashFlowRepository creditCashFlowRepository;
  @Autowired(required = false)
  private CreditCashFlowItemRepository creditCashFlowItemRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditCashRepository creditCashRepository;
  @Autowired(required = false)
  private CreditPayRepository creditPayRepository;

  @Override
  public Class<CreditPayCashDto> getClazz() {
    return CreditPayCashDto.class;
  }

  @Override
  @Transactional
  public List<CreditCashVo> onCreate(CreditPayCashDto dto) {
    CreditOccupyEntity creditOccupyEntity = this.creditOccupyRepository.findOccupyByOccupyTypeAndBusinessOrderCode(TenantUtils.getTenantCode(), OccupyTypeEnum.OCCUPY, dto.getFromCode());
    Validate.notNull(creditOccupyEntity, "未找到授信占用信息，无法执行扣减逻辑！");

    List<CreditCashFlowEntity> creditCashFlowEntities = this.creditCashFlowRepository.findByCashSerialNumbers(Collections.singletonList(creditOccupyEntity.getCashSerialNumber()), TenantUtils.getTenantCode());
    Validate.notEmpty(creditCashFlowEntities, "未找到授信占用资金流水信息，无法执行扣减逻辑！");

    CreditCashFlowEntity creditCashFlowEntity = creditCashFlowEntities.get(0);
    List<CreditCashFlowItemEntity> creditCashFlowItemEntities = this.creditCashFlowItemRepository.findByCashFlowIds(Collections.singletonList(creditCashFlowEntity.getId()));
    Validate.notEmpty(creditCashFlowItemEntities, "未找到授信占用资金流水明细，无法执行扣减逻辑！");
    String creditId = creditCashFlowItemEntities.get(0).getCreditId();

    // 记录扣减流水
    CreditCashFlowItemEntity flowItemEntity = new CreditCashFlowItemEntity();
    flowItemEntity.setCashFlowId(dto.getId());
    flowItemEntity.setCreditId(creditId);
    flowItemEntity.setItemAmount(creditCashFlowEntity.getOperateAmount());
    flowItemEntity.setTenantCode(creditCashFlowEntity.getTenantCode());
    this.creditCashFlowItemRepository.save(flowItemEntity);

    // 更新授信资金表中的金额
    List<CreditCashEntity> creditCashEntities = this.creditCashRepository.findByCreditIds(Collections.singletonList(creditId));
    Validate.notEmpty(creditCashEntities, "未找到授信资金信息");
    creditCashEntities.forEach(creditCashEntity -> {
      creditCashEntity.setCanUseAmount(creditCashEntity.getCanUseAmount().add(creditCashFlowEntity.getOperateAmount()));
    });
    this.creditCashRepository.updateBatchById(creditCashEntities);

    return (List<CreditCashVo>) this.nebulaToolkitService.copyCollectionByWhiteList(creditCashEntities, CreditCashEntity.class
        , CreditCashVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<CreditCashFlowExtendVo> onRequestByCreditCashFlowExtendDto(CreditCashFlowExtendDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<CreditPayEntity> entities = this.creditPayRepository.findByCreditCashFlowExtendDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return entities.stream().map(entity -> {
      CreditCashFlowExtendVo extendVo = new CreditCashFlowExtendVo();
      extendVo.setCashSerialNumber(entity.getCashSerialNumber());
      extendVo.setOrderCode(entity.getBusinessOrderCode());
      extendVo.setRemark(entity.getRemark());
      extendVo.setOrderType(entity.getOrderType());
      return extendVo;
    }).collect(Collectors.toList());
  }
}
